import React, { useState } from 'react';
import Icon from '../AppIcon';


const PaymentMarkingControls = ({ property, onPaymentMarked, showLabels = true, size = "md" }) => {
  const [isMarking, setIsMarking] = useState(false);

  const handlePaymentMark = async (paymentMethod) => {
    setIsMarking(true);
    
    try {
      const paymentData = {
        propertyId: property?.id,
        paymentMethod,
        markedAt: new Date()?.toISOString(),
        amount: property?.rentAmount + (property?.utilityTotalAmount || 0),
        breakdown: {
          rent: property?.rentAmount,
          water: property?.waterAmount || 25000,
          electricity: property?.electricityAmount || 45000,
          gas: property?.gasAmount || 18000,
          commonExpenses: property?.commonExpensesAmount || 45000
        }
      };
      
      // Simulate API call
      await new Promise(resolve => setTimeout(resolve, 1000));
      
      onPaymentMarked?.(paymentData);
      
      console.log(`Pago marcado como ${paymentMethod === 'manual' ? 'Efectivo/Manual' : 'TransBank'}:`, paymentData);
      
    } catch (error) {
      console.error('Error marking payment:', error);
    } finally {
      setIsMarking(false);
    }
  };

  const buttonSizeClass = size === "sm" ? "px-2 py-1 text-xs" : "px-3 py-1 text-sm";
  const iconSize = size === "sm" ? 12 : 14;

  return (
    <div className="flex items-center space-x-2">
      {showLabels && (
        <span className="text-sm text-gray-600">Marcar Pago:</span>
      )}
      
      <button
        onClick={() => handlePaymentMark('manual')}
        disabled={isMarking}
        className={`
          ${buttonSizeClass} rounded-lg transition-colors flex items-center space-x-1
          ${property?.paymentMethod === 'manual' && property?.paymentStatus === 'paid' ?'bg-green-100 text-green-800 border border-green-300' :'bg-gray-100 text-gray-600 hover:bg-green-50 hover:text-green-700'
          }
          ${isMarking ? 'opacity-50 cursor-not-allowed' : ''}
        `}
      >
        <Icon name="Banknote" size={iconSize} />
        <span>Efectivo</span>
      </button>
      
      <button
        onClick={() => handlePaymentMark('transbank')}
        disabled={isMarking}
        className={`
          ${buttonSizeClass} rounded-lg transition-colors flex items-center space-x-1
          ${property?.paymentMethod === 'transbank' && property?.paymentStatus === 'paid' ?'bg-blue-100 text-blue-800 border border-blue-300' :'bg-gray-100 text-gray-600 hover:bg-blue-50 hover:text-blue-700'
          }
          ${isMarking ? 'opacity-50 cursor-not-allowed' : ''}
        `}
      >
        <Icon name="CreditCard" size={iconSize} />
        <span>TransBank</span>
      </button>
      
      {/* Payment confirmation display */}
      {property?.paymentStatus === 'paid' && property?.lastPaymentDate && (
        <div className="text-xs text-green-600">
          ✓ Pagado el {new Date(property?.lastPaymentDate)?.toLocaleDateString('es-CL')}
        </div>
      )}
    </div>
  );
};

export default PaymentMarkingControls;