import React, { useState, useEffect } from 'react';
import { useLocation, useNavigate } from 'react-router-dom';
import Icon from '../AppIcon';

const QuickActionFloatingButton = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [isMobile, setIsMobile] = useState(false);
  const location = useLocation();
  const navigate = useNavigate();

  useEffect(() => {
    const checkMobile = () => {
      setIsMobile(window.innerWidth < 1024);
    };
    
    checkMobile();
    window.addEventListener('resize', checkMobile);
    return () => window.removeEventListener('resize', checkMobile);
  }, []);

  // Don't render on auth pages
  const authPages = ['/login', '/register', '/password-recovery'];
  if (authPages?.includes(location?.pathname) || !isMobile) {
    return null;
  }

  const getContextualActions = () => {
    const currentPath = location?.pathname;
    
    const actionMap = {
      '/dashboard': [
        { key: 'add-property', label: 'Nueva Propiedad', icon: 'Plus', path: '/properties-management' },
        { key: 'add-tenant', label: 'Nuevo Inquilino', icon: 'UserPlus', path: '/tenants-management' },
        { key: 'create-voucher', label: 'Crear Comprobante', icon: 'FileText', action: 'create-voucher' }
      ],
      '/properties-management': [
        { key: 'add-property', label: 'Nueva Propiedad', icon: 'Plus', action: 'add-property' },
        { key: 'import-properties', label: 'Importar Propiedades', icon: 'Upload', action: 'import' },
        { key: 'property-report', label: 'Generar Reporte', icon: 'BarChart3', path: '/reports-analytics' }
      ],
      '/tenants-management': [
        { key: 'add-tenant', label: 'Nuevo Inquilino', icon: 'UserPlus', action: 'add-tenant' },
        { key: 'send-reminder', label: 'Enviar Recordatorio', icon: 'Bell', path: '/reminders-system' },
        { key: 'tenant-report', label: 'Reporte de Inquilinos', icon: 'FileText', path: '/reports-analytics' }
      ],
      '/vouchers-management': [
        { key: 'create-voucher', label: 'Nuevo Comprobante', icon: 'Plus', action: 'create-voucher' },
        { key: 'bulk-generate', label: 'Generación Masiva', icon: 'Copy', action: 'bulk-generate' },
        { key: 'payment-reminder', label: 'Recordatorio de Pago', icon: 'Clock', path: '/reminders-system' }
      ],
      '/owners-management': [
        { key: 'add-owner', label: 'Nuevo Propietario', icon: 'UserPlus', action: 'add-owner' },
        { key: 'commission-report', label: 'Reporte Comisiones', icon: 'DollarSign', path: '/reports-analytics' }
      ],
      '/reminders-system': [
        { key: 'create-reminder', label: 'Nuevo Recordatorio', icon: 'Plus', action: 'create-reminder' },
        { key: 'bulk-reminder', label: 'Recordatorio Masivo', icon: 'Users', action: 'bulk-reminder' }
      ],
      '/reports-analytics': [
        { key: 'generate-report', label: 'Generar Reporte', icon: 'Plus', action: 'generate-report' },
        { key: 'export-data', label: 'Exportar Datos', icon: 'Download', action: 'export-data' }
      ]
    };

    return actionMap?.[currentPath] || [
      { key: 'dashboard', label: 'Ir al Dashboard', icon: 'LayoutDashboard', path: '/dashboard' }
    ];
  };

  const handleActionClick = (action) => {
    setIsOpen(false);
    
    if (action?.path) {
      navigate(action?.path);
    } else if (action?.action) {
      handleCustomAction(action?.action);
    }
  };

  const handleCustomAction = (actionType) => {
    switch (actionType) {
      case 'add-property': console.log('Opening add property modal');
        break;
      case 'add-tenant': console.log('Opening add tenant modal');
        break;
      case 'create-voucher': console.log('Opening create voucher modal');
        break;
      case 'add-owner': console.log('Opening add owner modal');
        break;
      case 'create-reminder': console.log('Opening create reminder modal');
        break;
      case 'generate-report': console.log('Opening report generator');
        break;
      case 'import': console.log('Opening import dialog');
        break;
      case 'bulk-generate': console.log('Opening bulk generation dialog');
        break;
      case 'bulk-reminder': console.log('Opening bulk reminder dialog');
        break;
      case 'export-data':
        console.log('Starting data export');
        break;
      default:
        console.log('Unknown action:', actionType);
    }
  };

  const actions = getContextualActions();

  return (
    <div className="fixed bottom-6 right-6 z-1000">
      {/* Action Menu */}
      {isOpen && (
        <div className="absolute bottom-16 right-0 mb-2 space-y-2">
          {actions?.map((action, index) => (
            <button
              key={action?.key}
              onClick={() => handleActionClick(action)}
              className="flex items-center space-x-3 bg-white text-gray-700 px-4 py-3 rounded-lg shadow-lg border border-gray-200 hover:bg-gray-50 transition-all duration-150 hover-lift min-w-48"
              style={{
                animationDelay: `${index * 50}ms`,
                animation: 'slideInUp 200ms ease-out forwards'
              }}
            >
              <div className="w-8 h-8 bg-primary/10 text-primary rounded-full flex items-center justify-center flex-shrink-0">
                <Icon name={action?.icon} size={16} />
              </div>
              <span className="text-sm font-medium">{action?.label}</span>
            </button>
          ))}
        </div>
      )}
      {/* Main FAB */}
      <button
        onClick={() => setIsOpen(!isOpen)}
        className={`
          w-14 h-14 bg-primary text-white rounded-full shadow-lg
          flex items-center justify-center
          transition-all duration-200 hover-lift elevation-3
          ${isOpen ? 'rotate-45' : 'rotate-0'}
        `}
      >
        <Icon name="Plus" size={24} />
      </button>
      {/* Overlay */}
      {isOpen && (
        <div 
          className="fixed inset-0 bg-black bg-opacity-20 -z-10"
          onClick={() => setIsOpen(false)}
        />
      )}
      <style jsx>{`
        @keyframes slideInUp {
          from {
            opacity: 0;
            transform: translateY(20px);
          }
          to {
            opacity: 1;
            transform: translateY(0);
          }
        }
      `}</style>
    </div>
  );
};

export default QuickActionFloatingButton;