import React, { useState } from 'react';
import { useLocation, useNavigate } from 'react-router-dom';
import Icon from '../AppIcon';

const Sidebar = ({ isOpen = false, onClose }) => {
  const location = useLocation();
  const navigate = useNavigate();
  const [expandedSections, setExpandedSections] = useState({});

  const navigationItems = [
    {
      key: 'dashboard',
      label: 'Panel Principal',
      path: '/dashboard',
      icon: 'LayoutDashboard',
      tooltip: 'Vista general y métricas principales'
    },
    {
      key: 'properties',
      label: 'Propiedades',
      path: '/properties-management',
      icon: 'Building2',
      tooltip: 'Gestión de cartera inmobiliaria',
      badge: '12'
    },
    {
      key: 'tenants',
      label: 'Inquilinos',
      path: '/tenants-management',
      icon: 'Users',
      tooltip: 'Administración de arrendatarios'
    },
    {
      key: 'owners',
      label: 'Propietarios',
      path: '/owners-management',
      icon: 'UserCheck',
      tooltip: 'Gestión de socios inversionistas'
    },
    {
      key: 'contracts',
      label: 'Redacción de Contratos',
      path: '/draft-contract-manager',
      icon: 'FileEdit',
      tooltip: 'Sistema independiente para crear contratos de arriendo'
    },
    {
      key: 'vouchers',
      label: 'Comprobantes',
      path: '/vouchers-management',
      icon: 'FileText',
      tooltip: 'Documentos financieros y pagos',
      badge: '3'
    },
    {
      key: 'reminders',
      label: 'Recordatorios',
      path: '/reminders-system',
      icon: 'Clock',
      tooltip: 'Sistema de notificaciones automáticas'
    },
    {
      key: 'reports',
      label: 'Reportes',
      path: '/reports-analytics',
      icon: 'BarChart3',
      tooltip: 'Análisis e inteligencia de negocio'
    }
  ];

  const handleNavigation = (path) => {
    navigate(path);
    if (onClose && window.innerWidth < 1024) {
      onClose();
    }
  };

  const isActiveRoute = (path) => {
    if (path === '/dashboard') {
      return location?.pathname === '/' || location?.pathname === '/dashboard';
    }
    return location?.pathname?.startsWith(path);
  };

  return (
    <>
      {/* Mobile Overlay */}
      {isOpen && (
        <div 
          className="fixed inset-0 bg-black bg-opacity-50 z-900 lg:hidden"
          onClick={onClose}
        />
      )}
      {/* Sidebar */}
      <aside className={`
        fixed top-16 left-0 h-[calc(100vh-4rem)] w-60 bg-white border-r border-gray-200 z-950
        transform transition-transform duration-300 ease-out
        ${isOpen ? 'translate-x-0' : '-translate-x-full lg:translate-x-0'}
      `}>
        <nav className="h-full overflow-y-auto py-4">
          <div className="px-4 space-y-1">
            {navigationItems?.map((item) => (
              <div key={item?.key}>
                <button
                  onClick={() => handleNavigation(item?.path)}
                  className={`
                    w-full flex items-center justify-between px-3 py-2.5 rounded-lg text-sm font-medium
                    transition-all duration-150 hover-lift group
                    ${isActiveRoute(item?.path)
                      ? 'bg-primary text-white shadow-sm'
                      : 'text-gray-700 hover:bg-gray-100 hover:text-gray-900'
                    }
                  `}
                  title={item?.tooltip}
                >
                  <div className="flex items-center space-x-3">
                    <Icon 
                      name={item?.icon} 
                      size={18} 
                      className={isActiveRoute(item?.path) ? 'text-white' : 'text-gray-500 group-hover:text-gray-700'}
                    />
                    <span>{item?.label}</span>
                  </div>
                  
                  {item?.badge && (
                    <span className={`
                      px-2 py-0.5 text-xs font-medium rounded-full
                      ${isActiveRoute(item?.path)
                        ? 'bg-white/20 text-white' :'bg-primary text-white'
                      }
                    `}>
                      {item?.badge}
                    </span>
                  )}
                </button>
              </div>
            ))}
          </div>

          {/* Bottom Section */}
          <div className="absolute bottom-4 left-4 right-4">
            <div className="glass-card p-4 rounded-lg">
              <div className="flex items-center space-x-3">
                <div className="w-8 h-8 bg-accent rounded-full flex items-center justify-center">
                  <Icon name="Zap" size={16} className="text-gray-900" />
                </div>
                <div className="flex-1 min-w-0">
                  <p className="text-sm font-medium text-gray-900">Estado del Sistema</p>
                  <p className="text-xs text-gray-500">Todos los servicios operativos</p>
                </div>
              </div>
            </div>
          </div>
        </nav>
      </aside>
    </>
  );
};

export default Sidebar;