import React, { useState, useRef, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import Icon from '../AppIcon';

const UserProfileDropdown = () => {
  const [isOpen, setIsOpen] = useState(false);
  const dropdownRef = useRef(null);
  const navigate = useNavigate();

  const user = {
    name: 'María González',
    email: 'maria.gonzalez@propmanager.cl',
    role: 'Administradora',
    avatar: null
  };

  const menuItems = [
    {
      key: 'profile',
      label: 'Mi Perfil',
      icon: 'User',
      path: '/user-profile-settings'
    },
    {
      key: 'settings',
      label: 'Configuración',
      icon: 'Settings',
      path: '/user-profile-settings'
    },
    {
      key: 'help',
      label: 'Ayuda y Soporte',
      icon: 'HelpCircle',
      action: 'help'
    },
    {
      key: 'logout',
      label: 'Cerrar Sesión',
      icon: 'LogOut',
      action: 'logout',
      variant: 'danger'
    }
  ];

  useEffect(() => {
    const handleClickOutside = (event) => {
      if (dropdownRef?.current && !dropdownRef?.current?.contains(event?.target)) {
        setIsOpen(false);
      }
    };

    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  const handleMenuClick = (item) => {
    setIsOpen(false);
    
    if (item?.path) {
      navigate(item?.path);
    } else if (item?.action === 'logout') {
      handleLogout();
    } else if (item?.action === 'help') {
      window.open('mailto:soporte@propmanager.cl', '_blank');
    }
  };

  const handleLogout = () => {
    localStorage.removeItem('authToken');
    navigate('/login');
  };

  const getInitials = (name) => {
    return name?.split(' ')?.map(word => word?.charAt(0))?.join('')?.toUpperCase()?.slice(0, 2);
  };

  return (
    <div className="relative" ref={dropdownRef}>
      {/* Profile Button */}
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="flex items-center space-x-3 p-2 rounded-lg hover:bg-gray-100 transition-colors duration-150"
      >
        {/* Avatar */}
        <div className="w-8 h-8 bg-primary text-white rounded-full flex items-center justify-center text-sm font-medium">
          {user?.avatar ? (
            <img src={user?.avatar} alt={user?.name} className="w-full h-full rounded-full object-cover" />
          ) : (
            getInitials(user?.name)
          )}
        </div>
        
        {/* User Info - Hidden on mobile */}
        <div className="hidden md:block text-left">
          <p className="text-sm font-medium text-gray-900">{user?.name}</p>
          <p className="text-xs text-gray-500">{user?.role}</p>
        </div>
        
        {/* Dropdown Arrow */}
        <Icon 
          name="ChevronDown" 
          size={16} 
          className={`text-gray-500 transition-transform duration-150 ${isOpen ? 'rotate-180' : ''}`}
        />
      </button>
      {/* Dropdown Menu */}
      {isOpen && (
        <div className="absolute right-0 top-full mt-2 w-64 bg-white rounded-lg shadow-lg border border-gray-200 py-2 z-1100 elevation-3">
          {/* User Info Header */}
          <div className="px-4 py-3 border-b border-gray-100">
            <div className="flex items-center space-x-3">
              <div className="w-10 h-10 bg-primary text-white rounded-full flex items-center justify-center text-sm font-medium">
                {user?.avatar ? (
                  <img src={user?.avatar} alt={user?.name} className="w-full h-full rounded-full object-cover" />
                ) : (
                  getInitials(user?.name)
                )}
              </div>
              <div className="flex-1 min-w-0">
                <p className="text-sm font-medium text-gray-900 truncate">{user?.name}</p>
                <p className="text-xs text-gray-500 truncate">{user?.email}</p>
                <p className="text-xs text-primary font-medium">{user?.role}</p>
              </div>
            </div>
          </div>

          {/* Menu Items */}
          <div className="py-1">
            {menuItems?.map((item) => (
              <button
                key={item?.key}
                onClick={() => handleMenuClick(item)}
                className={`
                  w-full flex items-center space-x-3 px-4 py-2.5 text-sm
                  transition-colors duration-150 hover:bg-gray-50
                  ${item?.variant === 'danger' ?'text-error hover:bg-red-50' :'text-gray-700 hover:text-gray-900'
                  }
                `}
              >
                <Icon 
                  name={item?.icon} 
                  size={16} 
                  className={item?.variant === 'danger' ? 'text-error' : 'text-gray-500'}
                />
                <span>{item?.label}</span>
              </button>
            ))}
          </div>

          {/* Footer */}
          <div className="px-4 py-2 border-t border-gray-100 mt-1">
            <p className="text-xs text-gray-400">
              PropManager v2.1.0
            </p>
          </div>
        </div>
      )}
    </div>
  );
};

export default UserProfileDropdown;