import React, { useState } from 'react';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';

const DataInputPanel = ({ contractData, onChange, selectedTemplate }) => {
  const [activeSection, setActiveSection] = useState('tenant');

  const handleFieldChange = (field, value) => {
    onChange({ [field]: value });
  };

  const formatRUT = (rut) => {
    // Simple RUT formatting for Chilean RUT numbers
    const cleaned = rut?.replace(/[^0-9kK]/g, '');
    if (cleaned?.length > 1) {
      const body = cleaned?.slice(0, -1);
      const dv = cleaned?.slice(-1);
      return `${body?.replace(/\B(?=(\d{3})+(?!\d))/g, '.')}-${dv}`;
    }
    return cleaned;
  };

  const sections = [
    { key: 'tenant', label: 'Arrendatario', icon: 'User' },
    { key: 'owner', label: 'Propietario', icon: 'UserCheck' },
    { key: 'property', label: 'Propiedad', icon: 'Building2' },
    { key: 'rental', label: 'Arriendo', icon: 'FileText' }
  ];

  const civilStatusOptions = [
    { value: 'soltero', label: 'Soltero(a)' },
    { value: 'casado', label: 'Casado(a)' },
    { value: 'divorciado', label: 'Divorciado(a)' },
    { value: 'viudo', label: 'Viudo(a)' }
  ];

  const currencyOptions = [
    { value: 'CLP', label: 'Pesos Chilenos (CLP)' },
    { value: 'UF', label: 'Unidad de Fomento (UF)' },
    { value: 'USD', label: 'Dólar Americano (USD)' }
  ];

  const regionOptions = [
    { value: 'metropolitana', label: 'Región Metropolitana' },
    { value: 'valparaiso', label: 'Valparaíso' },
    { value: 'biobio', label: 'Biobío' },
    { value: 'araucania', label: 'La Araucanía' },
    { value: 'los-lagos', label: 'Los Lagos' }
  ];

  return (
    <div className="h-full flex flex-col">
      {/* Section Navigation */}
      <div className="p-4 border-b border-gray-200">
        <div className="grid grid-cols-2 gap-2">
          {sections?.map((section) => (
            <button
              key={section?.key}
              onClick={() => setActiveSection(section?.key)}
              className={`flex items-center space-x-2 p-2 rounded-lg text-sm font-medium transition-colors duration-150 ${
                activeSection === section?.key
                  ? 'bg-primary text-white' :'text-gray-600 hover:bg-gray-100'
              }`}
            >
              <Icon name={section?.icon} size={14} />
              <span>{section?.label}</span>
            </button>
          ))}
        </div>
      </div>
      {/* Form Content */}
      <div className="flex-1 overflow-y-auto p-4">
        {activeSection === 'tenant' && (
          <div className="space-y-4">
            <h3 className="font-semibold text-gray-900 mb-4">Datos del Arrendatario</h3>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Nombre Completo
              </label>
              <Input
                value={contractData?.arrendatario || ''}
                onChange={(e) => handleFieldChange('arrendatario', e?.target?.value)}
                placeholder="Nombre y apellidos completos"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                RUT
              </label>
              <Input
                value={contractData?.rut_arrendatario || ''}
                onChange={(e) => handleFieldChange('rut_arrendatario', formatRUT(e?.target?.value))}
                placeholder="12.345.678-9"
                maxLength={12}
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Estado Civil
              </label>
              <Select
                value={contractData?.estado_civil_arrendatario || ''}
                onChange={(e) => handleFieldChange('estado_civil_arrendatario', e?.target?.value)}
                options={[{ value: '', label: 'Seleccionar...' }, ...civilStatusOptions]}
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Profesión u Ocupación
              </label>
              <Input
                value={contractData?.profesion_arrendatario || ''}
                onChange={(e) => handleFieldChange('profesion_arrendatario', e?.target?.value)}
                placeholder="Profesión u ocupación principal"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Dirección de Domicilio
              </label>
              <Input
                value={contractData?.direccion_arrendatario || ''}
                onChange={(e) => handleFieldChange('direccion_arrendatario', e?.target?.value)}
                placeholder="Dirección completa de domicilio"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Teléfono de Contacto
              </label>
              <Input
                value={contractData?.telefono_arrendatario || ''}
                onChange={(e) => handleFieldChange('telefono_arrendatario', e?.target?.value)}
                placeholder="+56 9 1234 5678"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Email
              </label>
              <Input
                type="email"
                value={contractData?.email_arrendatario || ''}
                onChange={(e) => handleFieldChange('email_arrendatario', e?.target?.value)}
                placeholder="correo@ejemplo.com"
              />
            </div>
          </div>
        )}

        {activeSection === 'owner' && (
          <div className="space-y-4">
            <h3 className="font-semibold text-gray-900 mb-4">Datos del Propietario</h3>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Nombre Completo
              </label>
              <Input
                value={contractData?.propietario || ''}
                onChange={(e) => handleFieldChange('propietario', e?.target?.value)}
                placeholder="Nombre y apellidos completos"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                RUT
              </label>
              <Input
                value={contractData?.rut_propietario || ''}
                onChange={(e) => handleFieldChange('rut_propietario', formatRUT(e?.target?.value))}
                placeholder="12.345.678-9"
                maxLength={12}
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Estado Civil
              </label>
              <Select
                value={contractData?.estado_civil || ''}
                onChange={(e) => handleFieldChange('estado_civil', e?.target?.value)}
                options={[{ value: '', label: 'Seleccionar...' }, ...civilStatusOptions]}
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Profesión u Ocupación
              </label>
              <Input
                value={contractData?.profesion || ''}
                onChange={(e) => handleFieldChange('profesion', e?.target?.value)}
                placeholder="Profesión u ocupación principal"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Dirección de Domicilio
              </label>
              <Input
                value={contractData?.direccion_propietario || ''}
                onChange={(e) => handleFieldChange('direccion_propietario', e?.target?.value)}
                placeholder="Dirección completa de domicilio"
              />
            </div>
          </div>
        )}

        {activeSection === 'property' && (
          <div className="space-y-4">
            <h3 className="font-semibold text-gray-900 mb-4">Datos de la Propiedad</h3>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Dirección de la Propiedad
              </label>
              <Input
                value={contractData?.direccion_propiedad || ''}
                onChange={(e) => handleFieldChange('direccion_propiedad', e?.target?.value)}
                placeholder="Dirección completa de la propiedad"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Comuna
              </label>
              <Input
                value={contractData?.comuna || ''}
                onChange={(e) => handleFieldChange('comuna', e?.target?.value)}
                placeholder="Comuna donde se ubica la propiedad"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Región
              </label>
              <Select
                value={contractData?.region || ''}
                onChange={(e) => handleFieldChange('region', e?.target?.value)}
                options={[{ value: '', label: 'Seleccionar región...' }, ...regionOptions]}
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Fojas de Inscripción
              </label>
              <Input
                value={contractData?.fojas || ''}
                onChange={(e) => handleFieldChange('fojas', e?.target?.value)}
                placeholder="Número de fojas"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Número de Inscripción
              </label>
              <Input
                value={contractData?.numero || ''}
                onChange={(e) => handleFieldChange('numero', e?.target?.value)}
                placeholder="Número de inscripción"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Conservador de Bienes Raíces
              </label>
              <Input
                value={contractData?.conservador || ''}
                onChange={(e) => handleFieldChange('conservador', e?.target?.value)}
                placeholder="Nombre del conservador"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Año de Inscripción
              </label>
              <Input
                type="number"
                value={contractData?.ano_inscripcion || ''}
                onChange={(e) => handleFieldChange('ano_inscripcion', e?.target?.value)}
                placeholder="2024"
                min="1900"
                max="2030"
              />
            </div>
          </div>
        )}

        {activeSection === 'rental' && (
          <div className="space-y-4">
            <h3 className="font-semibold text-gray-900 mb-4">Datos del Arriendo</h3>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Valor del Arriendo
              </label>
              <Input
                type="number"
                value={contractData?.valor_arriendo || ''}
                onChange={(e) => handleFieldChange('valor_arriendo', e?.target?.value)}
                placeholder="650000"
                min="0"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Moneda
              </label>
              <Select
                value={contractData?.moneda || 'CLP'}
                onChange={(e) => handleFieldChange('moneda', e?.target?.value)}
                options={currencyOptions}
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Fecha de Inicio
              </label>
              <Input
                type="date"
                value={contractData?.fecha_inicio || ''}
                onChange={(e) => handleFieldChange('fecha_inicio', e?.target?.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Fecha de Término
              </label>
              <Input
                type="date"
                value={contractData?.fecha_termino || ''}
                onChange={(e) => handleFieldChange('fecha_termino', e?.target?.value)}
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Plazo en Meses
              </label>
              <Input
                type="number"
                value={contractData?.plazo_meses || ''}
                onChange={(e) => handleFieldChange('plazo_meses', e?.target?.value)}
                placeholder="12"
                min="1"
                max="120"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Días para Pago
              </label>
              <Input
                type="number"
                value={contractData?.dias_pago || ''}
                onChange={(e) => handleFieldChange('dias_pago', e?.target?.value)}
                placeholder="5"
                min="1"
                max="30"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Cláusulas Adicionales
              </label>
              <textarea
                className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary resize-none"
                rows="4"
                value={contractData?.clausulas_adicionales || ''}
                onChange={(e) => handleFieldChange('clausulas_adicionales', e?.target?.value)}
                placeholder="Cláusulas especiales o condiciones adicionales..."
              />
            </div>
          </div>
        )}
      </div>
      {/* Auto-complete Button */}
      <div className="p-4 border-t border-gray-200">
        <Button
          variant="outline"
          iconName="Zap"
          iconPosition="left"
          onClick={() => {
            // Auto-complete with sample data for testing
            const sampleData = {
              propietario: 'Juan Carlos Pérez González',
              rut_propietario: '12.345.678-9',
              arrendatario: 'María Fernanda Silva López',
              rut_arrendatario: '98.765.432-1',
              direccion_propiedad: 'Av. Providencia 1234, Depto 501',
              comuna: 'Providencia',
              region: 'metropolitana',
              valor_arriendo: '850000',
              moneda: 'CLP',
              plazo_meses: '12',
              dias_pago: '5'
            };
            onChange(sampleData);
          }}
          className="w-full"
          size="sm"
        >
          Completar con Datos de Ejemplo
        </Button>
      </div>
    </div>
  );
};

export default DataInputPanel;