import React from 'react';
import Icon from '../../../components/AppIcon';

const DocumentPreview = ({ content, contractData }) => {
  // Process content to replace variables with actual data
  const processContent = (rawContent) => {
    let processedContent = rawContent || '';
    
    // Replace common variables with data
    const replacements = {
      'PROPIETARIO': contractData?.propietario || '[PROPIETARIO]',
      'RUT_PROPIETARIO': contractData?.rut_propietario || '[RUT_PROPIETARIO]',
      'ARRENDATARIO': contractData?.arrendatario || '[ARRENDATARIO]',
      'RUT_ARRENDATARIO': contractData?.rut_arrendatario || '[RUT_ARRENDATARIO]',
      'DIRECCION_PROPIEDAD': contractData?.direccion_propiedad || '[DIRECCION_PROPIEDAD]',
      'COMUNA': contractData?.comuna || '[COMUNA]',
      'REGION': contractData?.region || '[REGION]',
      'VALOR_ARRIENDO': contractData?.valor_arriendo ? `$${parseInt(contractData?.valor_arriendo)?.toLocaleString('es-CL')}` : '[VALOR_ARRIENDO]',
      'MONEDA': contractData?.moneda || '[MONEDA]',
      'FECHA_INICIO': contractData?.fecha_inicio || '[FECHA_INICIO]',
      'FECHA_TERMINO': contractData?.fecha_termino || '[FECHA_TERMINO]',
      'PLAZO_MESES': contractData?.plazo_meses || '[PLAZO_MESES]',
      'DIAS_PAGO': contractData?.dias_pago || '[DIAS_PAGO]',
      'CLAUSULAS_ADICIONALES': contractData?.clausulas_adicionales || '[CLAUSULAS_ADICIONALES]',
      'ESTADO_CIVIL': contractData?.estado_civil || '[ESTADO_CIVIL]',
      'ESTADO_CIVIL_ARRENDATARIO': contractData?.estado_civil_arrendatario || '[ESTADO_CIVIL_ARRENDATARIO]',
      'PROFESION': contractData?.profesion || '[PROFESION]',
      'PROFESION_ARRENDATARIO': contractData?.profesion_arrendatario || '[PROFESION_ARRENDATARIO]',
      'DIRECCION_PROPIETARIO': contractData?.direccion_propietario || '[DIRECCION_PROPIETARIO]',
      'DIRECCION_ARRENDATARIO': contractData?.direccion_arrendatario || '[DIRECCION_ARRENDATARIO]'
    };

    Object.entries(replacements)?.forEach(([key, value]) => {
      const regex = new RegExp(`\\[${key}\\]`, 'g');
      processedContent = processedContent?.replace(regex, value);
    });

    return processedContent;
  };

  let processedContent = processContent(content);

  // Get completion percentage
  const getCompletionPercentage = () => {
    const variables = (content?.match(/\[[A-Z_]+\]/g) || [])?.length;
    const filled = Object.values(contractData || {})?.filter(value => value && value?.toString()?.trim())?.length;
    const totalVariables = Math.max(variables, Object.keys(contractData || {})?.length);
    return totalVariables > 0 ? Math.round((filled / totalVariables) * 100) : 0;
  };

  const completionPercentage = getCompletionPercentage();

  return (
    <div className="h-full flex flex-col bg-gray-50">
      {/* Preview Header */}
      <div className="p-4 bg-white border-b border-gray-200">
        <div className="flex items-center justify-between mb-2">
          <h3 className="font-semibold text-gray-900">Vista Previa del Documento</h3>
          <div className="flex items-center space-x-2 text-sm">
            <Icon name="Eye" size={16} className="text-gray-500" />
            <span className="text-gray-500">Tiempo real</span>
          </div>
        </div>
        
        {/* Completion Progress */}
        <div className="flex items-center space-x-3">
          <div className="flex-1 bg-gray-200 rounded-full h-2">
            <div 
              className="bg-primary h-2 rounded-full transition-all duration-300"
              style={{ width: `${completionPercentage}%` }}
            />
          </div>
          <span className="text-sm font-medium text-gray-600">
            {completionPercentage}% completado
          </span>
        </div>
        
        <p className="text-xs text-gray-500 mt-1">
          Los campos resaltados necesitan ser completados
        </p>
      </div>
      {/* Document Content */}
      <div className="flex-1 overflow-y-auto p-6 bg-white mx-4 my-4 rounded-lg shadow-sm">
        {processedContent ? (
          <div className="max-w-none">
            {/* Document Header */}
            <div className="text-center mb-8 border-b border-gray-200 pb-4">
              <h1 className="text-2xl font-bold text-gray-900 mb-2">
                CONTRATO DE ARRENDAMIENTO
              </h1>
              <p className="text-sm text-gray-600">
                Documento generado el {new Date()?.toLocaleDateString('es-CL')}
              </p>
            </div>

            {/* Processed Content */}
            <div 
              className="prose prose-sm max-w-none font-serif leading-relaxed"
              style={{
                whiteSpace: 'pre-line',
                lineHeight: '1.8',
                fontSize: '14px',
                color: '#1f2937'
              }}
              dangerouslySetInnerHTML={{ 
                __html: processedContent?.replace(/\[([A-Z_]+)\]/g, '<mark class="bg-yellow-200 px-1 rounded">[$1]</mark>') 
              }}
            />

            {/* Property Images Placeholder */}
            {contractData?.direccion_propiedad && (
              <div className="mt-12 border-t pt-8">
                <h3 className="text-lg font-semibold text-gray-900 mb-4">
                  INVENTARIO DE LA PROPIEDAD
                </h3>
                
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4 mb-6">
                  {[1, 2, 3]?.map((i) => (
                    <div key={i} className="border-2 border-dashed border-gray-300 rounded-lg p-4 text-center">
                      <div className="w-full h-24 bg-gray-100 rounded mb-2 flex items-center justify-center">
                        <Icon name="Camera" size={24} className="text-gray-400" />
                      </div>
                      <p className="text-xs text-gray-600">Imagen de propiedad {i}</p>
                      <p className="text-xs text-gray-400 italic">
                        (Se insertará durante la edición)
                      </p>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* Signature Section */}
            <div className="mt-12 border-t pt-8">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                <div className="text-center">
                  <div className="border-t border-gray-400 w-48 mx-auto mb-2"></div>
                  <p className="text-sm font-medium">
                    {contractData?.propietario || '[PROPIETARIO]'}
                  </p>
                  <p className="text-xs text-gray-600">Arrendador</p>
                  <p className="text-xs text-gray-500 mt-1">
                    RUT: {contractData?.rut_propietario || '[RUT_PROPIETARIO]'}
                  </p>
                </div>
                
                <div className="text-center">
                  <div className="border-t border-gray-400 w-48 mx-auto mb-2"></div>
                  <p className="text-sm font-medium">
                    {contractData?.arrendatario || '[ARRENDATARIO]'}
                  </p>
                  <p className="text-xs text-gray-600">Arrendatario</p>
                  <p className="text-xs text-gray-500 mt-1">
                    RUT: {contractData?.rut_arrendatario || '[RUT_ARRENDATARIO]'}
                  </p>
                </div>
              </div>
              
              <div className="text-center mt-8">
                <p className="text-xs text-gray-500">
                  Fecha de firma: ________________
                </p>
              </div>
            </div>
          </div>
        ) : (
          <div className="flex items-center justify-center h-full text-center">
            <div>
              <Icon name="FileText" size={48} className="text-gray-300 mx-auto mb-4" />
              <p className="text-gray-500 mb-2">No hay contenido para mostrar</p>
              <p className="text-sm text-gray-400">
                Selecciona una plantilla o comienza a escribir
              </p>
            </div>
          </div>
        )}
      </div>
      {/* Preview Stats */}
      <div className="p-4 bg-white border-t border-gray-200">
        <div className="grid grid-cols-3 gap-4 text-center">
          <div>
            <p className="text-lg font-semibold text-gray-900">
              {processedContent?.replace(/<[^>]*>/g, '')?.trim()?.split(/\s+/)?.length || 0}
            </p>
            <p className="text-xs text-gray-500">Palabras</p>
          </div>
          
          <div>
            <p className="text-lg font-semibold text-gray-900">
              {(processedContent?.match(/\[([A-Z_]+)\]/g) || [])?.length || 0}
            </p>
            <p className="text-xs text-gray-500">Variables pendientes</p>
          </div>
          
          <div>
            <p className="text-lg font-semibold text-gray-900">
              {Object.values(contractData || {})?.filter(v => v && v?.toString()?.trim())?.length || 0}
            </p>
            <p className="text-xs text-gray-500">Campos completados</p>
          </div>
        </div>
      </div>
    </div>
  );
};

export default DocumentPreview;