import React, { useState } from 'react';
import Button from '../../../components/ui/Button';
import Select from '../../../components/ui/Select';
import { Checkbox } from '../../../components/ui/Checkbox';
import Icon from '../../../components/AppIcon';

const ExportPanel = ({ documentContent, contractData, selectedTemplate }) => {
  const [exportFormat, setExportFormat] = useState('pdf');
  const [exportOptions, setExportOptions] = useState({
    includeImages: true,
    includeHeader: true,
    includeFooter: true,
    includePageNumbers: true,
    includeWatermark: false
  });
  const [isExporting, setIsExporting] = useState(false);

  const formatOptions = [
    { value: 'pdf', label: 'PDF (.pdf)' },
    { value: 'docx', label: 'Word (.docx)' },
    { value: 'html', label: 'HTML (.html)' },
    { value: 'txt', label: 'Texto Plano (.txt)' }
  ];

  const handleExport = async () => {
    setIsExporting(true);
    
    try {
      // Simulate export process
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      if (exportFormat === 'pdf') {
        await exportToPDF();
      } else if (exportFormat === 'docx') {
        await exportToDocx();
      } else if (exportFormat === 'html') {
        await exportToHTML();
      } else if (exportFormat === 'txt') {
        await exportToTXT();
      }
      
    } catch (error) {
      console.error('Export error:', error);
    } finally {
      setIsExporting(false);
    }
  };

  const exportToPDF = async () => {
    // In a real implementation, you would use jsPDF or @react-pdf/renderer
    console.log('Exporting to PDF...');
    
    // Create a simple PDF using the browser's print functionality
    const printWindow = window.open('', '_blank');
    const htmlContent = `
      <!DOCTYPE html>
      <html>
        <head>
          <title>Contrato de Arriendo</title>
          <style>
            body { font-family: 'Times New Roman', serif; line-height: 1.6; margin: 40px; }
            h1 { text-align: center; margin-bottom: 30px; }
            .header { border-bottom: 2px solid #333; margin-bottom: 20px; padding-bottom: 10px; }
            .footer { border-top: 1px solid #ccc; margin-top: 20px; padding-top: 10px; font-size: 12px; }
            @media print { body { margin: 0; } }
          </style>
        </head>
        <body>
          ${exportOptions?.includeHeader ? '<div class="header"><h1>CONTRATO DE ARRENDAMIENTO</h1></div>' : ''}
          <div>${documentContent}</div>
          ${exportOptions?.includeFooter ? '<div class="footer">Documento generado automaticamente</div>' : ''}
        </body>
      </html>
    `;
    
    printWindow?.document?.write(htmlContent);
    printWindow?.document?.close();
    
    setTimeout(() => {
      printWindow?.print();
      printWindow?.close();
    }, 500);
  };

  const exportToDocx = async () => {
    // In a real implementation, you would use the docx library
    console.log('Exporting to DOCX...');
    
    // Simple fallback: create HTML file with .docx extension
    const htmlContent = `
      <html>
        <head>
          <meta charset="utf-8">
          <title>Contrato de Arriendo</title>
        </head>
        <body style="font-family: 'Times New Roman', serif; line-height: 1.6; margin: 40px;">
          <h1 style="text-align: center;">CONTRATO DE ARRENDAMIENTO</h1>
          ${documentContent}
        </body>
      </html>
    `;
    
    downloadFile(htmlContent, `contrato-${Date.now()}.doc`, 'application/msword');
  };

  const exportToHTML = async () => {
    const htmlContent = `
      <!DOCTYPE html>
      <html lang="es">
        <head>
          <meta charset="utf-8">
          <title>Contrato de Arriendo</title>
          <style>
            body { font-family: 'Times New Roman', serif; line-height: 1.6; max-width: 800px; margin: 0 auto; padding: 40px; }
            h1 { text-align: center; margin-bottom: 30px; }
            .contract-content { text-align: justify; }
          </style>
        </head>
        <body>
          <h1>CONTRATO DE ARRENDAMIENTO</h1>
          <div class="contract-content">${documentContent}</div>
        </body>
      </html>
    `;
    
    downloadFile(htmlContent, `contrato-${Date.now()}.html`, 'text/html');
  };

  const exportToTXT = async () => {
    // Strip HTML tags and format as plain text
    const textContent = documentContent?.replace(/<[^>]*>/g, '')?.replace(/\s+/g, ' ')?.trim();
    const formattedText = `CONTRATO DE ARRENDAMIENTO\n\n${textContent}`;
    
    downloadFile(formattedText, `contrato-${Date.now()}.txt`, 'text/plain');
  };

  const downloadFile = (content, filename, contentType) => {
    const blob = new Blob([content], { type: contentType });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = filename;
    document.body?.appendChild(link);
    link?.click();
    document.body?.removeChild(link);
    URL.revokeObjectURL(url);
  };

  const handleOptionChange = (option, checked) => {
    setExportOptions(prev => ({
      ...prev,
      [option]: checked
    }));
  };

  return (
    <div className="p-4 h-full flex flex-col">
      <h3 className="font-semibold text-gray-900 mb-4">Exportar Contrato</h3>
      {/* Export Format */}
      <div className="mb-6">
        <label className="block text-sm font-medium text-gray-700 mb-2">
          Formato de Exportación
        </label>
        <Select
          value={exportFormat}
          onChange={(e) => setExportFormat(e?.target?.value)}
          options={formatOptions}
        />
      </div>
      {/* Export Options */}
      <div className="mb-6">
        <label className="block text-sm font-medium text-gray-700 mb-3">
          Opciones de Exportación
        </label>
        
        <div className="space-y-3">
          <Checkbox
            checked={exportOptions?.includeHeader}
            onChange={(checked) => handleOptionChange('includeHeader', checked)}
            label="Incluir encabezado"
          />
          
          <Checkbox
            checked={exportOptions?.includeFooter}
            onChange={(checked) => handleOptionChange('includeFooter', checked)}
            label="Incluir pie de página"
          />
          
          <Checkbox
            checked={exportOptions?.includeImages}
            onChange={(checked) => handleOptionChange('includeImages', checked)}
            label="Incluir imágenes"
          />
          
          <Checkbox
            checked={exportOptions?.includePageNumbers}
            onChange={(checked) => handleOptionChange('includePageNumbers', checked)}
            label="Numerar páginas"
          />
          
          <Checkbox
            checked={exportOptions?.includeWatermark}
            onChange={(checked) => handleOptionChange('includeWatermark', checked)}
            label="Marca de agua"
          />
        </div>
      </div>
      {/* Document Info */}
      <div className="mb-6 p-3 bg-gray-50 rounded-lg">
        <h4 className="font-medium text-gray-900 mb-2 text-sm">Información del Documento</h4>
        <div className="space-y-1 text-xs text-gray-600">
          <div className="flex justify-between">
            <span>Plantilla:</span>
            <span>{selectedTemplate?.name || 'Sin plantilla'}</span>
          </div>
          <div className="flex justify-between">
            <span>Palabras:</span>
            <span>{documentContent?.replace(/<[^>]*>/g, '')?.trim()?.split(/\s+/)?.length || 0}</span>
          </div>
          <div className="flex justify-between">
            <span>Arrendatario:</span>
            <span>{contractData?.arrendatario || 'No especificado'}</span>
          </div>
          <div className="flex justify-between">
            <span>Propiedad:</span>
            <span>{contractData?.direccion_propiedad || 'No especificado'}</span>
          </div>
        </div>
      </div>
      {/* Export Button */}
      <div className="mt-auto">
        <Button
          variant="default"
          iconName={isExporting ? "RefreshCw" : "Download"}
          iconPosition="left"
          onClick={handleExport}
          disabled={isExporting || !documentContent?.trim()}
          className="w-full"
        >
          {isExporting ? 'Exportando...' : `Exportar como ${formatOptions?.find(f => f?.value === exportFormat)?.label}`}
        </Button>
        
        {!documentContent?.trim() && (
          <p className="text-xs text-red-500 mt-2 text-center">
            El documento está vacío
          </p>
        )}
      </div>
      {/* Quick Export Actions */}
      <div className="mt-4 pt-4 border-t border-gray-200">
        <h4 className="font-medium text-gray-900 mb-3 text-sm">Exportación Rápida</h4>
        
        <div className="grid grid-cols-2 gap-2">
          <Button
            variant="outline"
            iconName="FileText"
            iconPosition="left"
            onClick={() => {
              setExportFormat('pdf');
              setTimeout(handleExport, 100);
            }}
            disabled={isExporting}
            size="sm"
          >
            PDF
          </Button>
          
          <Button
            variant="outline"
            iconName="FileText"
            iconPosition="left"
            onClick={() => {
              setExportFormat('docx');
              setTimeout(handleExport, 100);
            }}
            disabled={isExporting}
            size="sm"
          >
            Word
          </Button>
        </div>
      </div>
      {/* Usage Tips */}
      <div className="mt-4 p-3 bg-blue-50 rounded-lg">
        <div className="flex items-start space-x-2">
          <Icon name="Info" size={16} className="text-blue-600 mt-0.5" />
          <div className="text-xs text-blue-800">
            <p className="font-medium mb-1">Consejos:</p>
            <ul className="space-y-1">
              <li>• PDF para documentos finales</li>
              <li>• Word para edición posterior</li>
              <li>• Revisa la vista previa antes de exportar</li>
            </ul>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ExportPanel;