import React, { forwardRef, useImperativeHandle, useRef, useEffect } from 'react';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';

const RichTextEditor = forwardRef(({ content, onChange }, ref) => {
  const editorRef = useRef(null);
  const isUpdatingRef = useRef(false);

  useImperativeHandle(ref, () => ({
    insertImage: (imageUrl) => {
      const img = `<img src="${imageUrl}" style="max-width: 100%; height: auto; margin: 10px 0;" />`;
      insertAtCursor(img);
    },
    insertTable: () => {
      const table = `
        <table border="1" style="border-collapse: collapse; width: 100%; margin: 10px 0;">
          <tr>
            <th style="padding: 8px; background-color: #f5f5f5;">Columna 1</th>
            <th style="padding: 8px; background-color: #f5f5f5;">Columna 2</th>
          </tr>
          <tr>
            <td style="padding: 8px;">Celda 1</td>
            <td style="padding: 8px;">Celda 2</td>
          </tr>
        </table>
      `;
      insertAtCursor(table);
    },
    insertList: () => {
      const list = `
        <ul style="margin: 10px 0;">
          <li>Elemento 1</li>
          <li>Elemento 2</li>
          <li>Elemento 3</li>
        </ul>
      `;
      insertAtCursor(list);
    }
  }));

  useEffect(() => {
    if (editorRef?.current && !isUpdatingRef?.current) {
      editorRef.current.innerHTML = content || '';
    }
  }, [content]);

  const insertAtCursor = (html) => {
    if (editorRef?.current) {
      const selection = window.getSelection();
      if (selection?.rangeCount > 0) {
        const range = selection?.getRangeAt(0);
        const fragment = range?.createContextualFragment(html);
        range?.insertNode(fragment);
        range?.collapse(false);
        selection?.removeAllRanges();
        selection?.addRange(range);
      } else {
        editorRef.current.innerHTML += html;
      }
      handleContentChange();
    }
  };

  const handleContentChange = () => {
    if (editorRef?.current) {
      isUpdatingRef.current = true;
      onChange?.(editorRef?.current?.innerHTML);
      setTimeout(() => {
        isUpdatingRef.current = false;
      }, 0);
    }
  };

  const execCommand = (command, value = null) => {
    document.execCommand(command, false, value);
    editorRef?.current?.focus();
    handleContentChange();
  };

  const formatButtons = [
    { command: 'bold', icon: 'Bold', label: 'Negrita' },
    { command: 'italic', icon: 'Italic', label: 'Cursiva' },
    { command: 'underline', icon: 'Underline', label: 'Subrayado' },
    { separator: true },
    { command: 'justifyLeft', icon: 'AlignLeft', label: 'Alinear izquierda' },
    { command: 'justifyCenter', icon: 'AlignCenter', label: 'Centrar' },
    { command: 'justifyRight', icon: 'AlignRight', label: 'Alinear derecha' },
    { command: 'justifyFull', icon: 'AlignJustify', label: 'Justificar' },
    { separator: true },
    { command: 'insertUnorderedList', icon: 'List', label: 'Lista con viñetas' },
    { command: 'insertOrderedList', icon: 'ListOrdered', label: 'Lista numerada' },
    { separator: true },
    { command: 'undo', icon: 'Undo', label: 'Deshacer' },
    { command: 'redo', icon: 'Redo', label: 'Rehacer' }
  ];

  const handleFontSize = (size) => {
    execCommand('fontSize', size);
  };

  const handleHeading = (level) => {
    execCommand('formatBlock', `h${level}`);
  };

  return (
    <div className="flex flex-col h-full bg-white">
      {/* Toolbar */}
      <div className="border-b border-gray-200 p-3 bg-gray-50">
        <div className="flex items-center space-x-2 flex-wrap gap-y-2">
          {/* Font Size */}
          <select
            onChange={(e) => handleFontSize(e?.target?.value)}
            className="text-sm border border-gray-300 rounded px-2 py-1 focus:ring-2 focus:ring-primary focus:border-primary"
          >
            <option value="">Tamaño</option>
            <option value="1">8pt</option>
            <option value="2">10pt</option>
            <option value="3">12pt</option>
            <option value="4">14pt</option>
            <option value="5">18pt</option>
            <option value="6">24pt</option>
            <option value="7">36pt</option>
          </select>

          {/* Headings */}
          <select
            onChange={(e) => e?.target?.value && handleHeading(e?.target?.value)}
            className="text-sm border border-gray-300 rounded px-2 py-1 focus:ring-2 focus:ring-primary focus:border-primary"
            defaultValue=""
          >
            <option value="">Estilo</option>
            <option value="1">Título 1</option>
            <option value="2">Título 2</option>
            <option value="3">Título 3</option>
            <option value="4">Título 4</option>
          </select>

          <div className="w-px h-6 bg-gray-300" />

          {/* Format Buttons */}
          {formatButtons?.map((button, index) => (
            button?.separator ? (
              <div key={index} className="w-px h-6 bg-gray-300" />
            ) : (
              <Button
                key={button?.command}
                variant="ghost"
                size="sm"
                onClick={() => execCommand(button?.command)}
                title={button?.label}
                className="p-2"
              >
                <Icon name={button?.icon} size={16} />
              </Button>
            )
          ))}

          <div className="w-px h-6 bg-gray-300" />

          {/* Font Color */}
          <input
            type="color"
            onChange={(e) => execCommand('foreColor', e?.target?.value)}
            className="w-8 h-8 rounded border border-gray-300 cursor-pointer"
            title="Color de texto"
          />

          {/* Background Color */}
          <input
            type="color"
            onChange={(e) => execCommand('hiliteColor', e?.target?.value)}
            className="w-8 h-8 rounded border border-gray-300 cursor-pointer"
            title="Color de fondo"
          />
        </div>
      </div>
      {/* Editor Content */}
      <div className="flex-1 overflow-auto">
        <div
          ref={editorRef}
          className="h-full p-6 focus:outline-none font-serif leading-relaxed"
          contentEditable={true}
          onInput={handleContentChange}
          onBlur={handleContentChange}
          style={{
            minHeight: '100%',
            fontSize: '14px',
            lineHeight: '1.6',
            color: '#333',
            backgroundColor: 'white'
          }}
          suppressContentEditableWarning={true}
        />
      </div>
      {/* Status Bar */}
      <div className="border-t border-gray-200 px-4 py-2 bg-gray-50 text-xs text-gray-600">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <span>
              Palabras: {content?.replace(/<[^>]*>/g, '')?.trim()?.split(/\s+/)?.length || 0}
            </span>
            <span>
              Caracteres: {content?.replace(/<[^>]*>/g, '')?.length || 0}
            </span>
          </div>
          
          <div className="text-gray-500">
            <Icon name="Type" size={12} className="inline mr-1" />
            Editor de texto enriquecido
          </div>
        </div>
      </div>
    </div>
  );
});

RichTextEditor.displayName = 'RichTextEditor';

export default RichTextEditor;