import React, { useState } from 'react';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Icon from '../../../components/AppIcon';

const TemplateSelector = ({ selectedTemplate, onTemplateChange, onDraftUpload }) => {
  const [searchTerm, setSearchTerm] = useState('');
  
  // Mock templates data
  const templates = [
    {
      id: 1,
      name: 'Contrato Residencial Estándar',
      category: 'residential',
      description: 'Plantilla básica para arriendo residencial',
      content: `CONTRATO DE ARRENDAMIENTO DE BIEN RAÍZ

Entre [PROPIETARIO], mayor de edad, estado civil [ESTADO_CIVIL], profesión [PROFESION], cédula de identidad N° [RUT_PROPIETARIO], domiciliado en [DIRECCION_PROPIETARIO], en adelante "EL ARRENDADOR", y [ARRENDATARIO], mayor de edad, estado civil [ESTADO_CIVIL_ARRENDATARIO], profesión [PROFESION_ARRENDATARIO], cédula de identidad N° [RUT_ARRENDATARIO], domiciliado en [DIRECCION_ARRENDATARIO], en adelante "EL ARRENDATARIO", se ha convenido el siguiente contrato de arrendamiento:

PRIMERO: OBJETO DEL CONTRATO
El ARRENDADOR da en arrendamiento al ARRENDATARIO la propiedad ubicada en [DIRECCION_PROPIEDAD], Comuna de [COMUNA], Región [REGION].

SEGUNDO: PRECIO Y FORMA DE PAGO
El precio del arrendamiento es de $[VALOR_ARRIENDO] mensuales, expresado en [MONEDA], pagadero dentro de los primeros [DIAS_PAGO] días de cada mes.

TERCERO: PLAZO
El presente contrato tendrá una duración de [PLAZO_MESES] meses, contados desde el [FECHA_INICIO] hasta el [FECHA_TERMINO].

[CLAUSULAS_ADICIONALES]`
    },
    {
      id: 2,
      name: 'Contrato Comercial',
      category: 'commercial',
      description: 'Plantilla para locales comerciales',
      content: 'Contenido del contrato comercial...'
    },
    {
      id: 3,
      name: 'Contrato Amoblado',
      category: 'furnished',
      description: 'Plantilla para propiedades amobladas',
      content: 'Contenido del contrato amoblado...'
    }
  ];

  const filteredTemplates = templates?.filter(template =>
    template?.name?.toLowerCase()?.includes(searchTerm?.toLowerCase()) ||
    template?.description?.toLowerCase()?.includes(searchTerm?.toLowerCase())
  );

  const handleFileUpload = (event) => {
    const file = event?.target?.files?.[0];
    if (file) {
      onDraftUpload?.(file);
    }
  };

  const getCategoryColor = (category) => {
    switch (category) {
      case 'residential': return 'bg-blue-100 text-blue-800';
      case 'commercial': return 'bg-purple-100 text-purple-800';
      case 'furnished': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getCategoryLabel = (category) => {
    switch (category) {
      case 'residential': return 'Residencial';
      case 'commercial': return 'Comercial';
      case 'furnished': return 'Amoblado';
      default: return 'General';
    }
  };

  return (
    <div className="p-4 h-full flex flex-col">
      <h3 className="font-semibold text-gray-900 mb-4">Seleccionar Plantilla</h3>
      {/* Search */}
      <div className="mb-4">
        <Input
          placeholder="Buscar plantillas..."
          value={searchTerm}
          onChange={(e) => setSearchTerm(e?.target?.value)}
          iconName="Search"
          iconPosition="left"
          size="sm"
        />
      </div>
      {/* Current Template */}
      {selectedTemplate && (
        <div className="mb-4 p-3 bg-primary/5 border border-primary/20 rounded-lg">
          <div className="flex items-start justify-between">
            <div className="flex-1">
              <h4 className="font-medium text-gray-900 text-sm mb-1">
                Plantilla Actual
              </h4>
              <p className="text-sm text-gray-600">{selectedTemplate?.name}</p>
              <span className={`inline-block px-2 py-1 text-xs font-medium rounded-full mt-2 ${getCategoryColor(selectedTemplate?.category)}`}>
                {getCategoryLabel(selectedTemplate?.category)}
              </span>
            </div>
            <Icon name="CheckCircle" size={20} className="text-primary mt-1" />
          </div>
        </div>
      )}
      {/* Templates List */}
      <div className="flex-1 overflow-y-auto space-y-3">
        {filteredTemplates?.map((template) => (
          <div
            key={template?.id}
            className={`p-3 rounded-lg border cursor-pointer transition-all duration-150 hover:shadow-sm ${
              selectedTemplate?.id === template?.id
                ? 'border-primary bg-primary/5' :'border-gray-200 hover:border-gray-300'
            }`}
            onClick={() => onTemplateChange?.(template)}
          >
            <div className="flex items-start justify-between mb-2">
              <h4 className="font-medium text-gray-900 text-sm line-clamp-1">
                {template?.name}
              </h4>
              {selectedTemplate?.id === template?.id && (
                <Icon name="CheckCircle" size={16} className="text-primary" />
              )}
            </div>
            
            <p className="text-xs text-gray-600 mb-2 line-clamp-2">
              {template?.description}
            </p>
            
            <span className={`inline-block px-2 py-1 text-xs font-medium rounded-full ${getCategoryColor(template?.category)}`}>
              {getCategoryLabel(template?.category)}
            </span>
          </div>
        ))}
      </div>
      {filteredTemplates?.length === 0 && (
        <div className="flex-1 flex items-center justify-center text-center">
          <div>
            <Icon name="FileText" size={32} className="text-gray-400 mx-auto mb-2" />
            <p className="text-sm text-gray-500">No se encontraron plantillas</p>
          </div>
        </div>
      )}
      {/* Upload Draft */}
      <div className="mt-4 pt-4 border-t border-gray-200">
        <h4 className="font-medium text-gray-900 mb-3 text-sm">Subir Borrador</h4>
        
        <div className="space-y-2">
          <Button
            variant="outline"
            iconName="Upload"
            iconPosition="left"
            onClick={() => document.getElementById('draft-upload')?.click()}
            className="w-full"
            size="sm"
          >
            Subir Documento
          </Button>
          
          <input
            id="draft-upload"
            type="file"
            accept=".docx,.doc,.txt"
            onChange={handleFileUpload}
            className="hidden"
          />
          
          <p className="text-xs text-gray-500">
            Formatos soportados: DOCX, DOC, TXT
          </p>
        </div>
      </div>
      {/* Quick Actions */}
      <div className="mt-4 pt-4 border-t border-gray-200">
        <h4 className="font-medium text-gray-900 mb-3 text-sm">Acciones Rápidas</h4>
        
        <div className="space-y-2">
          <Button
            variant="ghost"
            iconName="FileEdit"
            iconPosition="left"
            onClick={() => window.open('/contract-templates-library', '_blank')}
            className="w-full justify-start"
            size="sm"
          >
            Gestionar Plantillas
          </Button>
          
          <Button
            variant="ghost"
            iconName="Plus"
            iconPosition="left"
            onClick={() => {
              const emptyTemplate = {
                id: 'new',
                name: 'Contrato en Blanco',
                category: 'custom',
                description: 'Comienza desde cero',
                content: ''
              };
              onTemplateChange?.(emptyTemplate);
            }}
            className="w-full justify-start"
            size="sm"
          >
            Empezar desde Cero
          </Button>
        </div>
      </div>
    </div>
  );
};

export default TemplateSelector;