import React, { useState, useRef } from 'react';
import { useNavigate } from 'react-router-dom';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import Button from '../../components/ui/Button';
import Icon from '../../components/AppIcon';
import { jsPDF } from 'jspdf';
import { Document, Paragraph, TextRun, Packer, HeadingLevel } from 'docx';


const ContractDraftingWorkspace = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [contractData, setContractData] = useState({
    signatureDate: '',
    startDate: '',
    endDate: '',
    ownerName: '',
    ownerNationality: '',
    ownerRut: '',
    tenantName: '',
    tenantNationality: '',
    tenantRut: '',
    avalName: '',
    avalNationality: '',
    avalRut: '',
    propertyAddress: '',
    parkingNumber: '',
    storageNumber: '',
    propertyRole: '',
    rentPrice: '',
    adjustmentIndex: 'IPC',
    otherClauses: ''
  });
  
  const [inventoryItems, setInventoryItems] = useState([
    'Puerta principal madera roble',
    'Ventanas de vidrio templado',
    'Alfombra en sala de estar',
    'Muros en buen estado'
  ]);
  
  const [propertyImages, setPropertyImages] = useState([]);
  const [idImages, setIdImages] = useState([]);
  const [contractTemplate, setContractTemplate] = useState(`CONTRATO DE ARRIENDO

En Santiago, a [FECHA_FIRMA], entre:

[PROPIETARIO_NOMBRE], [PROPIETARIO_NACIONALIDAD], mayor de edad, portador del RUT N° [PROPIETARIO_RUT], en adelante "EL ARRENDADOR"; y

[ARRENDATARIO_NOMBRE], [ARRENDATARIO_NACIONALIDAD], mayor de edad, portador del RUT N° [ARRENDATARIO_RUT], en adelante "EL ARRENDATARIO";

Y en calidad de aval:

[AVAL_NOMBRE], [AVAL_NACIONALIDAD], mayor de edad, portador del RUT N° [AVAL_RUT], en adelante "EL AVAL";

Celebran el presente contrato de arrendamiento, el que se regirá por las siguientes cláusulas:

PRIMERA: EL ARRENDADOR da en arriendo a EL ARRENDATARIO el inmueble ubicado en [DIRECCION_PROPIEDAD].

SEGUNDA: El plazo del arriendo será desde el [FECHA_INICIO] hasta el [FECHA_TERMINO].

TERCERA: El valor del arriendo será de $[PRECIO_ARRIENDO] ([PRECIO_LETRAS]) pesos mensuales, el cual se reajustará anualmente según el [INDICE_REAJUSTE].

CUARTA: La propiedad cuenta con:
- Rol de avalúo: [ROL_PROPIEDAD]
- Estacionamiento N°: [NUMERO_ESTACIONAMIENTO]
- Bodega N°: [NUMERO_BODEGA]

QUINTA: El inmueble se entrega con los siguientes elementos:
[INVENTARIO]

SEXTA: [OTRAS_CLÁUSULAS]

ANEXOS:
- Fotografías del estado actual de la propiedad (adjuntas)
- Fotografías de carnet de las partes (adjuntas)

[FIRMAS]`);

  const [generatedContract, setGeneratedContract] = useState('');
  const navigate = useNavigate();
  const contractRef = useRef(null);

  const handleSidebarToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleSidebarClose = () => {
    setIsSidebarOpen(false);
  };

  const handleInputChange = (field, value) => {
    setContractData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const addInventoryItem = (itemText = '') => {
    setInventoryItems(prev => [...prev, itemText]);
  };

  const removeInventoryItem = (index) => {
    setInventoryItems(prev => prev?.filter((_, i) => i !== index));
  };

  const updateInventoryItem = (index, value) => {
    setInventoryItems(prev => prev?.map((item, i) => i === index ? value : item));
  };

  const handleImageUpload = (files, type) => {
    const imageArray = Array.from(files)?.slice(0, type === 'property' ? 6 : 4);
    const promises = imageArray?.map(file => {
      return new Promise((resolve) => {
        const reader = new FileReader();
        reader.onload = (e) => resolve(e.target.result);
        reader.readAsDataURL(file);
      });
    });

    Promise.all(promises)?.then(results => {
      if (type === 'property') {
        setPropertyImages(results);
      } else {
        setIdImages(results);
      }
    });
  };

  const formatDate = (dateString) => {
    if (!dateString) return '';
    const date = new Date(dateString);
    const day = date?.getDate();
    const month = date?.getMonth() + 1;
    const year = date?.getFullYear();
    return `${day}/${month}/${year}`;
  };

  const formatNumber = (num) => {
    return new Intl.NumberFormat('es-CL')?.format(num);
  };

  const numberToWords = (num) => {
    const unidades = ['', 'uno', 'dos', 'tres', 'cuatro', 'cinco', 'seis', 'siete', 'ocho', 'nueve'];
    const decenas = ['', 'diez', 'veinte', 'treinta', 'cuarenta', 'cincuenta', 'sesenta', 'setenta', 'ochenta', 'noventa'];
    const especiales = ['diez', 'once', 'doce', 'trece', 'catorce', 'quince', 'dieciséis', 'diecisiete', 'dieciocho', 'diecinueve'];
    const centenas = ['', 'ciento', 'doscientos', 'trescientos', 'cuatrocientos', 'quinientos', 'seiscientos', 'setecientos', 'ochocientos', 'novecientos'];

    if (num === 0) return 'cero';
    if (num < 10) return unidades?.[num];
    if (num < 20) return especiales?.[num - 10];

    let output = '';
    const millones = Math.floor(num / 1000000);
    num = num % 1000000;

    const miles = Math.floor(num / 1000);
    num = num % 1000;

    const cientos = Math.floor(num / 100);
    const resto = num % 100;

    if (millones > 0) {
      output += numberToWords(millones) + ' millones ';
    }

    if (miles > 0) {
      output += numberToWords(miles) + ' mil ';
    }

    if (cientos > 0) {
      output += centenas?.[cientos] + ' ';
    }

    if (resto > 0) {
      if (resto < 10) {
        output += unidades?.[resto];
      } else if (resto < 20) {
        output += especiales?.[resto - 10];
      } else {
        const decena = Math.floor(resto / 10);
        const unidad = resto % 10;
        output += decenas?.[decena];
        if (unidad !== 0) {
          output += ' y ' + unidades?.[unidad];
        }
      }
    }

    return output?.trim();
  };

  const generateContract = () => {
    let contractText = contractTemplate;

    // Replace placeholders
    contractText = contractText?.replace(/\[FECHA_FIRMA\]/g, formatDate(contractData?.signatureDate));
    contractText = contractText?.replace(/\[FECHA_INICIO\]/g, formatDate(contractData?.startDate));
    contractText = contractText?.replace(/\[FECHA_TERMINO\]/g, formatDate(contractData?.endDate));
    contractText = contractText?.replace(/\[PROPIETARIO_NOMBRE\]/g, contractData?.ownerName);
    contractText = contractText?.replace(/\[PROPIETARIO_NACIONALIDAD\]/g, contractData?.ownerNationality?.toLowerCase());
    contractText = contractText?.replace(/\[PROPIETARIO_RUT\]/g, contractData?.ownerRut);
    contractText = contractText?.replace(/\[ARRENDATARIO_NOMBRE\]/g, contractData?.tenantName);
    contractText = contractText?.replace(/\[ARRENDATARIO_NACIONALIDAD\]/g, contractData?.tenantNationality?.toLowerCase());
    contractText = contractText?.replace(/\[ARRENDATARIO_RUT\]/g, contractData?.tenantRut);
    contractText = contractText?.replace(/\[AVAL_NOMBRE\]/g, contractData?.avalName || 'No aplica');
    contractText = contractText?.replace(/\[AVAL_NACIONALIDAD\]/g, contractData?.avalNationality?.toLowerCase() || '');
    contractText = contractText?.replace(/\[AVAL_RUT\]/g, contractData?.avalRut || '');
    contractText = contractText?.replace(/\[DIRECCION_PROPIEDAD\]/g, contractData?.propertyAddress);
    contractText = contractText?.replace(/\[ROL_PROPIEDAD\]/g, contractData?.propertyRole || 'No especificado');
    contractText = contractText?.replace(/\[NUMERO_ESTACIONAMIENTO\]/g, contractData?.parkingNumber || 'No aplica');
    contractText = contractText?.replace(/\[NUMERO_BODEGA\]/g, contractData?.storageNumber || 'No aplica');
    contractText = contractText?.replace(/\[PRECIO_ARRIENDO\]/g, formatNumber(contractData?.rentPrice));
    contractText = contractText?.replace(/\[PRECIO_LETRAS\]/g, numberToWords(parseInt(contractData?.rentPrice)));
    contractText = contractText?.replace(/\[INDICE_REAJUSTE\]/g, contractData?.adjustmentIndex);
    contractText = contractText?.replace(/\[INVENTARIO\]/g, inventoryItems?.map(item => `- ${item}`)?.join('\n'));
    contractText = contractText?.replace(/\[OTRAS_CLÁUSULAS\]/g, contractData?.otherClauses || 'No se incluyen cláusulas adicionales.');

    const signatures = `
FIRMAS:

_________________________          _________________________          _________________________
${contractData?.ownerName}           ${contractData?.tenantName}         ${contractData?.avalName || 'No aplica'}
Arrendador                         Arrendatario                       Aval
RUT: ${contractData?.ownerRut}       RUT: ${contractData?.tenantRut}     RUT: ${contractData?.avalRut || ''}
`;

    contractText = contractText?.replace(/\[FIRMAS\]/g, signatures);
    setGeneratedContract(contractText);
  };

  const generatePDF = async () => {
    const pdf = new jsPDF();
    const lines = pdf?.splitTextToSize(generatedContract, 180);
    pdf?.text(lines, 10, 10);

    // Add property images
    if (propertyImages?.length > 0) {
      pdf?.addPage();
      pdf?.text('Fotografías de la propiedad:', 10, 10);
      let y = 20;
      for (let i = 0; i < propertyImages?.length; i++) {
        if (i > 0 && i % 2 === 0) {
          pdf?.addPage();
          y = 20;
        }
        const x = i % 2 === 0 ? 10 : 105;
        pdf?.addImage(propertyImages?.[i], 'JPEG', x, y, 85, 60);
        if (i % 2 !== 0) y += 70;
      }
    }

    // Add ID images
    if (idImages?.length > 0) {
      pdf?.addPage();
      pdf?.text('Fotografías de carnet:', 10, 10);
      let y = 20;
      for (let i = 0; i < idImages?.length; i++) {
        if (i > 0 && i % 2 === 0) {
          pdf?.addPage();
          y = 20;
        }
        const x = i % 2 === 0 ? 10 : 105;
        pdf?.addImage(idImages?.[i], 'JPEG', x, y, 85, 60);
        if (i % 2 !== 0) y += 70;
      }
    }

    pdf?.save('contrato_arriendo.pdf');
  };

  const generateWord = async () => {
    const doc = new Document({
      sections: [{
        properties: {},
        children: [
          new Paragraph({
            text: "CONTRATO DE ARRIENDO",
            heading: HeadingLevel.HEADING_1,
            spacing: { after: 200 }
          }),
          new Paragraph({
            children: [
              new TextRun({
                text: generatedContract,
                size: 24
              })
            ]
          })
        ]
      }]
    });

    const blob = await Packer?.toBlob(doc);
    const link = document.createElement('a');
    link.href = URL.createObjectURL(blob);
    link.download = 'contrato_arriendo.docx';
    link?.click();
  };

  const clearForm = () => {
    if (window.confirm('¿Estás seguro de que deseas limpiar todo el formulario?')) {
      setContractData({
        signatureDate: '',
        startDate: '',
        endDate: '',
        ownerName: '',
        ownerNationality: '',
        ownerRut: '',
        tenantName: '',
        tenantNationality: '',
        tenantRut: '',
        avalName: '',
        avalNationality: '',
        avalRut: '',
        propertyAddress: '',
        parkingNumber: '',
        storageNumber: '',
        propertyRole: '',
        rentPrice: '',
        adjustmentIndex: 'IPC',
        otherClauses: ''
      });
      setInventoryItems([
        'Puerta principal madera roble',
        'Ventanas de vidrio templado',
        'Alfombra en sala de estar',
        'Muros en buen estado'
      ]);
      setPropertyImages([]);
      setIdImages([]);
      setGeneratedContract('');
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleSidebarToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={handleSidebarClose} />
      <main className="lg:ml-60 pt-16">
        <div className="p-6">
          <NavigationBreadcrumbs />
          <div className="flex items-center justify-between mb-6">
            <h1 className="text-2xl font-bold text-gray-900">
              Generador de Contratos de Arriendo
            </h1>
            <Button
              variant="ghost"
              iconName="ArrowLeft"
              iconPosition="left"
              onClick={() => navigate(-1)}
              size="sm"
            >
              Volver
            </Button>
          </div>

          <div className="max-w-7xl mx-auto space-y-8">
            {/* Contract Template Section */}
            <div className="bg-white rounded-lg shadow p-6">
              <h2 className="text-lg font-semibold text-gray-900 mb-4">Plantilla del Contrato</h2>
              <textarea
                value={contractTemplate}
                onChange={(e) => setContractTemplate(e?.target?.value)}
                className="w-full h-64 p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                placeholder="Ingrese la plantilla del contrato..."
              />
              <Button
                onClick={() => generateContract()}
                className="mt-4"
                iconName="RefreshCw"
                iconPosition="left"
              >
                Actualizar Plantilla
              </Button>
            </div>

            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
              {/* Left Column - Forms */}
              <div className="space-y-6">
                {/* Contract Dates */}
                <div className="bg-white rounded-lg shadow p-6">
                  <h2 className="text-lg font-semibold text-gray-900 mb-4">Datos del Contrato</h2>
                  <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Fecha de Firma
                      </label>
                      <input
                        type="date"
                        value={contractData?.signatureDate}
                        onChange={(e) => handleInputChange('signatureDate', e?.target?.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Fecha de Inicio
                      </label>
                      <input
                        type="date"
                        value={contractData?.startDate}
                        onChange={(e) => handleInputChange('startDate', e?.target?.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Fecha de Término
                      </label>
                      <input
                        type="date"
                        value={contractData?.endDate}
                        onChange={(e) => handleInputChange('endDate', e?.target?.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                      />
                    </div>
                  </div>
                </div>

                {/* Owner Information */}
                <div className="bg-white rounded-lg shadow p-6">
                  <h2 className="text-lg font-semibold text-gray-900 mb-4">Datos del Propietario</h2>
                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Nombre Completo
                      </label>
                      <input
                        type="text"
                        value={contractData?.ownerName}
                        onChange={(e) => handleInputChange('ownerName', e?.target?.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                      />
                    </div>
                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Nacionalidad
                        </label>
                        <input
                          type="text"
                          value={contractData?.ownerNationality}
                          onChange={(e) => handleInputChange('ownerNationality', e?.target?.value)}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          RUT
                        </label>
                        <input
                          type="text"
                          value={contractData?.ownerRut}
                          onChange={(e) => handleInputChange('ownerRut', e?.target?.value)}
                          placeholder="12.345.678-9"
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                    </div>
                  </div>
                </div>

                {/* Tenant Information */}
                <div className="bg-white rounded-lg shadow p-6">
                  <h2 className="text-lg font-semibold text-gray-900 mb-4">Datos del Arrendatario</h2>
                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Nombre Completo
                      </label>
                      <input
                        type="text"
                        value={contractData?.tenantName}
                        onChange={(e) => handleInputChange('tenantName', e?.target?.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                      />
                    </div>
                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Nacionalidad
                        </label>
                        <input
                          type="text"
                          value={contractData?.tenantNationality}
                          onChange={(e) => handleInputChange('tenantNationality', e?.target?.value)}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          RUT
                        </label>
                        <input
                          type="text"
                          value={contractData?.tenantRut}
                          onChange={(e) => handleInputChange('tenantRut', e?.target?.value)}
                          placeholder="12.345.678-9"
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                    </div>
                  </div>
                </div>

                {/* Guarantor Information */}
                <div className="bg-white rounded-lg shadow p-6">
                  <h2 className="text-lg font-semibold text-gray-900 mb-4">Datos del Aval</h2>
                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Nombre Completo
                      </label>
                      <input
                        type="text"
                        value={contractData?.avalName}
                        onChange={(e) => handleInputChange('avalName', e?.target?.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                      />
                    </div>
                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Nacionalidad
                        </label>
                        <input
                          type="text"
                          value={contractData?.avalNationality}
                          onChange={(e) => handleInputChange('avalNationality', e?.target?.value)}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          RUT
                        </label>
                        <input
                          type="text"
                          value={contractData?.avalRut}
                          onChange={(e) => handleInputChange('avalRut', e?.target?.value)}
                          placeholder="12.345.678-9"
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                    </div>
                  </div>
                </div>

                {/* Property Information */}
                <div className="bg-white rounded-lg shadow p-6">
                  <h2 className="text-lg font-semibold text-gray-900 mb-4">Datos de la Propiedad</h2>
                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Dirección Completa
                      </label>
                      <input
                        type="text"
                        value={contractData?.propertyAddress}
                        onChange={(e) => handleInputChange('propertyAddress', e?.target?.value)}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                      />
                    </div>
                    <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Número de Estacionamiento
                        </label>
                        <input
                          type="text"
                          value={contractData?.parkingNumber}
                          onChange={(e) => handleInputChange('parkingNumber', e?.target?.value)}
                          placeholder="Ej: E-15"
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Número de Bodega
                        </label>
                        <input
                          type="text"
                          value={contractData?.storageNumber}
                          onChange={(e) => handleInputChange('storageNumber', e?.target?.value)}
                          placeholder="Ej: B-8"
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Rol de la Propiedad
                        </label>
                        <input
                          type="text"
                          value={contractData?.propertyRole}
                          onChange={(e) => handleInputChange('propertyRole', e?.target?.value)}
                          placeholder="Ej: 123-456"
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                    </div>
                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Precio de Arriendo ($)
                        </label>
                        <input
                          type="number"
                          value={contractData?.rentPrice}
                          onChange={(e) => handleInputChange('rentPrice', e?.target?.value)}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                      </div>
                      <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">
                          Índice de Reajuste
                        </label>
                        <select
                          value={contractData?.adjustmentIndex}
                          onChange={(e) => handleInputChange('adjustmentIndex', e?.target?.value)}
                          className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        >
                          <option value="IPC">IPC</option>
                          <option value="UF">UF</option>
                          <option value="Ninguno">Sin reajuste</option>
                        </select>
                      </div>
                    </div>
                  </div>
                </div>

                {/* Inventory */}
                <div className="bg-white rounded-lg shadow p-6">
                  <h2 className="text-lg font-semibold text-gray-900 mb-4">Inventario de la Propiedad</h2>
                  <div className="space-y-2 mb-4">
                    {inventoryItems?.map((item, index) => (
                      <div key={index} className="flex items-center space-x-2">
                        <input
                          type="text"
                          value={item}
                          onChange={(e) => updateInventoryItem(index, e?.target?.value)}
                          className="flex-1 px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                        />
                        <Button
                          variant="ghost"
                          size="sm"
                          onClick={() => removeInventoryItem(index)}
                          iconName="X"
                        />
                      </div>
                    ))}
                  </div>
                  <Button
                    variant="outline"
                    onClick={() => addInventoryItem('')}
                    iconName="Plus"
                    iconPosition="left"
                  >
                    Agregar Item
                  </Button>
                </div>

                {/* Property Images */}
                <div className="bg-white rounded-lg shadow p-6">
                  <h2 className="text-lg font-semibold text-gray-900 mb-4">Fotografías de la Propiedad</h2>
                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Subir imágenes (máx. 6)
                      </label>
                      <input
                        type="file"
                        accept="image/*"
                        multiple
                        onChange={(e) => handleImageUpload(e?.target?.files, 'property')}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                      />
                    </div>
                    {propertyImages?.length > 0 && (
                      <div className="grid grid-cols-2 sm:grid-cols-3 gap-4">
                        {propertyImages?.map((image, index) => (
                          <img
                            key={index}
                            src={image}
                            alt={`Propiedad ${index + 1}`}
                            className="w-full h-32 object-cover rounded-lg border border-gray-300"
                          />
                        ))}
                      </div>
                    )}
                  </div>
                </div>

                {/* ID Images */}
                <div className="bg-white rounded-lg shadow p-6">
                  <h2 className="text-lg font-semibold text-gray-900 mb-4">Fotografías de Carnet</h2>
                  <div className="space-y-4">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-2">
                        Subir fotografías de carnet (máx. 4)
                      </label>
                      <input
                        type="file"
                        accept="image/*"
                        multiple
                        onChange={(e) => handleImageUpload(e?.target?.files, 'id')}
                        className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                      />
                    </div>
                    {idImages?.length > 0 && (
                      <div className="grid grid-cols-2 sm:grid-cols-4 gap-4">
                        {idImages?.map((image, index) => (
                          <img
                            key={index}
                            src={image}
                            alt={`Carnet ${index + 1}`}
                            className="w-full h-32 object-cover rounded-lg border border-gray-300"
                          />
                        ))}
                      </div>
                    )}
                  </div>
                </div>

                {/* Other Clauses */}
                <div className="bg-white rounded-lg shadow p-6">
                  <h2 className="text-lg font-semibold text-gray-900 mb-4">Otras Cláusulas</h2>
                  <textarea
                    value={contractData?.otherClauses}
                    onChange={(e) => handleInputChange('otherClauses', e?.target?.value)}
                    rows="4"
                    placeholder="Texto adicional para el contrato..."
                    className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent"
                  />
                </div>

                {/* Action Buttons */}
                <div className="bg-white rounded-lg shadow p-6">
                  <div className="flex flex-wrap gap-3">
                    <Button
                      onClick={generateContract}
                      iconName="FileText"
                      iconPosition="left"
                    >
                      Generar Contrato
                    </Button>
                    <Button
                      variant="outline"
                      onClick={generatePDF}
                      iconName="Download"
                      iconPosition="left"
                      disabled={!generatedContract}
                    >
                      Descargar PDF
                    </Button>
                    <Button
                      variant="outline"
                      onClick={generateWord}
                      iconName="FileText"
                      iconPosition="left"
                      disabled={!generatedContract}
                    >
                      Descargar Word
                    </Button>
                    <Button
                      variant="ghost"
                      onClick={clearForm}
                      iconName="Trash2"
                      iconPosition="left"
                    >
                      Limpiar Formulario
                    </Button>
                  </div>
                </div>
              </div>

              {/* Right Column - Preview */}
              <div className="space-y-6">
                <div className="bg-white rounded-lg shadow">
                  <div className="p-6 border-b border-gray-200">
                    <h2 className="text-lg font-semibold text-gray-900">Vista Previa del Contrato</h2>
                  </div>
                  <div className="p-6">
                    {generatedContract ? (
                      <div
                        ref={contractRef}
                        className="prose prose-sm max-w-none whitespace-pre-line text-sm font-mono bg-gray-50 p-4 rounded-lg border"
                      >
                        {generatedContract}
                      </div>
                    ) : (
                      <div className="text-center py-12 text-gray-500">
                        <Icon name="FileText" size={48} className="mx-auto mb-4 text-gray-400" />
                        <p>Presiona "Generar Contrato" para ver la vista previa</p>
                      </div>
                    )}
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </main>
    </div>
  );
};

export default ContractDraftingWorkspace;