import React from 'react';
import Icon from '../../../components/AppIcon';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const ContractBuilder = ({ contractData, onDataChange }) => {
  const propertyTypes = [
    { value: 'apartment', label: 'Departamento' },
    { value: 'house', label: 'Casa' },
    { value: 'commercial', label: 'Local Comercial' },
    { value: 'office', label: 'Oficina' },
    { value: 'warehouse', label: 'Bodega' }
  ];

  const renewalOptions = [
    { value: 'automatic', label: 'Renovación Automática' },
    { value: 'manual', label: 'Renovación Manual' },
    { value: 'fixed_term', label: 'Plazo Fijo sin Renovación' }
  ];

  const handleInputChange = (section, field, value) => {
    onDataChange(section, { [field]: value });
  };

  const calculateEndDate = (startDate, duration) => {
    if (!startDate || !duration) return '';
    const start = new Date(startDate);
    const end = new Date(start);
    end?.setMonth(start?.getMonth() + duration);
    return end?.toISOString()?.split('T')?.[0];
  };

  return (
    <div className="space-y-6">
      {/* Tenant Information */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
          <Icon name="User" size={20} className="mr-2 text-blue-600" />
          Información del Inquilino
        </h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Nombre Completo *
            </label>
            <Input
              type="text"
              value={contractData?.tenantInfo?.name || ''}
              onChange={(e) => handleInputChange('tenantInfo', 'name', e?.target?.value)}
              placeholder="Ej: María González Pérez"
              className="w-full"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              RUT *
            </label>
            <Input
              type="text"
              value={contractData?.tenantInfo?.rut || ''}
              onChange={(e) => handleInputChange('tenantInfo', 'rut', e?.target?.value)}
              placeholder="Ej: 12.345.678-9"
              className="w-full"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Email
            </label>
            <Input
              type="email"
              value={contractData?.tenantInfo?.email || ''}
              onChange={(e) => handleInputChange('tenantInfo', 'email', e?.target?.value)}
              placeholder="Ej: maria@email.com"
              className="w-full"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Teléfono
            </label>
            <Input
              type="tel"
              value={contractData?.tenantInfo?.phone || ''}
              onChange={(e) => handleInputChange('tenantInfo', 'phone', e?.target?.value)}
              placeholder="Ej: +56 9 1234 5678"
              className="w-full"
            />
          </div>
          
          <div className="md:col-span-2">
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Dirección de Contacto
            </label>
            <Input
              type="text"
              value={contractData?.tenantInfo?.address || ''}
              onChange={(e) => handleInputChange('tenantInfo', 'address', e?.target?.value)}
              placeholder="Ej: Av. Providencia 1234, Providencia"
              className="w-full"
            />
          </div>
        </div>
      </div>
      {/* Property Details */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
          <Icon name="Home" size={20} className="mr-2 text-green-600" />
          Detalles de la Propiedad
        </h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="md:col-span-2">
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Dirección de la Propiedad *
            </label>
            <Input
              type="text"
              value={contractData?.propertyDetails?.address || ''}
              onChange={(e) => handleInputChange('propertyDetails', 'address', e?.target?.value)}
              placeholder="Ej: Av. Apoquindo 3000, Las Condes"
              className="w-full"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Tipo de Propiedad
            </label>
            <Select
              options={propertyTypes}
              value={contractData?.propertyDetails?.type || ''}
              onChange={(value) => handleInputChange('propertyDetails', 'type', value)}
              placeholder="Seleccionar tipo..."
              className="w-full"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Superficie (m²)
            </label>
            <Input
              type="number"
              value={contractData?.propertyDetails?.area || ''}
              onChange={(e) => handleInputChange('propertyDetails', 'area', e?.target?.value)}
              placeholder="Ej: 85"
              className="w-full"
              min="0"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Cantidad de Dormitorios
            </label>
            <Input
              type="number"
              value={contractData?.propertyDetails?.rooms || ''}
              onChange={(e) => handleInputChange('propertyDetails', 'rooms', e?.target?.value)}
              placeholder="Ej: 3"
              className="w-full"
              min="0"
              max="10"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Cantidad de Baños
            </label>
            <Input
              type="number"
              value={contractData?.propertyDetails?.bathrooms || ''}
              onChange={(e) => handleInputChange('propertyDetails', 'bathrooms', e?.target?.value)}
              placeholder="Ej: 2"
              className="w-full"
              min="0"
              max="10"
            />
          </div>
        </div>
      </div>
      {/* Rental Terms */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
          <Icon name="Calendar" size={20} className="mr-2 text-purple-600" />
          Términos del Arriendo
        </h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Fecha de Inicio *
            </label>
            <Input
              type="date"
              value={contractData?.rentalTerms?.startDate || ''}
              onChange={(e) => {
                const startDate = e?.target?.value;
                handleInputChange('rentalTerms', 'startDate', startDate);
                // Auto-calculate end date
                const endDate = calculateEndDate(startDate, contractData?.rentalTerms?.duration);
                if (endDate) {
                  handleInputChange('rentalTerms', 'endDate', endDate);
                }
              }}
              className="w-full"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Duración (meses)
            </label>
            <Input
              type="number"
              value={contractData?.rentalTerms?.duration || 12}
              onChange={(e) => {
                const duration = parseInt(e?.target?.value);
                handleInputChange('rentalTerms', 'duration', duration);
                // Auto-calculate end date
                const endDate = calculateEndDate(contractData?.rentalTerms?.startDate, duration);
                if (endDate) {
                  handleInputChange('rentalTerms', 'endDate', endDate);
                }
              }}
              placeholder="12"
              className="w-full"
              min="1"
              max="120"
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Fecha de Término
            </label>
            <Input
              type="date"
              value={contractData?.rentalTerms?.endDate || ''}
              onChange={(e) => handleInputChange('rentalTerms', 'endDate', e?.target?.value)}
              className="w-full"
              readOnly
            />
          </div>
          
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Opción de Renovación
            </label>
            <Select
              options={renewalOptions}
              value={contractData?.rentalTerms?.renewalOptions || 'automatic'}
              onChange={(value) => handleInputChange('rentalTerms', 'renewalOptions', value)}
              className="w-full"
            />
          </div>
        </div>

        {/* Duration Summary */}
        {contractData?.rentalTerms?.startDate && contractData?.rentalTerms?.endDate && (
          <div className="mt-4 p-4 bg-blue-50 rounded-lg">
            <div className="flex items-center space-x-2">
              <Icon name="Info" size={16} className="text-blue-600" />
              <span className="text-sm text-blue-800">
                <strong>Duración del Contrato:</strong> {' '}
                {contractData?.rentalTerms?.duration} meses ({' '}
                {new Date(contractData?.rentalTerms?.startDate)?.toLocaleDateString('es-CL')} - {' '}
                {new Date(contractData?.rentalTerms?.endDate)?.toLocaleDateString('es-CL')})
              </span>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default ContractBuilder;