import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const ContractList = ({ contracts, onEdit, onDelete, onExport, onSign }) => {
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [sortConfig, setSortConfig] = useState({ key: 'createdAt', direction: 'desc' });

  const statusOptions = [
    { value: '', label: 'Todos los Estados' },
    { value: 'draft', label: 'Borrador' },
    { value: 'pending_signature', label: 'Pendiente de Firma' },
    { value: 'active', label: 'Activo' },
    { value: 'expired', label: 'Expirado' }
  ];

  const statusStyles = {
    draft: { bg: 'bg-gray-100', text: 'text-gray-700', icon: 'FileText' },
    pending_signature: { bg: 'bg-yellow-100', text: 'text-yellow-700', icon: 'Clock' },
    active: { bg: 'bg-green-100', text: 'text-green-700', icon: 'CheckCircle' },
    expired: { bg: 'bg-red-100', text: 'text-red-700', icon: 'XCircle' }
  };

  const filteredContracts = contracts?.filter(contract => {
    const matchesSearch = !searchTerm || 
      contract?.tenantInfo?.name?.toLowerCase()?.includes(searchTerm?.toLowerCase()) ||
      contract?.propertyDetails?.address?.toLowerCase()?.includes(searchTerm?.toLowerCase());
    
    const matchesStatus = !statusFilter || contract?.status === statusFilter;
    
    return matchesSearch && matchesStatus;
  });

  const sortedContracts = [...filteredContracts]?.sort((a, b) => {
    const aValue = a?.[sortConfig?.key];
    const bValue = b?.[sortConfig?.key];
    
    if (sortConfig?.direction === 'asc') {
      return aValue > bValue ? 1 : -1;
    } else {
      return aValue < bValue ? 1 : -1;
    }
  });

  const handleSort = (key) => {
    setSortConfig(prev => ({
      key,
      direction: prev?.key === key && prev?.direction === 'asc' ? 'desc' : 'asc'
    }));
  };

  const formatDate = (dateString) => {
    return new Date(dateString)?.toLocaleDateString('es-CL');
  };

  return (
    <div className="space-y-6">
      {/* Filters and Search */}
      <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-4 md:space-y-0 md:space-x-4">
        <div className="flex-1 max-w-md">
          <Input
            type="search"
            placeholder="Buscar por inquilino o propiedad..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e?.target?.value)}
            className="w-full"
          />
        </div>
        
        <div className="flex space-x-3">
          <Select
            options={statusOptions}
            value={statusFilter}
            onChange={setStatusFilter}
            className="w-48"
          />
          <Button
            variant="outline"
            iconName="Download"
            iconPosition="left"
            onClick={() => console.log('Export contracts list')}
          >
            Exportar Lista
          </Button>
        </div>
      </div>
      {/* Contracts Table */}
      <div className="bg-white border border-gray-200 rounded-lg overflow-hidden">
        <div className="px-6 py-4 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">Lista de Contratos</h3>
          <p className="text-sm text-gray-600 mt-1">
            {sortedContracts?.length} contratos encontrados
          </p>
        </div>

        {sortedContracts?.length > 0 ? (
          <div className="overflow-x-auto">
            <table className="w-full">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    <button
                      onClick={() => handleSort('tenantInfo.name')}
                      className="flex items-center space-x-1 hover:text-gray-700"
                    >
                      <span>Inquilino</span>
                      <Icon name="ArrowUpDown" size={14} />
                    </button>
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Propiedad
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    <button
                      onClick={() => handleSort('paymentConfig.rentAmount')}
                      className="flex items-center space-x-1 hover:text-gray-700"
                    >
                      <span>Arriendo</span>
                      <Icon name="ArrowUpDown" size={14} />
                    </button>
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Período
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    <button
                      onClick={() => handleSort('status')}
                      className="flex items-center space-x-1 hover:text-gray-700"
                    >
                      <span>Estado</span>
                      <Icon name="ArrowUpDown" size={14} />
                    </button>
                  </th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                    <button
                      onClick={() => handleSort('createdAt')}
                      className="flex items-center space-x-1 hover:text-gray-700"
                    >
                      <span>Creado</span>
                      <Icon name="ArrowUpDown" size={14} />
                    </button>
                  </th>
                  <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                    Acciones
                  </th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {sortedContracts?.map((contract) => {
                  const statusStyle = statusStyles?.[contract?.status] || statusStyles?.draft;
                  
                  return (
                    <tr key={contract?.id} className="hover:bg-gray-50">
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div>
                          <div className="text-sm font-medium text-gray-900">
                            {contract?.tenantInfo?.name}
                          </div>
                          <div className="text-sm text-gray-500">
                            {contract?.tenantInfo?.email}
                          </div>
                        </div>
                      </td>
                      <td className="px-6 py-4">
                        <div className="text-sm text-gray-900">
                          {contract?.propertyDetails?.address}
                        </div>
                        <div className="text-sm text-gray-500">
                          {contract?.propertyDetails?.type}
                        </div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <div className="text-sm font-medium text-gray-900">
                          ${contract?.paymentConfig?.rentAmount?.toLocaleString()} {contract?.paymentConfig?.currency}
                        </div>
                        {contract?.paymentConfig?.ipcAdjustment && (
                          <div className="text-xs text-green-600">Con reajuste IPC</div>
                        )}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                        <div>{formatDate(contract?.rentalTerms?.startDate)}</div>
                        <div className="text-gray-500">a {formatDate(contract?.rentalTerms?.endDate)}</div>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap">
                        <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${statusStyle?.bg} ${statusStyle?.text}`}>
                          <Icon name={statusStyle?.icon} size={12} className="mr-1" />
                          {statusOptions?.find(s => s?.value === contract?.status)?.label || 'Desconocido'}
                        </span>
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        {formatDate(contract?.createdAt)}
                      </td>
                      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                        <div className="flex items-center justify-end space-x-2">
                          {contract?.status === 'draft' && (
                            <Button
                              variant="ghost"
                              size="sm"
                              iconName="Send"
                              onClick={() => onSign(contract?.id)}
                              className="text-blue-600 hover:text-blue-700"
                            />
                          )}
                          
                          <Button
                            variant="ghost"
                            size="sm"
                            iconName="Edit"
                            onClick={() => onEdit(contract)}
                            className="text-gray-600 hover:text-gray-700"
                          />
                          
                          <div className="relative group">
                            <Button
                              variant="ghost"
                              size="sm"
                              iconName="Download"
                              className="text-green-600 hover:text-green-700"
                            />
                            <div className="absolute right-0 mt-2 w-32 bg-white border border-gray-200 rounded-md shadow-lg opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all z-10">
                              <button
                                onClick={() => onExport(contract, 'pdf')}
                                className="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-50"
                              >
                                Descargar PDF
                              </button>
                              <button
                                onClick={() => onExport(contract, 'word')}
                                className="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-50"
                              >
                                Descargar Word
                              </button>
                            </div>
                          </div>
                          
                          <Button
                            variant="ghost"
                            size="sm"
                            iconName="Trash2"
                            onClick={() => onDelete(contract?.id)}
                            className="text-red-600 hover:text-red-700"
                          />
                        </div>
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </table>
          </div>
        ) : (
          <div className="text-center py-12">
            <Icon name="FileText" size={48} className="mx-auto text-gray-300 mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No hay contratos</h3>
            <p className="text-gray-500 mb-6">
              {searchTerm || statusFilter
                ? 'No se encontraron contratos con los filtros aplicados'
                : 'Comienza creando tu primer contrato de arriendo'
              }
            </p>
            {!searchTerm && !statusFilter && (
              <Button
                iconName="Plus"
                iconPosition="left"
                onClick={() => console.log('Create new contract')}
              >
                Crear Primer Contrato
              </Button>
            )}
          </div>
        )}
      </div>
      {/* Summary Stats */}
      {contracts?.length > 0 && (
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          {Object.entries(statusOptions?.slice(1))?.map(([index, status]) => {
            const count = contracts?.filter(c => c?.status === status?.value)?.length || 0;
            const statusStyle = statusStyles?.[status?.value];
            
            return (
              <div key={status?.value} className="bg-white border border-gray-200 rounded-lg p-4">
                <div className="flex items-center space-x-3">
                  <div className={`w-10 h-10 rounded-full flex items-center justify-center ${statusStyle?.bg}`}>
                    <Icon name={statusStyle?.icon} size={20} className={statusStyle?.text} />
                  </div>
                  <div>
                    <p className="text-2xl font-bold text-gray-900">{count}</p>
                    <p className="text-sm text-gray-600">{status?.label}</p>
                  </div>
                </div>
              </div>
            );
          })}
        </div>
      )}
    </div>
  );
};

export default ContractList;