import React from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const DigitalSignatureSection = ({ contractId, status, onSignatureInitiated }) => {
  const signatureStatuses = {
    draft: { label: 'Borrador', color: 'gray', icon: 'FileText' },
    pending_signature: { label: 'Pendiente de Firma', color: 'yellow', icon: 'Clock' },
    signed: { label: 'Firmado', color: 'green', icon: 'CheckCircle' },
    expired: { label: 'Expirado', color: 'red', icon: 'XCircle' }
  };

  const currentStatus = signatureStatuses?.[status] || signatureStatuses?.draft;

  const handleInitiateSignature = () => {
    if (contractId) {
      onSignatureInitiated(contractId);
    } else {
      alert('Primero debes guardar el contrato antes de iniciar el proceso de firma');
    }
  };

  return (
    <div className="bg-white border border-gray-200 rounded-lg p-6">
      <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
        <Icon name="PenTool" size={20} className="mr-2 text-blue-600" />
        Firma Digital del Contrato
      </h3>
      <div className="space-y-6">
        {/* Current Status */}
        <div className="flex items-center space-x-4 p-4 bg-gray-50 rounded-lg">
          <div className={`w-12 h-12 rounded-full flex items-center justify-center ${
            currentStatus?.color === 'gray' ? 'bg-gray-100' :
            currentStatus?.color === 'yellow' ? 'bg-yellow-100' :
            currentStatus?.color === 'green' ? 'bg-green-100' : 'bg-red-100'
          }`}>
            <Icon 
              name={currentStatus?.icon} 
              size={24} 
              className={
                currentStatus?.color === 'gray' ? 'text-gray-600' :
                currentStatus?.color === 'yellow' ? 'text-yellow-600' :
                currentStatus?.color === 'green' ? 'text-green-600' : 'text-red-600'
              } 
            />
          </div>
          <div>
            <h4 className="font-medium text-gray-900">Estado Actual</h4>
            <p className={`text-sm font-medium ${
              currentStatus?.color === 'gray' ? 'text-gray-600' :
              currentStatus?.color === 'yellow' ? 'text-yellow-600' :
              currentStatus?.color === 'green' ? 'text-green-600' : 'text-red-600'
            }`}>
              {currentStatus?.label}
            </p>
          </div>
        </div>

        {/* Signature Process */}
        <div>
          <h4 className="font-medium text-gray-900 mb-4">Proceso de Firma Digital</h4>
          
          {status === 'draft' && (
            <div className="space-y-4">
              <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                <div className="flex items-start space-x-3">
                  <Icon name="Info" size={20} className="text-blue-600 mt-0.5" />
                  <div>
                    <h5 className="font-medium text-blue-900">Listo para Enviar</h5>
                    <p className="text-sm text-blue-700 mt-1">
                      El contrato está completo y listo para ser enviado para firma digital. 
                      Una vez enviado, se generará un enlace seguro para que el arrendatario pueda firmar.
                    </p>
                  </div>
                </div>
              </div>

              <div className="flex space-x-3">
                <Button
                  onClick={handleInitiateSignature}
                  iconName="Send"
                  iconPosition="left"
                >
                  Enviar para Firma
                </Button>
                <Button
                  variant="outline"
                  iconName="Eye"
                  iconPosition="left"
                  onClick={() => console.log('Preview before signature')}
                >
                  Vista Previa
                </Button>
              </div>
            </div>
          )}

          {status === 'pending_signature' && (
            <div className="space-y-4">
              <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                <div className="flex items-start space-x-3">
                  <Icon name="Clock" size={20} className="text-yellow-600 mt-0.5" />
                  <div>
                    <h5 className="font-medium text-yellow-900">Esperando Firma</h5>
                    <p className="text-sm text-yellow-700 mt-1">
                      El contrato ha sido enviado al arrendatario para su firma digital. 
                      Se enviará una notificación automática cuando sea firmado.
                    </p>
                  </div>
                </div>
              </div>

              <div className="flex space-x-3">
                <Button
                  variant="outline"
                  iconName="RefreshCw"
                  iconPosition="left"
                  onClick={() => console.log('Check signature status')}
                >
                  Verificar Estado
                </Button>
                <Button
                  variant="outline"
                  iconName="Send"
                  iconPosition="left"
                  onClick={() => console.log('Resend signature request')}
                >
                  Reenviar
                </Button>
              </div>
            </div>
          )}

          {status === 'signed' && (
            <div className="space-y-4">
              <div className="bg-green-50 border border-green-200 rounded-lg p-4">
                <div className="flex items-start space-x-3">
                  <Icon name="CheckCircle" size={20} className="text-green-600 mt-0.5" />
                  <div>
                    <h5 className="font-medium text-green-900">Contrato Firmado</h5>
                    <p className="text-sm text-green-700 mt-1">
                      El contrato ha sido firmado digitalmente por todas las partes. 
                      El documento está legalmente válido y ejecutorio.
                    </p>
                    <p className="text-xs text-green-600 mt-2">
                      Firmado el: {new Date()?.toLocaleDateString('es-CL')} - Certificado digital válido
                    </p>
                  </div>
                </div>
              </div>

              <div className="flex space-x-3">
                <Button
                  iconName="Download"
                  iconPosition="left"
                  onClick={() => console.log('Download signed contract')}
                >
                  Descargar Original
                </Button>
                <Button
                  variant="outline"
                  iconName="Eye"
                  iconPosition="left"
                  onClick={() => console.log('View signature details')}
                >
                  Ver Detalles de Firma
                </Button>
              </div>
            </div>
          )}
        </div>

        {/* Signature Features */}
        <div>
          <h4 className="font-medium text-gray-900 mb-4">Características de la Firma Digital</h4>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="flex items-start space-x-3">
              <Icon name="Shield" size={20} className="text-blue-600 mt-0.5" />
              <div>
                <h5 className="text-sm font-medium text-gray-900">Seguridad Certificada</h5>
                <p className="text-xs text-gray-600 mt-1">
                  Certificados digitales con validez legal en Chile
                </p>
              </div>
            </div>
            
            <div className="flex items-start space-x-3">
              <Icon name="Clock" size={20} className="text-green-600 mt-0.5" />
              <div>
                <h5 className="text-sm font-medium text-gray-900">Marca de Tiempo</h5>
                <p className="text-xs text-gray-600 mt-1">
                  Timestamp confiable para validar la fecha y hora
                </p>
              </div>
            </div>
            
            <div className="flex items-start space-x-3">
              <Icon name="Users" size={20} className="text-purple-600 mt-0.5" />
              <div>
                <h5 className="text-sm font-medium text-gray-900">Multi-Firmante</h5>
                <p className="text-xs text-gray-600 mt-1">
                  Soporte para múltiples firmantes en orden específico
                </p>
              </div>
            </div>
            
            <div className="flex items-start space-x-3">
              <Icon name="Smartphone" size={20} className="text-orange-600 mt-0.5" />
              <div>
                <h5 className="text-sm font-medium text-gray-900">Firma Móvil</h5>
                <p className="text-xs text-gray-600 mt-1">
                  Compatible con dispositivos móviles y tabletas
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* Authentication Methods */}
        <div>
          <h4 className="font-medium text-gray-900 mb-4">Métodos de Autenticación</h4>
          <div className="space-y-3">
            <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-3">
                <Icon name="Mail" size={16} className="text-blue-600" />
                <span className="text-sm text-gray-900">Verificación por Email</span>
              </div>
              <span className="text-xs text-green-600 font-medium">Activo</span>
            </div>
            
            <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-3">
                <Icon name="Smartphone" size={16} className="text-green-600" />
                <span className="text-sm text-gray-900">SMS de Verificación</span>
              </div>
              <span className="text-xs text-green-600 font-medium">Activo</span>
            </div>
            
            <div className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
              <div className="flex items-center space-x-3">
                <Icon name="CreditCard" size={16} className="text-purple-600" />
                <span className="text-sm text-gray-900">Firma Avanzada (Cédula)</span>
              </div>
              <span className="text-xs text-gray-500 font-medium">Opcional</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default DigitalSignatureSection;