import React from 'react';
import Icon from '../../../components/AppIcon';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import { Checkbox } from '../../../components/ui/Checkbox';

const PaymentConfiguration = ({ paymentConfig, onChange }) => {
  const currencyOptions = [
    { value: 'CLP', label: 'Pesos Chilenos (CLP)' },
    { value: 'UF', label: 'Unidad de Fomento (UF)' }
  ];

  const adjustmentFrequencies = [
    { value: 'quarterly', label: 'Trimestral (cada 3 meses)' },
    { value: 'semi_annual', label: 'Semestral (cada 6 meses)' },
    { value: 'annual', label: 'Anual (cada 12 meses)' }
  ];

  // Mock UF to CLP conversion rate
  const UF_TO_CLP_RATE = 37500;

  const handleInputChange = (field, value) => {
    onChange({ ...paymentConfig, [field]: value });
  };

  const convertAmount = (amount, fromCurrency, toCurrency) => {
    if (fromCurrency === toCurrency) return amount;
    
    if (fromCurrency === 'UF' && toCurrency === 'CLP') {
      return amount * UF_TO_CLP_RATE;
    } else if (fromCurrency === 'CLP' && toCurrency === 'UF') {
      return amount / UF_TO_CLP_RATE;
    }
    
    return amount;
  };

  const formatCurrency = (amount, currency) => {
    if (currency === 'UF') {
      return `${amount?.toFixed(2)} UF`;
    }
    return `$${amount?.toLocaleString()} CLP`;
  };

  return (
    <div className="bg-white border border-gray-200 rounded-lg p-6">
      <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
        <Icon name="CreditCard" size={20} className="mr-2 text-green-600" />
        Configuración de Pagos
      </h3>
      <div className="space-y-6">
        {/* Basic Payment Configuration */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Monto del Arriendo *
            </label>
            <Input
              type="number"
              value={paymentConfig?.rentAmount || 0}
              onChange={(e) => handleInputChange('rentAmount', parseFloat(e?.target?.value) || 0)}
              placeholder="Ej: 850000"
              className="w-full"
              min="0"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Tipo de Moneda *
            </label>
            <Select
              options={currencyOptions}
              value={paymentConfig?.currency || 'CLP'}
              onChange={(value) => handleInputChange('currency', value)}
              className="w-full"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Día de Pago
            </label>
            <Input
              type="number"
              value={paymentConfig?.paymentDay || 5}
              onChange={(e) => handleInputChange('paymentDay', parseInt(e?.target?.value) || 5)}
              placeholder="5"
              className="w-full"
              min="1"
              max="31"
            />
            <p className="text-xs text-gray-500 mt-1">Día del mes para el pago</p>
          </div>
        </div>

        {/* Currency Conversion Display */}
        {paymentConfig?.rentAmount > 0 && (
          <div className="bg-blue-50 rounded-lg p-4">
            <h4 className="font-medium text-blue-900 mb-3">Equivalencia de Moneda</h4>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
              <div>
                <span className="text-blue-700">Monto en {paymentConfig?.currency}:</span>
                <p className="text-lg font-semibold text-blue-900">
                  {formatCurrency(paymentConfig?.rentAmount, paymentConfig?.currency)}
                </p>
              </div>
              <div>
                <span className="text-blue-700">
                  Equivalente en {paymentConfig?.currency === 'CLP' ? 'UF' : 'CLP'}:
                </span>
                <p className="text-lg font-semibold text-blue-900">
                  {formatCurrency(
                    convertAmount(
                      paymentConfig?.rentAmount,
                      paymentConfig?.currency,
                      paymentConfig?.currency === 'CLP' ? 'UF' : 'CLP'
                    ),
                    paymentConfig?.currency === 'CLP' ? 'UF' : 'CLP'
                  )}
                </p>
              </div>
            </div>
            <div className="text-xs text-blue-600 mt-2 pt-2 border-t border-blue-200">
              Tasa de conversión: 1 UF = ${UF_TO_CLP_RATE?.toLocaleString()} CLP
            </div>
          </div>
        )}

        {/* IPC Adjustment Configuration */}
        <div className="border border-gray-200 rounded-lg p-4">
          <div className="flex items-start space-x-3 mb-4">
            <Checkbox
              checked={paymentConfig?.ipcAdjustment || false}
              onChange={(checked) => handleInputChange('ipcAdjustment', checked)}
              className="mt-1"
            />
            <div className="flex-1">
              <h4 className="font-medium text-gray-900">Reajuste Automático por IPC</h4>
              <p className="text-sm text-gray-600 mt-1">
                Aplica reajustes automáticos basados en el Índice de Precios al Consumidor
              </p>
            </div>
          </div>

          {paymentConfig?.ipcAdjustment && (
            <div className="ml-7 space-y-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Frecuencia de Reajuste
                  </label>
                  <Select
                    options={adjustmentFrequencies}
                    value={paymentConfig?.adjustmentFrequency || 'annual'}
                    onChange={(value) => handleInputChange('adjustmentFrequency', value)}
                    className="w-full"
                  />
                </div>

                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Multa por Atraso (%)
                  </label>
                  <Input
                    type="number"
                    value={paymentConfig?.lateFeePercentage || 2}
                    onChange={(e) => handleInputChange('lateFeePercentage', parseFloat(e?.target?.value) || 2)}
                    placeholder="2"
                    className="w-full"
                    min="0"
                    max="10"
                    step="0.5"
                  />
                </div>
              </div>

              <div className="bg-green-50 border border-green-200 rounded p-3">
                <div className="flex items-center space-x-2">
                  <Icon name="Info" size={16} className="text-green-600" />
                  <span className="text-sm text-green-800">
                    Los reajustes se aplicarán automáticamente según la frecuencia seleccionada, 
                    basándose en las variaciones del IPC publicadas por el INE.
                  </span>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Common Expenses */}
        <div className="border border-gray-200 rounded-lg p-4">
          <div className="flex items-start space-x-3 mb-4">
            <Checkbox
              checked={paymentConfig?.commonExpenses || false}
              onChange={(checked) => handleInputChange('commonExpenses', checked)}
              className="mt-1"
            />
            <div className="flex-1">
              <h4 className="font-medium text-gray-900">Gastos Comunes Incluidos</h4>
              <p className="text-sm text-gray-600 mt-1">
                Los gastos comunes se incluirán dentro de los pagos de servicios del arrendatario
              </p>
            </div>
          </div>

          {paymentConfig?.commonExpenses && (
            <div className="ml-7">
              <div className="bg-orange-50 border border-orange-200 rounded p-3">
                <div className="flex items-start space-x-2">
                  <Icon name="Building" size={16} className="text-orange-600 mt-0.5" />
                  <div className="text-sm text-orange-800">
                    <p className="font-medium">Gastos Comunes Incluidos:</p>
                    <ul className="mt-1 list-disc list-inside space-y-1">
                      <li>Administración del edificio</li>
                      <li>Mantención de áreas comunes</li>
                      <li>Servicios básicos comunes</li>
                      <li>Seguridad y vigilancia</li>
                    </ul>
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Payment Summary */}
        <div className="bg-gray-50 rounded-lg p-4">
          <h4 className="font-medium text-gray-900 mb-3">Resumen de Configuración de Pagos</h4>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-gray-600">Arriendo Mensual:</span>
                <span className="font-medium">
                  {formatCurrency(paymentConfig?.rentAmount || 0, paymentConfig?.currency || 'CLP')}
                </span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Día de Pago:</span>
                <span className="font-medium">Día {paymentConfig?.paymentDay || 5} de cada mes</span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Reajuste IPC:</span>
                <span className="font-medium">
                  {paymentConfig?.ipcAdjustment ? 'Activo' : 'Inactivo'}
                </span>
              </div>
            </div>
            <div className="space-y-2">
              <div className="flex justify-between">
                <span className="text-gray-600">Gastos Comunes:</span>
                <span className="font-medium">
                  {paymentConfig?.commonExpenses ? 'Incluidos' : 'No incluidos'}
                </span>
              </div>
              <div className="flex justify-between">
                <span className="text-gray-600">Multa por Atraso:</span>
                <span className="font-medium">{paymentConfig?.lateFeePercentage || 2}% mensual</span>
              </div>
              {paymentConfig?.ipcAdjustment && (
                <div className="flex justify-between">
                  <span className="text-gray-600">Frecuencia Reajuste:</span>
                  <span className="font-medium">
                    {adjustmentFrequencies?.find(f => f?.value === paymentConfig?.adjustmentFrequency)?.label || 'Anual'}
                  </span>
                </div>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default PaymentConfiguration;