import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Input from '../../../components/ui/Input';
import Button from '../../../components/ui/Button';

const TemplateCustomization = ({ customization, onChange }) => {
  const [newClause, setNewClause] = useState({ title: '', content: '' });

  const handleInputChange = (field, value) => {
    onChange({ ...customization, [field]: value });
  };

  const handleLogoUpload = (event) => {
    const file = event?.target?.files?.[0];
    if (file) {
      const reader = new FileReader();
      reader.onload = (e) => {
        handleInputChange('logo', e?.target?.result);
      };
      reader?.readAsDataURL(file);
    }
  };

  const addCustomClause = () => {
    if (newClause?.title && newClause?.content) {
      const updatedClauses = [...(customization?.customClauses || []), {
        id: Date.now()?.toString(),
        ...newClause
      }];
      handleInputChange('customClauses', updatedClauses);
      setNewClause({ title: '', content: '' });
    }
  };

  const removeClause = (clauseId) => {
    const updatedClauses = (customization?.customClauses || [])?.filter(
      clause => clause?.id !== clauseId
    );
    handleInputChange('customClauses', updatedClauses);
  };

  return (
    <div className="bg-white border border-gray-200 rounded-lg p-6">
      <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
        <Icon name="Palette" size={20} className="mr-2 text-purple-600" />
        Personalización de Plantilla
      </h3>
      <div className="space-y-6">
        {/* Logo Upload */}
        <div>
          <h4 className="font-medium text-gray-900 mb-3">Logo de la Empresa</h4>
          <div className="flex items-start space-x-6">
            <div className="flex-1">
              <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                <input
                  type="file"
                  accept="image/*"
                  onChange={handleLogoUpload}
                  className="hidden"
                  id="logo-upload"
                />
                <label
                  htmlFor="logo-upload"
                  className="cursor-pointer flex flex-col items-center space-y-2"
                >
                  <Icon name="Upload" size={32} className="text-gray-400" />
                  <span className="text-sm text-gray-600">
                    Haz clic para subir tu logo
                  </span>
                  <span className="text-xs text-gray-500">
                    Formatos soportados: PNG, JPG (máx. 2MB)
                  </span>
                </label>
              </div>
            </div>
            
            {customization?.logo && (
              <div className="w-32 h-32 border border-gray-200 rounded-lg p-2">
                <img
                  src={customization?.logo}
                  alt="Logo preview"
                  className="w-full h-full object-contain"
                />
              </div>
            )}
          </div>
        </div>

        {/* Header Text */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Texto del Encabezado
          </label>
          <Input
            type="text"
            value={customization?.headerText || ''}
            onChange={(e) => handleInputChange('headerText', e?.target?.value)}
            placeholder="Ej: INMOBILIARIA XYZ - CONTRATO DE ARRIENDO"
            className="w-full"
          />
          <p className="text-xs text-gray-500 mt-1">
            Este texto aparecerá en la parte superior del contrato
          </p>
        </div>

        {/* Footer Text */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Pie de Página
          </label>
          <textarea
            value={customization?.footer || ''}
            onChange={(e) => handleInputChange('footer', e?.target?.value)}
            placeholder="Ej: Inmobiliaria XYZ | contacto@inmobiliariaxyz.cl | +56 2 2345 6789"
            className="w-full h-20 p-3 border border-gray-300 rounded-md resize-none focus:ring-2 focus:ring-primary focus:border-primary"
          />
          <p className="text-xs text-gray-500 mt-1">
            Información de contacto y datos de la empresa
          </p>
        </div>

        {/* Custom Clauses */}
        <div>
          <h4 className="font-medium text-gray-900 mb-3">Cláusulas Personalizadas</h4>
          
          {/* Add New Clause */}
          <div className="border border-gray-200 rounded-lg p-4 mb-4">
            <h5 className="text-sm font-medium text-gray-700 mb-3">Agregar Nueva Cláusula</h5>
            <div className="space-y-3">
              <Input
                type="text"
                value={newClause?.title}
                onChange={(e) => setNewClause({...newClause, title: e?.target?.value})}
                placeholder="Título de la cláusula (ej: Uso de la Propiedad)"
                className="w-full"
              />
              <textarea
                value={newClause?.content}
                onChange={(e) => setNewClause({...newClause, content: e?.target?.value})}
                placeholder="Contenido de la cláusula..."
                className="w-full h-24 p-3 border border-gray-300 rounded-md resize-none focus:ring-2 focus:ring-primary focus:border-primary"
              />
              <div className="flex justify-end">
                <Button
                  onClick={addCustomClause}
                  disabled={!newClause?.title || !newClause?.content}
                  iconName="Plus"
                  size="sm"
                >
                  Agregar Cláusula
                </Button>
              </div>
            </div>
          </div>

          {/* Existing Clauses */}
          {customization?.customClauses?.length > 0 ? (
            <div className="space-y-3">
              {customization?.customClauses?.map((clause) => (
                <div key={clause?.id} className="border border-gray-200 rounded-lg p-4">
                  <div className="flex items-start justify-between mb-2">
                    <h6 className="font-medium text-gray-900">{clause?.title}</h6>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => removeClause(clause?.id)}
                      iconName="Trash2"
                      className="text-red-600 hover:text-red-700 hover:bg-red-50"
                    />
                  </div>
                  <p className="text-sm text-gray-600">{clause?.content}</p>
                </div>
              ))}
            </div>
          ) : (
            <div className="text-center py-6 text-gray-500 border border-gray-200 rounded-lg">
              <Icon name="FileText" size={32} className="mx-auto mb-2 text-gray-300" />
              <p>No hay cláusulas personalizadas</p>
              <p className="text-xs mt-1">Usa el formulario de arriba para agregar cláusulas específicas</p>
            </div>
          )}
        </div>

        {/* Format Selection */}
        <div>
          <h4 className="font-medium text-gray-900 mb-3">Formato de Exportación</h4>
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div className="border border-gray-200 rounded-lg p-4 text-center">
              <Icon name="FileText" size={32} className="mx-auto mb-2 text-red-600" />
              <h5 className="font-medium text-gray-900 mb-1">Formato PDF</h5>
              <p className="text-sm text-gray-600 mb-3">
                Ideal para firma digital y envío por email
              </p>
              <Button variant="outline" size="sm" iconName="Download">
                Configurar PDF
              </Button>
            </div>
            
            <div className="border border-gray-200 rounded-lg p-4 text-center">
              <Icon name="File" size={32} className="mx-auto mb-2 text-blue-600" />
              <h5 className="font-medium text-gray-900 mb-1">Formato Word</h5>
              <p className="text-sm text-gray-600 mb-3">
                Permite edición adicional antes de la firma
              </p>
              <Button variant="outline" size="sm" iconName="Download">
                Configurar Word
              </Button>
            </div>
          </div>
        </div>

        {/* Preview */}
        <div className="bg-gray-50 rounded-lg p-4">
          <h4 className="font-medium text-gray-900 mb-3">Vista Previa de Personalización</h4>
          <div className="bg-white border border-gray-200 rounded p-4 min-h-40">
            {/* Header Preview */}
            {customization?.headerText && (
              <div className="text-center mb-4">
                <h1 className="text-lg font-bold text-gray-900">
                  {customization?.headerText}
                </h1>
              </div>
            )}
            
            {/* Logo Preview */}
            {customization?.logo && (
              <div className="text-center mb-4">
                <img
                  src={customization?.logo}
                  alt="Logo"
                  className="h-16 mx-auto"
                />
              </div>
            )}
            
            {/* Sample Content */}
            <div className="text-sm text-gray-600 mb-4">
              <p className="mb-2">CONTRATO DE ARRIENDO</p>
              <p className="mb-2">Entre [ARRENDADOR] y [ARRENDATARIO]...</p>
              <div className="h-8 bg-gray-100 rounded mb-2"></div>
              <div className="h-8 bg-gray-100 rounded mb-2"></div>
            </div>
            
            {/* Custom Clauses Preview */}
            {customization?.customClauses?.length > 0 && (
              <div className="mb-4">
                <h6 className="font-medium text-gray-800 mb-2">Cláusulas Personalizadas:</h6>
                {customization?.customClauses?.map((clause, index) => (
                  <div key={clause?.id} className="text-sm text-gray-600 mb-2">
                    <strong>{index + 1}. {clause?.title}:</strong> {clause?.content?.substring(0, 50)}...
                  </div>
                ))}
              </div>
            )}
            
            {/* Footer Preview */}
            {customization?.footer && (
              <div className="text-center border-t pt-4 text-xs text-gray-500">
                {customization?.footer}
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default TemplateCustomization;