import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import Icon from '../../components/AppIcon';
import Button from '../../components/ui/Button';

import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';

// Import page components
import ContractTemplateSelector from './components/ContractTemplateSelector';
import ContractBuilder from './components/ContractBuilder';
import PaymentConfiguration from './components/PaymentConfiguration';
import TemplateCustomization from './components/TemplateCustomization';
import DigitalSignatureSection from './components/DigitalSignatureSection';

import ContractList from './components/ContractList';

const ContractManagementSystem = () => {
  const navigate = useNavigate();
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [activeTab, setActiveTab] = useState('list');
  const [isMobile, setIsMobile] = useState(false);
  const [selectedContract, setSelectedContract] = useState(null);
  const [isEditing, setIsEditing] = useState(false);

  // Contract form state
  const [contractData, setContractData] = useState({
    id: null,
    templateId: 'standard-rental',
    tenantInfo: {
      name: '',
      rut: '',
      email: '',
      phone: '',
      address: ''
    },
    propertyDetails: {
      address: '',
      type: '',
      area: '',
      rooms: '',
      bathrooms: ''
    },
    rentalTerms: {
      startDate: '',
      endDate: '',
      duration: 12,
      renewalOptions: 'automatic'
    },
    paymentConfig: {
      rentAmount: 0,
      currency: 'CLP',
      ipcAdjustment: false,
      adjustmentFrequency: 'annual',
      commonExpenses: true,
      paymentDay: 5,
      lateFeePercentage: 2
    },
    customization: {
      logo: null,
      footer: '',
      headerText: '',
      customClauses: []
    },
    status: 'draft',
    createdAt: new Date()?.toISOString(),
    modifiedAt: new Date()?.toISOString(),
    versions: []
  });

  // Mock contracts data
  const mockContracts = [
    {
      id: 'contract-001',
      templateId: 'standard-rental',
      tenantInfo: { name: 'María González Pérez', rut: '12.345.678-9', email: 'maria@email.com' },
      propertyDetails: { address: 'Av. Apoquindo 3000, Las Condes', type: 'Apartamento' },
      rentalTerms: { startDate: '2024-09-01', endDate: '2025-09-01', duration: 12 },
      paymentConfig: { rentAmount: 850000, currency: 'CLP', ipcAdjustment: true },
      status: 'active',
      createdAt: '2024-08-15T10:00:00Z',
      signedAt: '2024-08-20T14:30:00Z'
    },
    {
      id: 'contract-002',
      templateId: 'commercial-rental',
      tenantInfo: { name: 'Empresa ABC Ltda.', rut: '98.765.432-1', email: 'contacto@abc.cl' },
      propertyDetails: { address: 'Av. Providencia 1234, Providencia', type: 'Local Comercial' },
      rentalTerms: { startDate: '2024-10-01', endDate: '2026-10-01', duration: 24 },
      paymentConfig: { rentAmount: 2500000, currency: 'UF', ipcAdjustment: false },
      status: 'pending_signature',
      createdAt: '2024-08-20T09:00:00Z'
    }
  ];

  const tabs = [
    { id: 'list', label: 'Lista de Contratos', icon: 'FileText' },
    { id: 'create', label: 'Crear Contrato', icon: 'Plus' },
    { id: 'templates', label: 'Plantillas', icon: 'Layout' },
    { id: 'settings', label: 'Configuración', icon: 'Settings' }
  ];

  useEffect(() => {
    const checkMobile = () => {
      const mobile = window.innerWidth < 1024;
      setIsMobile(mobile);
      if (!mobile) {
        setIsSidebarOpen(false);
      }
    };

    checkMobile();
    window.addEventListener('resize', checkMobile);
    return () => window.removeEventListener('resize', checkMobile);
  }, []);

  const handleMenuToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleContractDataChange = (section, data) => {
    setContractData(prev => ({
      ...prev,
      [section]: { ...prev?.[section], ...data },
      modifiedAt: new Date()?.toISOString()
    }));
  };

  const handleCreateContract = () => {
    setActiveTab('create');
    setSelectedContract(null);
    setIsEditing(false);
    setContractData({
      id: null,
      templateId: 'standard-rental',
      tenantInfo: { name: '', rut: '', email: '', phone: '', address: '' },
      propertyDetails: { address: '', type: '', area: '', rooms: '', bathrooms: '' },
      rentalTerms: { startDate: '', endDate: '', duration: 12, renewalOptions: 'automatic' },
      paymentConfig: { 
        rentAmount: 0, currency: 'CLP', ipcAdjustment: false, 
        adjustmentFrequency: 'annual', commonExpenses: true, 
        paymentDay: 5, lateFeePercentage: 2 
      },
      customization: { logo: null, footer: '', headerText: '', customClauses: [] },
      status: 'draft',
      createdAt: new Date()?.toISOString(),
      modifiedAt: new Date()?.toISOString(),
      versions: []
    });
  };

  const handleEditContract = (contract) => {
    setSelectedContract(contract);
    setContractData(contract);
    setIsEditing(true);
    setActiveTab('create');
  };

  const handleSaveContract = async () => {
    try {
      const contractToSave = {
        ...contractData,
        id: contractData?.id || `contract-${Date.now()}`,
        modifiedAt: new Date()?.toISOString()
      };

      console.log('Saving contract:', contractToSave);
      
      // Simulate API call
      await new Promise(resolve => setTimeout(resolve, 1000));
      
      alert(`Contrato ${isEditing ? 'actualizado' : 'creado'} exitosamente`);
      setActiveTab('list');
      
    } catch (error) {
      console.error('Error saving contract:', error);
      alert('Error al guardar el contrato');
    }
  };

  const handleDeleteContract = async (contractId) => {
    if (confirm('¿Estás seguro de que deseas eliminar este contrato?')) {
      try {
        console.log('Deleting contract:', contractId);
        alert('Contrato eliminado exitosamente');
      } catch (error) {
        console.error('Error deleting contract:', error);
        alert('Error al eliminar el contrato');
      }
    }
  };

  const handleExportContract = async (contract, format) => {
    try {
      console.log(`Exporting contract ${contract?.id} as ${format}`);
      alert(`Generando contrato en formato ${format?.toUpperCase()}...`);
    } catch (error) {
      console.error('Error exporting contract:', error);
      alert('Error al exportar el contrato');
    }
  };

  const handleSignContract = async (contractId) => {
    try {
      console.log('Initiating signature process for:', contractId);
      alert('Iniciando proceso de firma digital...');
    } catch (error) {
      console.error('Error initiating signature:', error);
      alert('Error al iniciar el proceso de firma');
    }
  };

  const renderTabContent = () => {
    switch (activeTab) {
      case 'list':
        return (
          <ContractList
            contracts={mockContracts}
            onEdit={handleEditContract}
            onDelete={handleDeleteContract}
            onExport={handleExportContract}
            onSign={handleSignContract}
          />
        );
      
      case 'create':
        return (
          <div className="space-y-6">
            {/* Contract Builder Header */}
            <div className="flex items-center justify-between">
              <h3 className="text-lg font-semibold text-gray-900">
                {isEditing ? 'Editar Contrato' : 'Crear Nuevo Contrato'}
              </h3>
              <div className="flex space-x-3">
                <Button
                  variant="outline"
                  iconName="Eye"
                  iconPosition="left"
                  onClick={() => console.log('Preview contract')}
                >
                  Vista Previa
                </Button>
                <Button
                  iconName="Save"
                  iconPosition="left"
                  onClick={handleSaveContract}
                >
                  {isEditing ? 'Actualizar' : 'Guardar'} Contrato
                </Button>
              </div>
            </div>

            {/* Contract Building Steps */}
            <ContractTemplateSelector
              selectedTemplate={contractData?.templateId}
              onTemplateSelect={(templateId) => handleContractDataChange('templateId', templateId)}
            />

            <ContractBuilder
              contractData={contractData}
              onDataChange={handleContractDataChange}
            />

            <PaymentConfiguration
              paymentConfig={contractData?.paymentConfig}
              onChange={(config) => handleContractDataChange('paymentConfig', config)}
            />

            <TemplateCustomization
              customization={contractData?.customization}
              onChange={(custom) => handleContractDataChange('customization', custom)}
            />

            <DigitalSignatureSection
              contractId={contractData?.id}
              status={contractData?.status}
              onSignatureInitiated={handleSignContract}
            />
          </div>
        );
      
      case 'templates':
        return (
          <div className="bg-white rounded-lg border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Gestión de Plantillas</h3>
            <p className="text-gray-600 mb-4">
              Administra las plantillas de contratos disponibles en el sistema.
            </p>
            <div className="text-center py-8">
              <Icon name="Layout" size={48} className="text-gray-300 mx-auto mb-4" />
              <p className="text-gray-500">Gestión de plantillas en desarrollo</p>
            </div>
          </div>
        );
      
      case 'settings':
        return (
          <div className="bg-white rounded-lg border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4">Configuración del Sistema</h3>
            <p className="text-gray-600 mb-4">
              Configura las opciones generales del sistema de contratos.
            </p>
            <div className="text-center py-8">
              <Icon name="Settings" size={48} className="text-gray-300 mx-auto mb-4" />
              <p className="text-gray-500">Panel de configuración en desarrollo</p>
            </div>
          </div>
        );
      
      default:
        return null;
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleMenuToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={() => setIsSidebarOpen(false)} />
      <main className={`transition-all duration-300 ${isSidebarOpen && isMobile ? 'lg:ml-60' : 'lg:ml-60'}`}>
        <div className="pt-16">
          <div className="flex">
            {/* Contract Templates Sidebar */}
            <div className="w-80 bg-white border-r border-gray-200 p-6 hidden lg:block">
              <div className="mb-6">
                <h2 className="text-lg font-semibold text-gray-900 mb-2">Sistema de Contratos</h2>
                <p className="text-sm text-gray-600">Gestión completa de contratos de arriendo</p>
              </div>
              
              {/* Quick Actions */}
              <div className="space-y-3 mb-6">
                <Button
                  onClick={handleCreateContract}
                  iconName="Plus"
                  iconPosition="left"
                  className="w-full"
                >
                  Nuevo Contrato
                </Button>
                <Button
                  variant="outline"
                  iconName="Upload"
                  iconPosition="left"
                  className="w-full"
                  onClick={() => console.log('Import contract')}
                >
                  Importar Contrato
                </Button>
              </div>

              {/* Contract Statistics */}
              <div className="bg-gray-50 rounded-lg p-4 mb-6">
                <h3 className="text-sm font-medium text-gray-900 mb-3">Estadísticas</h3>
                <div className="space-y-2">
                  <div className="flex justify-between text-sm">
                    <span className="text-gray-600">Activos:</span>
                    <span className="font-medium">8</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span className="text-gray-600">Pendientes:</span>
                    <span className="font-medium text-yellow-600">3</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span className="text-gray-600">Expirados:</span>
                    <span className="font-medium text-red-600">2</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span className="text-gray-600">Borradores:</span>
                    <span className="font-medium text-gray-500">1</span>
                  </div>
                </div>
              </div>

              {/* Recent Activity */}
              <div>
                <h3 className="text-sm font-medium text-gray-900 mb-3">Actividad Reciente</h3>
                <div className="space-y-3">
                  <div className="flex items-start space-x-3">
                    <div className="w-2 h-2 bg-green-500 rounded-full mt-2"></div>
                    <div className="flex-1 text-sm">
                      <p className="text-gray-900">Contrato firmado</p>
                      <p className="text-gray-500">María González - hace 2 horas</p>
                    </div>
                  </div>
                  <div className="flex items-start space-x-3">
                    <div className="w-2 h-2 bg-blue-500 rounded-full mt-2"></div>
                    <div className="flex-1 text-sm">
                      <p className="text-gray-900">Nuevo borrador creado</p>
                      <p className="text-gray-500">Local Comercial - hace 1 día</p>
                    </div>
                  </div>
                </div>
              </div>
            </div>

            {/* Main Content */}
            <div className="flex-1 p-6">
              {/* Breadcrumbs */}
              <NavigationBreadcrumbs />

              {/* Page Header */}
              <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-8">
                <div className="flex items-center space-x-4 mb-4 lg:mb-0">
                  <div className="w-12 h-12 bg-primary text-white rounded-lg flex items-center justify-center">
                    <Icon name="FileText" size={24} />
                  </div>
                  <div>
                    <h1 className="text-2xl font-bold text-gray-900">Sistema de Gestión de Contratos</h1>
                    <p className="text-gray-600">Crea, modifica y gestiona contratos de arriendo digitalmente</p>
                  </div>
                </div>

                {/* Mobile Menu Toggle */}
                <div className="lg:hidden">
                  <Button
                    variant="outline"
                    size="sm"
                    iconName="Menu"
                    onClick={() => console.log('Toggle mobile menu')}
                  >
                    Menú
                  </Button>
                </div>
              </div>

              {/* Tab Navigation */}
              <div className="border-b border-gray-200 mb-8">
                <nav className="flex space-x-8">
                  {tabs?.map((tab) => (
                    <button
                      key={tab?.id}
                      onClick={() => setActiveTab(tab?.id)}
                      className={`flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm transition-colors ${
                        activeTab === tab?.id
                          ? 'border-primary text-primary' :'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                      }`}
                    >
                      <Icon name={tab?.icon} size={18} />
                      <span>{tab?.label}</span>
                    </button>
                  ))}
                </nav>
              </div>

              {/* Tab Content */}
              <div className="min-h-96">
                {renderTabContent()}
              </div>
            </div>
          </div>
        </div>
      </main>
      <QuickActionFloatingButton />
    </div>
  );
};

export default ContractManagementSystem;