import React, { useState, useEffect } from 'react';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Icon from '../../../components/AppIcon';

const TemplateBuilderModal = ({ isOpen, onClose, template, onSave }) => {
  const [formData, setFormData] = useState({
    name: '',
    category: 'residential',
    description: '',
    content: ''
  });

  useEffect(() => {
    if (template) {
      setFormData({
        name: template?.name || '',
        category: template?.category || 'residential',
        description: template?.description || '',
        content: template?.content || ''
      });
    } else {
      setFormData({
        name: '',
        category: 'residential',
        description: '',
        content: `CONTRATO DE ARRENDAMIENTO DE BIEN RAÍZ

Entre [PROPIETARIO], mayor de edad, estado civil [ESTADO_CIVIL], profesión [PROFESION], cédula de identidad N° [RUT_PROPIETARIO], domiciliado en [DIRECCION_PROPIETARIO], en adelante "EL ARRENDADOR", y [ARRENDATARIO], mayor de edad, estado civil [ESTADO_CIVIL_ARRENDATARIO], profesión [PROFESION_ARRENDATARIO], cédula de identidad N° [RUT_ARRENDATARIO], domiciliado en [DIRECCION_ARRENDATARIO], en adelante "EL ARRENDATARIO", se ha convenido el siguiente contrato de arrendamiento:

PRIMERO: OBJETO DEL CONTRATO
El ARRENDADOR da en arrendamiento al ARRENDATARIO la propiedad ubicada en [DIRECCION_PROPIEDAD], Comuna de [COMUNA], Región [REGION].

SEGUNDO: PRECIO Y FORMA DE PAGO
El precio del arrendamiento es de $[VALOR_ARRIENDO] mensuales, expresado en [MONEDA], pagadero dentro de los primeros [DIAS_PAGO] días de cada mes.

TERCERO: PLAZO
El presente contrato tendrá una duración de [PLAZO_MESES] meses, contados desde el [FECHA_INICIO] hasta el [FECHA_TERMINO].

[CLAUSULAS_ADICIONALES]`
      });
    }
  }, [template, isOpen]);

  if (!isOpen) return null;

  const categoryOptions = [
    { value: 'residential', label: 'Residencial' },
    { value: 'commercial', label: 'Comercial' },
    { value: 'furnished', label: 'Amoblado' },
    { value: 'custom', label: 'Personalizado' }
  ];

  const handleSave = () => {
    const templateData = {
      ...template,
      ...formData,
      id: template?.id || Date.now(),
      lastModified: new Date()?.toISOString()?.split('T')?.[0],
      usageCount: template?.usageCount || 0,
      clauses: (formData?.content?.match(/[A-Z]+:/g) || [])?.length,
      isDefault: template?.isDefault || false
    };
    
    onSave(templateData);
  };

  const insertVariable = (variable) => {
    const textarea = document.getElementById('template-content');
    const start = textarea?.selectionStart || 0;
    const end = textarea?.selectionEnd || 0;
    const newContent = formData?.content?.substring(0, start) + 
                      `[${variable}]` + 
                      formData?.content?.substring(end);
    
    setFormData({ ...formData, content: newContent });
  };

  const commonVariables = [
    'PROPIETARIO',
    'RUT_PROPIETARIO',
    'DIRECCION_PROPIETARIO',
    'ARRENDATARIO',
    'RUT_ARRENDATARIO',
    'DIRECCION_ARRENDATARIO',
    'DIRECCION_PROPIEDAD',
    'COMUNA',
    'REGION',
    'VALOR_ARRIENDO',
    'MONEDA',
    'FECHA_INICIO',
    'FECHA_TERMINO',
    'PLAZO_MESES',
    'DIAS_PAGO',
    'CLAUSULAS_ADICIONALES'
  ];

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-1050 flex items-center justify-center p-4">
      <div className="bg-white rounded-lg shadow-xl max-w-6xl w-full max-h-[90vh] overflow-hidden">
        {/* Modal Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <div>
            <h2 className="text-xl font-semibold text-gray-900">
              {template ? 'Editar Plantilla' : 'Nueva Plantilla'}
            </h2>
            <p className="text-sm text-gray-600 mt-1">
              Crea o modifica plantillas de contratos con variables personalizables
            </p>
          </div>
          
          <Button
            variant="ghost"
            iconName="X"
            onClick={onClose}
            size="sm"
          />
        </div>

        {/* Modal Content */}
        <div className="flex h-[calc(90vh-180px)]">
          {/* Form Sidebar */}
          <div className="w-80 p-6 border-r border-gray-200 bg-gray-50 overflow-y-auto">
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Nombre de la Plantilla
                </label>
                <Input
                  value={formData?.name}
                  onChange={(e) => setFormData({ ...formData, name: e?.target?.value })}
                  placeholder="Ej: Contrato Residencial Básico"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Categoría
                </label>
                <Select
                  value={formData?.category}
                  onChange={(e) => setFormData({ ...formData, category: e?.target?.value })}
                  options={categoryOptions}
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Descripción
                </label>
                <textarea
                  className="w-full p-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary resize-none"
                  rows="3"
                  value={formData?.description}
                  onChange={(e) => setFormData({ ...formData, description: e?.target?.value })}
                  placeholder="Describe el propósito de esta plantilla..."
                />
              </div>
            </div>

            {/* Variables Panel */}
            <div className="mt-6">
              <h4 className="font-medium text-gray-900 mb-3">Variables Disponibles</h4>
              <p className="text-xs text-gray-600 mb-3">
                Haz clic en una variable para insertarla en el cursor
              </p>
              
              <div className="space-y-1">
                {commonVariables?.map((variable) => (
                  <button
                    key={variable}
                    onClick={() => insertVariable(variable)}
                    className="w-full flex items-center space-x-2 p-2 text-left hover:bg-white rounded border-l-2 border-transparent hover:border-primary transition-colors duration-150"
                  >
                    <Icon name="Hash" size={12} className="text-primary" />
                    <span className="text-xs font-mono text-primary">[{variable}]</span>
                  </button>
                ))}
              </div>
            </div>
          </div>

          {/* Content Editor */}
          <div className="flex-1 p-6 overflow-y-auto">
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Contenido del Contrato
            </label>
            <textarea
              id="template-content"
              className="w-full h-full p-4 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-primary font-serif leading-relaxed resize-none"
              value={formData?.content}
              onChange={(e) => setFormData({ ...formData, content: e?.target?.value })}
              placeholder="Escribe el contenido del contrato aquí..."
              style={{ minHeight: '500px' }}
            />
            
            <div className="mt-3 text-xs text-gray-500">
              <p>
                💡 <strong>Tip:</strong> Usa variables entre corchetes (ej: [PROPIETARIO]) 
                para crear campos que se completarán automáticamente al generar contratos.
              </p>
            </div>
          </div>
        </div>

        {/* Modal Footer */}
        <div className="flex items-center justify-between p-6 border-t border-gray-200 bg-gray-50">
          <div className="text-sm text-gray-600">
            Variables detectadas: {(formData?.content?.match(/\[[A-Z_]+\]/g) || [])?.length}
          </div>
          
          <div className="flex items-center space-x-3">
            <Button
              variant="outline"
              onClick={onClose}
            >
              Cancelar
            </Button>
            
            <Button
              variant="default"
              iconName="Save"
              iconPosition="left"
              onClick={handleSave}
              disabled={!formData?.name || !formData?.content}
            >
              {template ? 'Actualizar' : 'Guardar'} Plantilla
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default TemplateBuilderModal;