import React, { useState } from 'react';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';

const TemplateCard = ({ template, onUse, onEdit, onPreview, onDuplicate }) => {
  const [imageError, setImageError] = useState(false);

  const getCategoryColor = (category) => {
    switch (category) {
      case 'residential': return 'bg-blue-100 text-blue-800';
      case 'commercial': return 'bg-purple-100 text-purple-800';
      case 'furnished': return 'bg-green-100 text-green-800';
      case 'custom': return 'bg-orange-100 text-orange-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  };

  const getCategoryLabel = (category) => {
    switch (category) {
      case 'residential': return 'Residencial';
      case 'commercial': return 'Comercial';
      case 'furnished': return 'Amoblado';
      case 'custom': return 'Personalizado';
      default: return 'Sin categoría';
    }
  };

  return (
    <div className="glass-card rounded-lg overflow-hidden hover:shadow-lg transition-shadow duration-150">
      {/* Template Thumbnail */}
      <div className="relative h-32 bg-gray-100 overflow-hidden">
        {!imageError ? (
          <img
            src={template?.thumbnail}
            alt={template?.name}
            className="w-full h-full object-cover"
            onError={() => setImageError(true)}
          />
        ) : (
          <div className="w-full h-full flex items-center justify-center bg-gradient-to-br from-gray-50 to-gray-100">
            <Icon name="FileText" size={32} className="text-gray-400" />
          </div>
        )}
        
        {/* Category Badge */}
        <div className="absolute top-2 left-2">
          <span className={`px-2 py-1 text-xs font-medium rounded-full ${getCategoryColor(template?.category)}`}>
            {getCategoryLabel(template?.category)}
          </span>
        </div>

        {/* Default Badge */}
        {template?.isDefault && (
          <div className="absolute top-2 right-2">
            <span className="px-2 py-1 text-xs font-medium bg-primary text-white rounded-full">
              Por defecto
            </span>
          </div>
        )}
      </div>

      {/* Template Info */}
      <div className="p-4">
        <h3 className="font-semibold text-gray-900 mb-1 line-clamp-1">
          {template?.name}
        </h3>
        
        <p className="text-sm text-gray-600 mb-3 line-clamp-2">
          {template?.description}
        </p>

        {/* Stats */}
        <div className="flex items-center justify-between text-xs text-gray-500 mb-4">
          <div className="flex items-center space-x-3">
            <span className="flex items-center">
              <Icon name="Calendar" size={12} className="mr-1" />
              {template?.lastModified}
            </span>
            <span className="flex items-center">
              <Icon name="FileText" size={12} className="mr-1" />
              {template?.clauses} cláusulas
            </span>
          </div>
          
          <div className="flex items-center">
            <Icon name="TrendingUp" size={12} className="mr-1" />
            <span>{template?.usageCount} usos</span>
          </div>
        </div>

        {/* Action Buttons */}
        <div className="flex flex-col space-y-2">
          <Button
            variant="default"
            size="sm"
            onClick={onUse}
            className="w-full"
          >
            Usar Plantilla
          </Button>
          
          <div className="flex space-x-2">
            <Button
              variant="outline"
              size="sm"
              onClick={onPreview}
              className="flex-1"
              iconName="Eye"
              iconPosition="left"
            >
              Vista Previa
            </Button>
            
            <Button
              variant="ghost"
              size="sm"
              onClick={onEdit}
              className="flex-1"
              iconName="Edit"
              iconPosition="left"
            >
              Editar
            </Button>
            
            <Button
              variant="ghost"
              size="sm"
              onClick={onDuplicate}
              className="px-2"
              iconName="Copy"
            />
          </div>
        </div>
      </div>
    </div>
  );
};

export default TemplateCard;