import React from 'react';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';

const TemplatePreviewModal = ({ isOpen, onClose, template, onUse }) => {
  if (!isOpen || !template) return null;

  // Mock contract content for preview
  const mockContractContent = `
    CONTRATO DE ARRENDAMIENTO DE BIEN RAÍZ

    Entre [PROPIETARIO], mayor de edad, estado civil [ESTADO_CIVIL], profesión [PROFESION], 
    cédula de identidad N° [RUT_PROPIETARIO], domiciliado en [DIRECCION_PROPIETARIO], 
    en adelante "EL ARRENDADOR", y [ARRENDATARIO], mayor de edad, estado civil [ESTADO_CIVIL_ARRENDATARIO], 
    profesión [PROFESION_ARRENDATARIO], cédula de identidad N° [RUT_ARRENDATARIO], 
    domiciliado en [DIRECCION_ARRENDATARIO], en adelante "EL ARRENDATARIO", 
    se ha convenido el siguiente contrato de arrendamiento:

    PRIMERO: OBJETO DEL CONTRATO
    El ARRENDADOR da en arrendamiento al ARRENDATARIO la propiedad ubicada en [DIRECCION_PROPIEDAD], 
    Comuna de [COMUNA], Región [REGION], inscrita a fojas [FOJAS] N° [NUMERO] del Conservador 
    de Bienes Raíces de [CONSERVADOR], correspondiente al año [AÑO_INSCRIPCION].

    SEGUNDO: PRECIO Y FORMA DE PAGO
    El precio del arrendamiento es de $[VALOR_ARRIENDO] mensuales, expresado en [MONEDA], 
    pagadero dentro de los primeros [DIAS_PAGO] días de cada mes, en [FORMA_PAGO].

    TERCERO: PLAZO
    El presente contrato tendrá una duración de [PLAZO_MESES] meses, contados desde el 
    [FECHA_INICIO] hasta el [FECHA_TERMINO].

    [CLAUSULAS_ADICIONALES]
  `;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-1050 flex items-center justify-center p-4">
      <div className="bg-white rounded-lg shadow-xl max-w-4xl w-full max-h-[90vh] overflow-hidden">
        {/* Modal Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <div>
            <h2 className="text-xl font-semibold text-gray-900">
              Vista Previa: {template?.name}
            </h2>
            <p className="text-sm text-gray-600 mt-1">
              {template?.description}
            </p>
          </div>
          
          <div className="flex items-center space-x-2">
            <Button
              variant="default"
              iconName="FileEdit"
              iconPosition="left"
              onClick={onUse}
              size="sm"
            >
              Usar Plantilla
            </Button>
            
            <Button
              variant="ghost"
              iconName="X"
              onClick={onClose}
              size="sm"
            />
          </div>
        </div>

        {/* Modal Content */}
        <div className="flex h-[calc(90vh-120px)]">
          {/* Template Info Sidebar */}
          <div className="w-80 p-6 border-r border-gray-200 bg-gray-50 overflow-y-auto">
            <h3 className="font-semibold text-gray-900 mb-4">Información de la Plantilla</h3>
            
            <div className="space-y-4">
              <div>
                <label className="text-sm font-medium text-gray-600">Categoría</label>
                <p className="text-sm text-gray-900 capitalize">{template?.category}</p>
              </div>
              
              <div>
                <label className="text-sm font-medium text-gray-600">Última Modificación</label>
                <p className="text-sm text-gray-900">{template?.lastModified}</p>
              </div>
              
              <div>
                <label className="text-sm font-medium text-gray-600">Número de Cláusulas</label>
                <p className="text-sm text-gray-900">{template?.clauses}</p>
              </div>
              
              <div>
                <label className="text-sm font-medium text-gray-600">Veces Utilizada</label>
                <p className="text-sm text-gray-900">{template?.usageCount}</p>
              </div>
            </div>

            {/* Variable Placeholders */}
            <div className="mt-6">
              <h4 className="font-medium text-gray-900 mb-3">Variables Disponibles</h4>
              <div className="space-y-2 text-xs">
                {[
                  'PROPIETARIO',
                  'RUT_PROPIETARIO',
                  'ARRENDATARIO',
                  'RUT_ARRENDATARIO',
                  'DIRECCION_PROPIEDAD',
                  'VALOR_ARRIENDO',
                  'FECHA_INICIO',
                  'FECHA_TERMINO'
                ]?.map((variable) => (
                  <div
                    key={variable}
                    className="flex items-center space-x-2 p-2 bg-white rounded border"
                  >
                    <Icon name="Hash" size={12} className="text-primary" />
                    <span className="font-mono text-primary">[{variable}]</span>
                  </div>
                ))}
              </div>
            </div>
          </div>

          {/* Contract Preview */}
          <div className="flex-1 p-6 overflow-y-auto">
            <div className="bg-white border border-gray-200 rounded-lg p-8 font-serif leading-relaxed">
              <div className="text-center mb-8">
                <h1 className="text-2xl font-bold text-gray-900 mb-2">
                  CONTRATO DE ARRENDAMIENTO
                </h1>
                <p className="text-gray-600">Vista previa con datos de ejemplo</p>
              </div>
              
              <div 
                className="prose prose-sm max-w-none"
                style={{ whiteSpace: 'pre-line' }}
              >
                {mockContractContent}
              </div>
              
              {/* Sample Property Images Section */}
              <div className="mt-12 border-t pt-8">
                <h3 className="text-lg font-semibold text-gray-900 mb-4">
                  INVENTARIO DE LA PROPIEDAD
                </h3>
                
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4 mb-6">
                  {[1, 2, 3]?.map((i) => (
                    <div key={i} className="border border-gray-200 rounded-lg p-4 text-center">
                      <div className="w-full h-24 bg-gray-100 rounded mb-2 flex items-center justify-center">
                        <Icon name="Camera" size={24} className="text-gray-400" />
                      </div>
                      <p className="text-xs text-gray-600">Imagen de propiedad {i}</p>
                    </div>
                  ))}
                </div>
                
                <p className="text-sm text-gray-600 italic">
                  * Las imágenes de la propiedad se insertarán aquí durante la creación del contrato
                </p>
              </div>
            </div>
          </div>
        </div>

        {/* Modal Footer */}
        <div className="flex items-center justify-end space-x-3 p-6 border-t border-gray-200 bg-gray-50">
          <Button
            variant="outline"
            onClick={onClose}
          >
            Cerrar
          </Button>
          
          <Button
            variant="default"
            iconName="FileEdit"
            iconPosition="left"
            onClick={onUse}
          >
            Usar esta Plantilla
          </Button>
        </div>
      </div>
    </div>
  );
};

export default TemplatePreviewModal;