import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';
import Button from '../../components/ui/Button';
import Input from '../../components/ui/Input';
import Select from '../../components/ui/Select';
import TemplateCard from './components/TemplateCard';
import TemplatePreviewModal from './components/TemplatePreviewModal';
import TemplateBuilderModal from './components/TemplateBuilderModal';

const ContractTemplatesLibrary = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('all');
  const [previewModal, setPreviewModal] = useState({ isOpen: false, template: null });
  const [builderModal, setBuilderModal] = useState({ isOpen: false, template: null });
  const navigate = useNavigate();

  useEffect(() => {
    const timer = setTimeout(() => {
      setLoading(false);
    }, 1000);
    return () => clearTimeout(timer);
  }, []);

  const handleSidebarToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleSidebarClose = () => {
    setIsSidebarOpen(false);
  };

  // Mock template data
  const templates = [
    {
      id: 1,
      name: 'Contrato Residencial Estándar',
      category: 'residential',
      description: 'Plantilla básica para arriendo de propiedades residenciales',
      thumbnail: '/api/placeholder/300/200',
      lastModified: '2024-08-25',
      usageCount: 45,
      clauses: 12,
      isDefault: true
    },
    {
      id: 2,
      name: 'Contrato Comercial',
      category: 'commercial',
      description: 'Plantilla para locales comerciales y oficinas',
      thumbnail: '/api/placeholder/300/200',
      lastModified: '2024-08-20',
      usageCount: 23,
      clauses: 18,
      isDefault: false
    },
    {
      id: 3,
      name: 'Contrato Amoblado',
      category: 'furnished',
      description: 'Plantilla especializada para propiedades amobladas',
      thumbnail: '/api/placeholder/300/200',
      lastModified: '2024-08-18',
      usageCount: 31,
      clauses: 15,
      isDefault: false
    },
    {
      id: 4,
      name: 'Contrato Personalizado',
      category: 'custom',
      description: 'Plantilla personalizable con cláusulas específicas',
      thumbnail: '/api/placeholder/300/200',
      lastModified: '2024-08-22',
      usageCount: 8,
      clauses: 20,
      isDefault: false
    }
  ];

  const categories = [
    { value: 'all', label: 'Todas las Categorías' },
    { value: 'residential', label: 'Residencial' },
    { value: 'commercial', label: 'Comercial' },
    { value: 'furnished', label: 'Amoblado' },
    { value: 'custom', label: 'Personalizado' }
  ];

  const filteredTemplates = templates?.filter(template => {
    const matchesSearch = template?.name?.toLowerCase()?.includes(searchTerm?.toLowerCase()) ||
                         template?.description?.toLowerCase()?.includes(searchTerm?.toLowerCase());
    const matchesCategory = selectedCategory === 'all' || template?.category === selectedCategory;
    return matchesSearch && matchesCategory;
  });

  const handleUseTemplate = (template) => {
    navigate('/contract-drafting-workspace', { 
      state: { 
        selectedTemplate: template,
        mode: 'create' 
      } 
    });
  };

  const handleEditTemplate = (template) => {
    setBuilderModal({ isOpen: true, template });
  };

  const handlePreviewTemplate = (template) => {
    setPreviewModal({ isOpen: true, template });
  };

  const handleDuplicateTemplate = (template) => {
    const newTemplate = {
      ...template,
      id: templates?.length + 1,
      name: `${template?.name} (Copia)`,
      usageCount: 0,
      lastModified: new Date()?.toISOString()?.split('T')?.[0],
      isDefault: false
    };
    console.log('Duplicating template:', newTemplate);
  };

  const handleCreateNew = () => {
    setBuilderModal({ isOpen: true, template: null });
  };

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleSidebarToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={handleSidebarClose} />
      <main className="lg:ml-60 pt-16">
        <div className="p-6 max-w-7xl mx-auto">
          <NavigationBreadcrumbs />
          
          {/* Page Header */}
          <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-8">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 mb-2">
                Biblioteca de Plantillas
              </h1>
              <p className="text-gray-600">
                Gestiona y personaliza plantillas de contratos de arriendo
              </p>
            </div>
            
            <div className="mt-4 lg:mt-0">
              <Button
                variant="default"
                iconName="Plus"
                iconPosition="left"
                onClick={handleCreateNew}
              >
                Nueva Plantilla
              </Button>
            </div>
          </div>

          {/* Filters and Search */}
          <div className="glass-card p-6 rounded-lg mb-8">
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div className="md:col-span-2">
                <Input
                  placeholder="Buscar plantillas..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e?.target?.value)}
                  iconName="Search"
                  iconPosition="left"
                />
              </div>
              
              <div>
                <Select
                  value={selectedCategory}
                  onChange={(e) => setSelectedCategory(e?.target?.value)}
                  options={categories}
                />
              </div>
            </div>
          </div>

          {/* Templates Grid */}
          {loading ? (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
              {[1, 2, 3, 4]?.map((i) => (
                <div key={i} className="glass-card p-6 rounded-lg animate-pulse">
                  <div className="w-full h-32 bg-gray-200 rounded mb-4"></div>
                  <div className="h-4 bg-gray-200 rounded mb-2"></div>
                  <div className="h-3 bg-gray-200 rounded mb-4"></div>
                  <div className="flex justify-between">
                    <div className="h-3 bg-gray-200 rounded w-16"></div>
                    <div className="h-3 bg-gray-200 rounded w-12"></div>
                  </div>
                </div>
              ))}
            </div>
          ) : (
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
              {filteredTemplates?.map((template) => (
                <TemplateCard
                  key={template?.id}
                  template={template}
                  onUse={() => handleUseTemplate(template)}
                  onEdit={() => handleEditTemplate(template)}
                  onPreview={() => handlePreviewTemplate(template)}
                  onDuplicate={() => handleDuplicateTemplate(template)}
                />
              ))}
            </div>
          )}

          {filteredTemplates?.length === 0 && !loading && (
            <div className="text-center py-12">
              <div className="text-gray-400 mb-4">
                <svg className="w-16 h-16 mx-auto" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
              </div>
              <p className="text-gray-500">No se encontraron plantillas con los criterios seleccionados</p>
              <Button
                variant="outline"
                iconName="Plus"
                iconPosition="left"
                onClick={handleCreateNew}
                className="mt-4"
              >
                Crear Nueva Plantilla
              </Button>
            </div>
          )}

          {/* Usage Statistics */}
          <div className="mt-8 grid grid-cols-1 md:grid-cols-3 gap-6">
            <div className="glass-card p-6 rounded-lg text-center">
              <div className="text-2xl font-bold text-primary mb-2">
                {templates?.length}
              </div>
              <p className="text-gray-600">Plantillas Disponibles</p>
            </div>
            
            <div className="glass-card p-6 rounded-lg text-center">
              <div className="text-2xl font-bold text-success mb-2">
                {templates?.reduce((sum, t) => sum + t?.usageCount, 0)}
              </div>
              <p className="text-gray-600">Contratos Generados</p>
            </div>
            
            <div className="glass-card p-6 rounded-lg text-center">
              <div className="text-2xl font-bold text-accent mb-2">
                {templates?.filter(t => t?.isDefault)?.length}
              </div>
              <p className="text-gray-600">Plantillas Predeterminadas</p>
            </div>
          </div>
        </div>
      </main>
      {/* Modals */}
      <TemplatePreviewModal
        isOpen={previewModal?.isOpen}
        onClose={() => setPreviewModal({ isOpen: false, template: null })}
        template={previewModal?.template}
        onUse={() => {
          setPreviewModal({ isOpen: false, template: null });
          handleUseTemplate(previewModal?.template);
        }}
      />
      <TemplateBuilderModal
        isOpen={builderModal?.isOpen}
        onClose={() => setBuilderModal({ isOpen: false, template: null })}
        template={builderModal?.template}
        onSave={(template) => {
          console.log('Saving template:', template);
          setBuilderModal({ isOpen: false, template: null });
        }}
      />
      <QuickActionFloatingButton />
    </div>
  );
};

export default ContractTemplatesLibrary;