import React from 'react';
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer } from 'recharts';

const IncomeChart = ({ loading = false }) => {
  const incomeData = [
    { month: 'Mar', income: 2850000, uf: 95.2 },
    { month: 'Abr', income: 3200000, uf: 106.8 },
    { month: 'May', income: 2950000, uf: 98.5 },
    { month: 'Jun', income: 3400000, uf: 113.6 },
    { month: 'Jul', income: 3100000, uf: 103.5 },
    { month: 'Ago', income: 3650000, uf: 121.9 }
  ];

  const formatCurrency = (value) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(value);
  };

  const CustomTooltip = ({ active, payload, label }) => {
    if (active && payload && payload?.length) {
      return (
        <div className="bg-white p-3 rounded-lg shadow-lg border border-gray-200">
          <p className="text-sm font-medium text-gray-900">{`${label} 2024`}</p>
          <p className="text-sm text-primary">
            {`Ingresos: ${formatCurrency(payload?.[0]?.value)}`}
          </p>
          <p className="text-xs text-gray-500">
            {`≈ ${payload?.[0]?.payload?.uf} UF`}
          </p>
        </div>
      );
    }
    return null;
  };

  if (loading) {
    return (
      <div className="glass-card p-6 rounded-lg">
        <div className="skeleton w-48 h-6 mb-4"></div>
        <div className="skeleton w-full h-64"></div>
      </div>
    );
  }

  return (
    <div className="glass-card p-6 rounded-lg">
      <div className="flex items-center justify-between mb-6">
        <h3 className="text-lg font-semibold text-gray-900">Ingresos Últimos 6 Meses</h3>
        <div className="flex items-center space-x-2 text-sm text-gray-500">
          <div className="w-3 h-3 bg-primary rounded-full"></div>
          <span>Ingresos Mensuales</span>
        </div>
      </div>
      <div className="w-full h-64">
        <ResponsiveContainer width="100%" height="100%">
          <LineChart data={incomeData} margin={{ top: 5, right: 30, left: 20, bottom: 5 }}>
            <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
            <XAxis 
              dataKey="month" 
              axisLine={false}
              tickLine={false}
              tick={{ fontSize: 12, fill: '#6b7280' }}
            />
            <YAxis 
              axisLine={false}
              tickLine={false}
              tick={{ fontSize: 12, fill: '#6b7280' }}
              tickFormatter={(value) => `$${(value / 1000000)?.toFixed(1)}M`}
            />
            <Tooltip content={<CustomTooltip />} />
            <Line 
              type="monotone" 
              dataKey="income" 
              stroke="var(--color-primary)"
              strokeWidth={3}
              dot={{ fill: 'var(--color-primary)', strokeWidth: 2, r: 4 }}
              activeDot={{ r: 6, stroke: 'var(--color-primary)', strokeWidth: 2 }}
            />
          </LineChart>
        </ResponsiveContainer>
      </div>
    </div>
  );
};

export default IncomeChart;