import React from 'react';
import Icon from '../../../components/AppIcon';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const UpcomingEvents = ({ loading = false }) => {
  const upcomingEvents = [
    {
      id: 1,
      type: 'payment',
      title: 'Vencimiento Arriendo - Depto 301',
      description: 'Las Condes, Santiago',
      date: '2024-08-28',
      priority: 'high',
      amount: '$850.000',
      tenant: 'Carlos Mendoza'
    },
    {
      id: 2,
      type: 'maintenance',
      title: 'Inspección Técnica Programada',
      description: 'Casa Providencia - Revisión anual',
      date: '2024-08-30',
      priority: 'medium',
      owner: 'Ana Rodríguez'
    },
    {
      id: 3,
      type: 'contract',
      title: 'Renovación Contrato',
      description: 'Oficina Las Condes - Vence en 15 días',
      date: '2024-09-10',
      priority: 'medium',
      tenant: 'Empresa TechSol'
    },
    {
      id: 4,
      type: 'reminder',
      title: 'Recordatorio Pago Pendiente',
      description: 'Depto 205 - 3 días de atraso',
      date: '2024-08-27',
      priority: 'high',
      amount: '$650.000'
    }
  ];

  const getEventIcon = (type) => {
    const iconMap = {
      payment: 'DollarSign',
      maintenance: 'Wrench',
      contract: 'FileText',
      reminder: 'Bell'
    };
    return iconMap?.[type] || 'Calendar';
  };

  const getEventColor = (type) => {
    const colorMap = {
      payment: 'text-success',
      maintenance: 'text-warning',
      contract: 'text-primary',
      reminder: 'text-error'
    };
    return colorMap?.[type] || 'text-gray-500';
  };

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    const today = new Date();
    const diffTime = date?.getTime() - today?.getTime();
    const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
    
    if (diffDays === 0) return 'Hoy';
    if (diffDays === 1) return 'Mañana';
    if (diffDays < 0) return `${Math.abs(diffDays)} días atrás`;
    return `En ${diffDays} días`;
  };

  if (loading) {
    return (
      <div className="glass-card p-6 rounded-lg">
        <div className="skeleton w-48 h-6 mb-4"></div>
        <div className="space-y-4">
          {[1, 2, 3]?.map((i) => (
            <div key={i} className="flex items-center space-x-3">
              <div className="skeleton w-10 h-10 rounded-full"></div>
              <div className="flex-1 space-y-2">
                <div className="skeleton w-3/4 h-4"></div>
                <div className="skeleton w-1/2 h-3"></div>
              </div>
            </div>
          ))}
        </div>
      </div>
    );
  }

  return (
    <div className="glass-card p-6 rounded-lg">
      <div className="flex items-center justify-between mb-6">
        <h3 className="text-lg font-semibold text-gray-900">Eventos Próximos</h3>
        <StatusIndicatorBadges status="urgent" size="sm">
          {upcomingEvents?.filter(e => e?.priority === 'high')?.length} Urgentes
        </StatusIndicatorBadges>
      </div>
      <div className="space-y-4 max-h-80 overflow-y-auto">
        {upcomingEvents?.map((event) => (
          <div key={event?.id} className="flex items-start space-x-3 p-3 rounded-lg hover:bg-gray-50 transition-colors duration-150">
            <div className={`w-10 h-10 rounded-full bg-gray-100 flex items-center justify-center ${getEventColor(event?.type)}`}>
              <Icon name={getEventIcon(event?.type)} size={18} />
            </div>
            
            <div className="flex-1 min-w-0">
              <div className="flex items-start justify-between">
                <div className="flex-1">
                  <h4 className="text-sm font-medium text-gray-900 truncate">
                    {event?.title}
                  </h4>
                  <p className="text-xs text-gray-500 mt-1">
                    {event?.description}
                  </p>
                  {event?.tenant && (
                    <p className="text-xs text-gray-600 mt-1">
                      <Icon name="User" size={12} className="inline mr-1" />
                      {event?.tenant}
                    </p>
                  )}
                  {event?.owner && (
                    <p className="text-xs text-gray-600 mt-1">
                      <Icon name="UserCheck" size={12} className="inline mr-1" />
                      {event?.owner}
                    </p>
                  )}
                </div>
                
                <div className="text-right ml-3">
                  <StatusIndicatorBadges status={event?.priority} size="sm" />
                  <p className="text-xs text-gray-500 mt-1">
                    {formatDate(event?.date)}
                  </p>
                  {event?.amount && (
                    <p className="text-xs font-medium text-success mt-1">
                      {event?.amount}
                    </p>
                  )}
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>
      <div className="mt-4 pt-4 border-t border-gray-100">
        <button className="w-full text-sm text-primary hover:text-primary/80 font-medium transition-colors duration-150">
          Ver todos los eventos →
        </button>
      </div>
    </div>
  );
};

export default UpcomingEvents;