import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';
import MetricCard from './components/MetricCard';
import IncomeChart from './components/IncomeChart';
import VoucherDistributionChart from './components/VoucherDistributionChart';
import QuickActionButtons from './components/QuickActionButtons';
import UpcomingEvents from './components/UpcomingEvents';

const Dashboard = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [loading, setLoading] = useState(true);
  const navigate = useNavigate();

  useEffect(() => {
    // Simulate data loading
    const timer = setTimeout(() => {
      setLoading(false);
    }, 1500);

    return () => clearTimeout(timer);
  }, []);

  const handleSidebarToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleSidebarClose = () => {
    setIsSidebarOpen(false);
  };

  // Mock metrics data
  const metricsData = {
    currentIncome: {
      value: '$3.650.000',
      change: '+12.5%',
      changeType: 'positive',
      subtitle: '≈ 121.9 UF'
    },
    voucherStatus: {
      paid: 45,
      pending: 28,
      overdue: 12,
      sent: 15
    },
    properties: {
      active: 28,
      inactive: 4,
      total: 32
    },
    upcomingEvents: 8
  };

  const handleMetricClick = (metric) => {
    switch (metric) {
      case 'income': navigate('/reports-analytics');
        break;
      case 'vouchers': navigate('/vouchers-management');
        break;
      case 'properties': navigate('/properties-management');
        break;
      case 'events': navigate('/reminders-system');
        break;
      default:
        break;
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleSidebarToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={handleSidebarClose} />
      <main className="lg:ml-60 pt-16">
        <div className="p-6 max-w-7xl mx-auto">
          {/* Breadcrumbs */}
          <NavigationBreadcrumbs />
          
          {/* Page Header */}
          <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-8">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 mb-2">
                Panel Principal
              </h1>
              <p className="text-gray-600">
                Resumen ejecutivo de tu cartera inmobiliaria - Agosto 2024
              </p>
            </div>
            
            <div className="mt-4 lg:mt-0">
              <QuickActionButtons />
            </div>
          </div>

          {/* Metrics Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-6 gap-6 mb-8">
            <div className="xl:col-span-2">
              <MetricCard
                title="Ingresos del Mes"
                value={metricsData?.currentIncome?.value}
                change={metricsData?.currentIncome?.change}
                changeType={metricsData?.currentIncome?.changeType}
                subtitle={metricsData?.currentIncome?.subtitle}
                icon="DollarSign"
                iconColor="text-success"
                onClick={() => handleMetricClick('income')}
                loading={loading}
              />
            </div>
            
            <MetricCard
              title="Comprobantes Pagados"
              value={metricsData?.voucherStatus?.paid?.toString()}
              change="+8.3%"
              changeType="positive"
              icon="CheckCircle"
              iconColor="text-success"
              onClick={() => handleMetricClick('vouchers')}
              loading={loading}
            />
            
            <MetricCard
              title="Pendientes de Pago"
              value={metricsData?.voucherStatus?.pending?.toString()}
              change="-5.2%"
              changeType="positive"
              icon="Clock"
              iconColor="text-warning"
              onClick={() => handleMetricClick('vouchers')}
              loading={loading}
            />
            
            <MetricCard
              title="Propiedades Activas"
              value={metricsData?.properties?.active?.toString()}
              subtitle={`de ${metricsData?.properties?.total} totales`}
              icon="Building2"
              iconColor="text-primary"
              onClick={() => handleMetricClick('properties')}
              loading={loading}
              change=""
              changeType="neutral"
            />
            
            <MetricCard
              title="Eventos Próximos"
              value={metricsData?.upcomingEvents?.toString()}
              change="3 urgentes"
              changeType="negative"
              icon="Calendar"
              iconColor="text-error"
              onClick={() => handleMetricClick('events')}
              loading={loading}
            />
          </div>

          {/* Charts and Events Section */}
          <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-6 mb-8">
            {/* Income Chart */}
            <div className="xl:col-span-1">
              <IncomeChart loading={loading} />
            </div>
            
            {/* Voucher Distribution Chart */}
            <div className="xl:col-span-1">
              <VoucherDistributionChart loading={loading} />
            </div>
            
            {/* Upcoming Events */}
            <div className="xl:col-span-1">
              <UpcomingEvents loading={loading} />
            </div>
          </div>

          {/* Additional Insights */}
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
            {/* Performance Summary */}
            <div className="glass-card p-6 rounded-lg">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">
                Resumen de Rendimiento
              </h3>
              <div className="space-y-4">
                <div className="flex items-center justify-between">
                  <span className="text-sm text-gray-600">Tasa de Ocupación</span>
                  <div className="flex items-center space-x-2">
                    <div className="w-24 bg-gray-200 rounded-full h-2">
                      <div className="bg-success h-2 rounded-full" style={{ width: '87.5%' }}></div>
                    </div>
                    <span className="text-sm font-medium text-gray-900">87.5%</span>
                  </div>
                </div>
                
                <div className="flex items-center justify-between">
                  <span className="text-sm text-gray-600">Pagos a Tiempo</span>
                  <div className="flex items-center space-x-2">
                    <div className="w-24 bg-gray-200 rounded-full h-2">
                      <div className="bg-primary h-2 rounded-full" style={{ width: '92%' }}></div>
                    </div>
                    <span className="text-sm font-medium text-gray-900">92%</span>
                  </div>
                </div>
                
                <div className="flex items-center justify-between">
                  <span className="text-sm text-gray-600">Satisfacción Inquilinos</span>
                  <div className="flex items-center space-x-2">
                    <div className="w-24 bg-gray-200 rounded-full h-2">
                      <div className="bg-accent h-2 rounded-full" style={{ width: '95%' }}></div>
                    </div>
                    <span className="text-sm font-medium text-gray-900">4.8/5</span>
                  </div>
                </div>
              </div>
            </div>

            {/* Recent Activity */}
            <div className="glass-card p-6 rounded-lg">
              <h3 className="text-lg font-semibold text-gray-900 mb-4">
                Actividad Reciente
              </h3>
              <div className="space-y-3">
                <div className="flex items-center space-x-3 text-sm">
                  <div className="w-2 h-2 bg-success rounded-full"></div>
                  <span className="text-gray-600">Pago recibido - Depto 301</span>
                  <span className="text-gray-400 ml-auto">Hace 2h</span>
                </div>
                
                <div className="flex items-center space-x-3 text-sm">
                  <div className="w-2 h-2 bg-primary rounded-full"></div>
                  <span className="text-gray-600">Nuevo inquilino registrado</span>
                  <span className="text-gray-400 ml-auto">Hace 4h</span>
                </div>
                
                <div className="flex items-center space-x-3 text-sm">
                  <div className="w-2 h-2 bg-warning rounded-full"></div>
                  <span className="text-gray-600">Recordatorio enviado - Casa Providencia</span>
                  <span className="text-gray-400 ml-auto">Hace 6h</span>
                </div>
                
                <div className="flex items-center space-x-3 text-sm">
                  <div className="w-2 h-2 bg-secondary rounded-full"></div>
                  <span className="text-gray-600">Contrato renovado - Oficina Las Condes</span>
                  <span className="text-gray-400 ml-auto">Ayer</span>
                </div>
              </div>
              
              <button className="w-full mt-4 text-sm text-primary hover:text-primary/80 font-medium transition-colors duration-150">
                Ver toda la actividad →
              </button>
            </div>
          </div>
        </div>
      </main>
      <QuickActionFloatingButton />
    </div>
  );
};

export default Dashboard;