import React, { useState } from 'react';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';

const ContractTemplateSelector = ({ isOpen, onClose, onSelect }) => {
  const [selectedTemplate, setSelectedTemplate] = useState(null);

  const contractTemplates = [
    {
      id: 1,
      name: 'Arriendo Residencial',
      description: 'Contrato estándar para propiedades habitacionales',
      type: 'residential',
      icon: 'Home',
      features: [
        'Cláusulas de uso habitacional',
        'Inventario de bienes',
        'Garantías y depósitos',
        'Normas de convivencia'
      ],
      recommended: true
    },
    {
      id: 2,
      name: 'Arriendo Comercial',
      description: 'Contrato para locales comerciales y oficinas',
      type: 'commercial',
      icon: 'Building2',
      features: [
        'Cláusulas comerciales',
        'Patente comercial',
        'Modificaciones permitidas',
        'Horarios de funcionamiento'
      ]
    },
    {
      id: 3,
      name: 'Arriendo Temporal',
      description: 'Para arriendos de corta duración (vacaciones, temporadas)',
      type: 'temporary',
      icon: 'Calendar',
      features: [
        'Estadías cortas',
        'Servicios incluidos',
        'Check-in/Check-out',
        'Política de cancelación'
      ]
    },
    {
      id: 4,
      name: 'Subarriendo',
      description: 'Contrato para subarriendo de propiedades',
      type: 'sublease',
      icon: 'Users',
      features: [
        'Autorización del propietario',
        'Responsabilidades compartidas',
        'Cláusulas especiales',
        'Garantías adicionales'
      ]
    },
    {
      id: 5,
      name: 'Arriendo Estudiantil',
      description: 'Especializado para estudiantes y residencias',
      type: 'student',
      icon: 'GraduationCap',
      features: [
        'Normas estudiantiles',
        'Periodos académicos',
        'Espacios comunes',
        'Garantía parental'
      ]
    },
    {
      id: 6,
      name: 'Arriendo Amoblado',
      description: 'Para propiedades completamente amobladas',
      type: 'furnished',
      icon: 'Sofa',
      features: [
        'Inventario detallado',
        'Estado de los muebles',
        'Seguro de contenidos',
        'Reposición de elementos'
      ]
    }
  ];

  if (!isOpen) return null;

  const handleSelectTemplate = (template) => {
    setSelectedTemplate(template);
  };

  const handleConfirmSelection = () => {
    if (selectedTemplate) {
      onSelect?.(selectedTemplate);
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-[1000] flex items-center justify-center p-4">
      <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto">
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-xl font-bold text-gray-900">Seleccionar Plantilla de Contrato</h2>
              <p className="text-gray-600 mt-1">Elige el tipo de contrato que mejor se adapte a tus necesidades</p>
            </div>
            <Button variant="ghost" size="sm" onClick={onClose}>
              <Icon name="X" size={20} />
            </Button>
          </div>
        </div>

        <div className="p-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            {contractTemplates?.map((template) => (
              <div
                key={template?.id}
                onClick={() => handleSelectTemplate(template)}
                className={`
                  relative border-2 rounded-lg p-6 cursor-pointer transition-all duration-200 hover:shadow-md
                  ${selectedTemplate?.id === template?.id
                    ? 'border-primary bg-primary/5 shadow-md'
                    : 'border-gray-200 hover:border-gray-300'
                  }
                `}
              >
                {/* Recommended Badge */}
                {template?.recommended && (
                  <div className="absolute -top-2 -right-2">
                    <span className="bg-primary text-white text-xs font-medium px-2 py-1 rounded-full">
                      Recomendado
                    </span>
                  </div>
                )}

                <div className="flex items-start space-x-4">
                  <div className={`
                    w-12 h-12 rounded-lg flex items-center justify-center
                    ${selectedTemplate?.id === template?.id
                      ? 'bg-primary text-white' :'bg-gray-100 text-gray-600'
                    }
                  `}>
                    <Icon name={template?.icon} size={24} />
                  </div>

                  <div className="flex-1">
                    <h3 className={`
                      text-lg font-semibold mb-2
                      ${selectedTemplate?.id === template?.id ? 'text-primary' : 'text-gray-900'}
                    `}>
                      {template?.name}
                    </h3>
                    <p className="text-gray-600 text-sm mb-4">
                      {template?.description}
                    </p>

                    <div className="space-y-1">
                      <p className="text-xs font-medium text-gray-700 mb-2">CARACTERÍSTICAS:</p>
                      {template?.features?.map((feature, index) => (
                        <div key={index} className="flex items-center space-x-2">
                          <Icon 
                            name="Check" 
                            size={12} 
                            className={selectedTemplate?.id === template?.id ? 'text-primary' : 'text-gray-400'} 
                          />
                          <span className="text-xs text-gray-600">{feature}</span>
                        </div>
                      ))}
                    </div>
                  </div>
                </div>

                {/* Selection Indicator */}
                {selectedTemplate?.id === template?.id && (
                  <div className="absolute top-3 right-3">
                    <div className="w-6 h-6 bg-primary rounded-full flex items-center justify-center">
                      <Icon name="Check" size={14} className="text-white" />
                    </div>
                  </div>
                )}
              </div>
            ))}
          </div>

          {/* Custom Template Option */}
          <div className="mt-6 p-4 border border-dashed border-gray-300 rounded-lg">
            <div className="text-center">
              <Icon name="FileText" size={24} className="mx-auto text-gray-400 mb-2" />
              <h3 className="text-sm font-medium text-gray-900 mb-1">¿Necesitas una plantilla personalizada?</h3>
              <p className="text-xs text-gray-600 mb-3">
                Puedes crear una plantilla desde cero o modificar una existente
              </p>
              <Button variant="outline" size="sm">
                <Icon name="Plus" size={14} className="mr-2" />
                Crear Plantilla Personalizada
              </Button>
            </div>
          </div>
        </div>

        <div className="p-6 border-t border-gray-200 flex items-center justify-between">
          <div className="text-sm text-gray-600">
            {selectedTemplate ? (
              <span>Plantilla seleccionada: <strong>{selectedTemplate?.name}</strong></span>
            ) : (
              <span>Selecciona una plantilla para continuar</span>
            )}
          </div>
          
          <div className="flex items-center space-x-3">
            <Button variant="outline" onClick={onClose}>
              Cancelar
            </Button>
            <Button 
              onClick={handleConfirmSelection}
              disabled={!selectedTemplate}
            >
              <Icon name="ArrowRight" size={16} className="mr-2" />
              Continuar
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ContractTemplateSelector;