import React, { useState } from 'react';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';
import { saveAs } from 'file-saver';
import jsPDF from 'jspdf';

const DraftPreviewModal = ({ isOpen, onClose, draft, onEdit }) => {
  const [activeTab, setActiveTab] = useState('preview');

  if (!isOpen || !draft) return null;

  const generatePDF = () => {
    const doc = new jsPDF();
    
    // Add title
    doc?.setFontSize(20);
    doc?.text('CONTRATO DE ARRIENDO', 20, 30);
    
    // Add basic info
    doc?.setFontSize(12);
    doc?.text(`Nombre: ${draft?.name}`, 20, 50);
    doc?.text(`Tipo: ${draft?.template}`, 20, 60);
    doc?.text(`Inquilino: ${draft?.tenant}`, 20, 70);
    doc?.text(`Propietario: ${draft?.owner}`, 20, 80);
    doc?.text(`Propiedad: ${draft?.property}`, 20, 90);
    
    // Add contract content
    doc?.setFontSize(10);
    const content = draft?.extractedContent || `
Este es un contrato de arriendo entre ${draft?.owner} (Propietario) y ${draft?.tenant} (Inquilino)
para la propiedad ubicada en ${draft?.property}.

CLAUSULAS PRINCIPALES:

1. OBJETO DEL CONTRATO
Se arrienda la propiedad descrita para uso habitacional.

2. PLAZO
El presente contrato tendrá una duración de 12 meses.

3. RENTA
El valor mensual del arriendo será determinado según las condiciones del mercado.

4. GARANTÍAS
El inquilino deberá constituir garantías según lo establecido por ley.

5. OBLIGACIONES DE LAS PARTES
Ambas partes se comprometen a cumplir con las obligaciones establecidas.
    `;
    
    const lines = doc?.splitTextToSize(content, 170);
    doc?.text(lines, 20, 110);
    
    // Save the PDF
    doc?.save(`${draft?.name}.pdf`);
  };

  const generateWord = () => {
    const content = `
CONTRATO DE ARRIENDO

Nombre: ${draft?.name}
Tipo: ${draft?.template}
Inquilino: ${draft?.tenant}
Propietario: ${draft?.owner}
Propiedad: ${draft?.property}

${draft?.extractedContent || 'Contenido del contrato...'}
    `;

    const blob = new Blob([content], { type: 'application/msword' });
    saveAs(blob, `${draft?.name}.doc`);
  };

  const getStatusBadge = (status) => {
    const statusConfig = {
      'in-progress': {
        label: 'En Progreso',
        className: 'bg-warning/10 text-warning border-warning/20'
      },
      'review': {
        label: 'En Revisión',
        className: 'bg-accent/10 text-accent border-accent/20'
      },
      'completed': {
        label: 'Completado',
        className: 'bg-success/10 text-success border-success/20'
      }
    };

    const config = statusConfig?.[status] || statusConfig?.['in-progress'];

    return (
      <span className={`px-3 py-1 rounded-full text-sm font-medium border ${config?.className}`}>
        {config?.label}
      </span>
    );
  };

  const mockContractContent = `
CONTRATO DE ARRIENDO

Entre ${draft?.owner} (en adelante "EL ARRENDADOR") y ${draft?.tenant} (en adelante "EL ARRENDATARIO"), se conviene en el siguiente contrato de arriendo:

PRIMERA: OBJETO DEL CONTRATO
Se arrienda la propiedad ubicada en ${draft?.property}, en adelante "LA PROPIEDAD", para uso habitacional exclusivamente.

SEGUNDA: PLAZO
El presente contrato tendrá una duración de 12 (doce) meses, contados desde la fecha de entrega de la propiedad.

TERCERA: RENTA
El valor mensual del arriendo será de $XXX.XXX, pagadero dentro de los primeros 5 días de cada mes.

CUARTA: GARANTÍA
El arrendatario constituye como garantía del cumplimiento de las obligaciones emanadas del presente contrato, un depósito equivalente a un mes de arriendo.

QUINTA: INVENTARIO DE LA PROPIEDAD
La propiedad se entrega amoblada/semi-amoblada con los siguientes elementos:

INVENTARIO:
- Living: Sofá, mesa de centro, televisor
- Dormitorio: Cama 2 plazas, veladores, closet
- Cocina: Refrigerador, microondas, menaje básico
- Baño: Cortina de ducha, espejo

IMAGENES DE LA PROPIEDAD:
[Aquí se insertarían las imágenes de la propiedad]

SEXTA: OBLIGACIONES DEL ARRENDATARIO
- Pagar la renta puntualmente
- Mantener la propiedad en buen estado
- No subarrendar sin autorización
- Respetar las normas de convivencia

SÉPTIMA: OBLIGACIONES DEL ARRENDADOR
- Entregar la propiedad en condiciones habitables
- Realizar reparaciones mayores
- Respetar el derecho de uso pacífico

Firmado en Santiago, a ${new Date()?.toLocaleDateString('es-CL')}

_________________                    _________________
EL ARRENDADOR                       EL ARRENDATARIO
${draft?.owner}                       ${draft?.tenant}
  `;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-[1000] flex items-center justify-center p-4">
      <div className="bg-white rounded-lg max-w-6xl w-full max-h-[90vh] overflow-hidden flex flex-col">
        {/* Header */}
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-xl font-bold text-gray-900">{draft?.name}</h2>
              <div className="flex items-center space-x-3 mt-2">
                {getStatusBadge(draft?.status)}
                <span className="text-sm text-gray-600">
                  Progreso: {draft?.completion}%
                </span>
                <span className="text-sm text-gray-600">
                  Modificado: {draft?.lastModified}
                </span>
              </div>
            </div>
            <Button variant="ghost" size="sm" onClick={onClose}>
              <Icon name="X" size={20} />
            </Button>
          </div>

          {/* Tabs */}
          <div className="flex space-x-4 mt-4">
            <button
              onClick={() => setActiveTab('preview')}
              className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                activeTab === 'preview' ?'bg-primary text-white' :'text-gray-600 hover:bg-gray-100'
              }`}
            >
              Vista Previa
            </button>
            <button
              onClick={() => setActiveTab('details')}
              className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${
                activeTab === 'details' ?'bg-primary text-white' :'text-gray-600 hover:bg-gray-100'
              }`}
            >
              Detalles
            </button>
          </div>
        </div>

        {/* Content */}
        <div className="flex-1 overflow-y-auto p-6">
          {activeTab === 'preview' ? (
            <div className="bg-white border rounded-lg p-8 min-h-[600px]">
              <div className="prose max-w-none">
                <pre className="whitespace-pre-wrap font-sans text-sm leading-relaxed">
                  {draft?.extractedContent || mockContractContent}
                </pre>
              </div>
            </div>
          ) : (
            <div className="space-y-6">
              {/* Contract Information */}
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="glass-card p-4 rounded-lg">
                  <h3 className="font-semibold text-gray-900 mb-3">Información del Contrato</h3>
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between">
                      <span className="text-gray-600">Plantilla:</span>
                      <span className="font-medium">{draft?.template}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Tipo:</span>
                      <span className="font-medium capitalize">{draft?.type}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Estado:</span>
                      <span className="font-medium">{getStatusBadge(draft?.status)}</span>
                    </div>
                  </div>
                </div>

                <div className="glass-card p-4 rounded-lg">
                  <h3 className="font-semibold text-gray-900 mb-3">Partes del Contrato</h3>
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between">
                      <span className="text-gray-600">Inquilino:</span>
                      <span className="font-medium">{draft?.tenant}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Propietario:</span>
                      <span className="font-medium">{draft?.owner}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Propiedad:</span>
                      <span className="font-medium">{draft?.property}</span>
                    </div>
                  </div>
                </div>
              </div>

              {/* Collaborators */}
              <div className="glass-card p-4 rounded-lg">
                <h3 className="font-semibold text-gray-900 mb-3">Colaboradores</h3>
                <div className="flex flex-wrap gap-2">
                  {draft?.collaborators?.map((collaborator, index) => (
                    <div
                      key={index}
                      className="flex items-center space-x-2 bg-gray-100 rounded-full px-3 py-1"
                    >
                      <div className="w-6 h-6 bg-primary rounded-full flex items-center justify-center">
                        <span className="text-xs font-medium text-white">
                          {collaborator?.charAt(0)?.toUpperCase()}
                        </span>
                      </div>
                      <span className="text-sm">{collaborator}</span>
                    </div>
                  ))}
                </div>
              </div>

              {/* Progress */}
              <div className="glass-card p-4 rounded-lg">
                <h3 className="font-semibold text-gray-900 mb-3">Progreso de Completado</h3>
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <span className="text-sm text-gray-600">Progreso general</span>
                    <span className="text-sm font-medium">{draft?.completion}%</span>
                  </div>
                  <div className="w-full bg-gray-200 rounded-full h-2">
                    <div
                      className="bg-primary h-2 rounded-full transition-all duration-300"
                      style={{ width: `${draft?.completion}%` }}
                    ></div>
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Footer */}
        <div className="p-6 border-t border-gray-200 flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <Button variant="outline" onClick={generatePDF}>
              <Icon name="FileText" size={16} className="mr-2" />
              Descargar PDF
            </Button>
            <Button variant="outline" onClick={generateWord}>
              <Icon name="FileText" size={16} className="mr-2" />
              Descargar Word
            </Button>
          </div>
          
          <div className="flex items-center space-x-3">
            <Button variant="outline" onClick={onClose}>
              Cerrar
            </Button>
            <Button onClick={onEdit}>
              <Icon name="Edit" size={16} className="mr-2" />
              Editar Contrato
            </Button>
          </div>
        </div>
      </div>
    </div>
  );
};

export default DraftPreviewModal;