import React, { useState, useMemo } from 'react';
import Icon from '../../../components/AppIcon';
import { Checkbox } from '../../../components/ui/Checkbox';
import Button from '../../../components/ui/Button';
import { format } from 'date-fns';

const DraftTable = ({
  drafts = [],
  selectedDrafts = [],
  onDraftSelect,
  onEdit,
  onPreview,
  onDuplicate,
  onDelete,
  loading = false
}) => {
  const [sortField, setSortField] = useState('lastModified');
  const [sortDirection, setSortDirection] = useState('desc');

  const handleSort = (field) => {
    if (sortField === field) {
      setSortDirection(prev => prev === 'asc' ? 'desc' : 'asc');
    } else {
      setSortField(field);
      setSortDirection('asc');
    }
  };

  const handleSelectAll = (checked) => {
    if (checked) {
      onDraftSelect?.(drafts?.map(draft => draft?.id) || []);
    } else {
      onDraftSelect?.([]);
    }
  };

  const handleSelectDraft = (draftId, checked) => {
    if (checked) {
      onDraftSelect?.([...selectedDrafts, draftId]);
    } else {
      onDraftSelect?.(selectedDrafts?.filter(id => id !== draftId));
    }
  };

  const getStatusBadge = (status) => {
    const statusConfig = {
      'in-progress': {
        label: 'En Progreso',
        className: 'bg-warning/10 text-warning border-warning/20'
      },
      'review': {
        label: 'En Revisión',
        className: 'bg-accent/10 text-accent border-accent/20'
      },
      'completed': {
        label: 'Completado',
        className: 'bg-success/10 text-success border-success/20'
      }
    };

    const config = statusConfig?.[status] || statusConfig?.['in-progress'];

    return (
      <span className={`px-2 py-1 rounded-full text-xs font-medium border ${config?.className}`}>
        {config?.label}
      </span>
    );
  };

  const getCompletionBar = (percentage) => {
    const colorClass = 
      percentage >= 100 ? 'bg-success' :
      percentage >= 75 ? 'bg-primary' :
      percentage >= 50 ? 'bg-accent': 'bg-warning';

    return (
      <div className="flex items-center space-x-2">
        <div className="w-16 bg-gray-200 rounded-full h-2">
          <div 
            className={`h-2 rounded-full transition-all duration-300 ${colorClass}`}
            style={{ width: `${percentage}%` }}
          ></div>
        </div>
        <span className="text-xs text-gray-600 min-w-[32px]">{percentage}%</span>
      </div>
    );
  };

  const formatDate = (dateString) => {
    try {
      return format(new Date(dateString), 'dd/MM/yyyy');
    } catch {
      return dateString;
    }
  };

  const sortedDrafts = React.useMemo(() => {
    if (!drafts?.length) return [];

    return [...drafts]?.sort((a, b) => {
      let aValue = a?.[sortField];
      let bValue = b?.[sortField];

      if (sortField === 'lastModified' || sortField === 'createdDate') {
        aValue = new Date(aValue);
        bValue = new Date(bValue);
      }

      if (typeof aValue === 'string') {
        aValue = aValue?.toLowerCase();
        bValue = bValue?.toLowerCase();
      }

      if (sortDirection === 'asc') {
        return aValue > bValue ? 1 : -1;
      } else {
        return aValue < bValue ? 1 : -1;
      }
    });
  }, [drafts, sortField, sortDirection]);

  if (loading) {
    return (
      <div className="p-8">
        <div className="animate-pulse space-y-4">
          {[...Array(5)]?.map((_, i) => (
            <div key={i} className="flex items-center space-x-4">
              <div className="h-4 w-4 bg-gray-300 rounded"></div>
              <div className="flex-1 space-y-2">
                <div className="h-4 bg-gray-300 rounded w-1/2"></div>
                <div className="h-3 bg-gray-300 rounded w-1/3"></div>
              </div>
            </div>
          ))}
        </div>
      </div>
    );
  }

  if (!sortedDrafts?.length) {
    return (
      <div className="p-12 text-center">
        <div className="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
          <Icon name="FileText" size={24} className="text-gray-400" />
        </div>
        <h3 className="text-lg font-medium text-gray-900 mb-2">No hay borradores</h3>
        <p className="text-gray-600">Crea tu primer borrador de contrato para comenzar</p>
      </div>
    );
  }

  return (
    <div className="overflow-x-auto">
      <table className="w-full">
        <thead className="bg-gray-50">
          <tr>
            <th className="px-6 py-3 text-left">
              <Checkbox
                checked={selectedDrafts?.length === drafts?.length && drafts?.length > 0}
                indeterminate={selectedDrafts?.length > 0 && selectedDrafts?.length < drafts?.length}
                onChange={handleSelectAll}
              />
            </th>
            <th className="px-6 py-3 text-left">
              <button
                onClick={() => handleSort('name')}
                className="flex items-center space-x-1 text-xs font-medium text-gray-500 uppercase tracking-wider hover:text-gray-700 transition-colors"
              >
                <span>Nombre del Borrador</span>
                <Icon 
                  name={sortField === 'name' ? (sortDirection === 'asc' ? 'ChevronUp' : 'ChevronDown') : 'ChevronsUpDown'}
                  size={14}
                />
              </button>
            </th>
            <th className="px-6 py-3 text-left">
              <button
                onClick={() => handleSort('template')}
                className="flex items-center space-x-1 text-xs font-medium text-gray-500 uppercase tracking-wider hover:text-gray-700 transition-colors"
              >
                <span>Plantilla</span>
                <Icon 
                  name={sortField === 'template' ? (sortDirection === 'asc' ? 'ChevronUp' : 'ChevronDown') : 'ChevronsUpDown'}
                  size={14}
                />
              </button>
            </th>
            <th className="px-6 py-3 text-left">
              <span className="text-xs font-medium text-gray-500 uppercase tracking-wider">Estado</span>
            </th>
            <th className="px-6 py-3 text-left">
              <span className="text-xs font-medium text-gray-500 uppercase tracking-wider">Progreso</span>
            </th>
            <th className="px-6 py-3 text-left">
              <button
                onClick={() => handleSort('lastModified')}
                className="flex items-center space-x-1 text-xs font-medium text-gray-500 uppercase tracking-wider hover:text-gray-700 transition-colors"
              >
                <span>Última Modificación</span>
                <Icon 
                  name={sortField === 'lastModified' ? (sortDirection === 'asc' ? 'ChevronUp' : 'ChevronDown') : 'ChevronsUpDown'}
                  size={14}
                />
              </button>
            </th>
            <th className="px-6 py-3 text-left">
              <span className="text-xs font-medium text-gray-500 uppercase tracking-wider">Colaboradores</span>
            </th>
            <th className="px-6 py-3 text-right">
              <span className="text-xs font-medium text-gray-500 uppercase tracking-wider">Acciones</span>
            </th>
          </tr>
        </thead>
        <tbody className="bg-white divide-y divide-gray-200">
          {sortedDrafts?.map((draft) => (
            <tr key={draft?.id} className="hover:bg-gray-50 transition-colors">
              <td className="px-6 py-4">
                <Checkbox
                  checked={selectedDrafts?.includes(draft?.id)}
                  onChange={(checked) => handleSelectDraft(draft?.id, checked)}
                />
              </td>
              <td className="px-6 py-4">
                <div>
                  <button
                    onClick={() => onPreview?.(draft)}
                    className="text-sm font-medium text-gray-900 hover:text-primary transition-colors"
                  >
                    {draft?.name}
                  </button>
                  <div className="text-xs text-gray-500 mt-1">
                    <span>Inquilino: {draft?.tenant}</span> • 
                    <span className="ml-1">Propietario: {draft?.owner}</span>
                  </div>
                </div>
              </td>
              <td className="px-6 py-4">
                <div className="flex items-center space-x-2">
                  <div className={`w-3 h-3 rounded-full ${
                    draft?.type === 'residential' ? 'bg-blue-400' : 'bg-green-400'
                  }`}></div>
                  <span className="text-sm text-gray-900">{draft?.template}</span>
                </div>
              </td>
              <td className="px-6 py-4">
                {getStatusBadge(draft?.status)}
              </td>
              <td className="px-6 py-4">
                {getCompletionBar(draft?.completion)}
              </td>
              <td className="px-6 py-4">
                <div className="text-sm text-gray-900">
                  {formatDate(draft?.lastModified)}
                </div>
                <div className="text-xs text-gray-500">
                  Creado: {formatDate(draft?.createdDate)}
                </div>
              </td>
              <td className="px-6 py-4">
                <div className="flex -space-x-2">
                  {draft?.collaborators?.slice(0, 3)?.map((collaborator, index) => (
                    <div
                      key={index}
                      className="w-6 h-6 bg-gray-300 rounded-full border-2 border-white flex items-center justify-center"
                      title={collaborator}
                    >
                      <span className="text-xs font-medium text-gray-600">
                        {collaborator?.charAt(0)?.toUpperCase()}
                      </span>
                    </div>
                  ))}
                  {(draft?.collaborators?.length || 0) > 3 && (
                    <div className="w-6 h-6 bg-gray-100 rounded-full border-2 border-white flex items-center justify-center">
                      <span className="text-xs font-medium text-gray-500">
                        +{(draft?.collaborators?.length || 0) - 3}
                      </span>
                    </div>
                  )}
                </div>
              </td>
              <td className="px-6 py-4 text-right">
                <div className="flex items-center justify-end space-x-2">
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => onPreview?.(draft)}
                    title="Vista previa"
                  >
                    <Icon name="Eye" size={16} />
                  </Button>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => onEdit?.(draft)}
                    title="Editar"
                  >
                    <Icon name="Edit" size={16} />
                  </Button>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => onDuplicate?.(draft)}
                    title="Duplicar"
                  >
                    <Icon name="Copy" size={16} />
                  </Button>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => onDelete?.(draft?.id)}
                    className="text-error hover:text-error hover:bg-error/10"
                    title="Eliminar"
                  >
                    <Icon name="Trash2" size={16} />
                  </Button>
                </div>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
};

export default DraftTable;