import React, { useState, useCallback } from 'react';
import { useDropzone } from 'react-dropzone';
import mammoth from 'mammoth';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Icon from '../../../components/AppIcon';

const UploadDraftModal = ({ isOpen, onClose, onUpload }) => {
  const [uploadedFile, setUploadedFile] = useState(null);
  const [draftData, setDraftData] = useState({
    name: '',
    template: 'Arriendo Residencial',
    tenant: '',
    owner: '',
    property: '',
    extractedContent: ''
  });
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');

  const onDrop = useCallback(async (acceptedFiles) => {
    const file = acceptedFiles?.[0];
    if (!file) return;

    setLoading(true);
    setError('');
    
    try {
      const fileExtension = file?.name?.split('.')?.pop()?.toLowerCase();
      
      if (fileExtension === 'docx') {
        // Extract text from Word document
        const arrayBuffer = await file?.arrayBuffer();
        const result = await mammoth?.extractRawText({ arrayBuffer });
        
        setDraftData(prev => ({
          ...prev,
          name: file?.name?.replace(/\.[^/.]+$/, ''),
          extractedContent: result?.value || ''
        }));
      } else if (fileExtension === 'pdf') {
        // For PDF files, we'll store the file for later processing
        setDraftData(prev => ({
          ...prev,
          name: file?.name?.replace(/\.[^/.]+$/, ''),
          extractedContent: 'Contenido PDF detectado - Se procesará al crear el borrador'
        }));
      }
      
      setUploadedFile(file);
    } catch (err) {
      setError('Error al procesar el archivo. Verifique que sea un documento válido.');
      console.error('File processing error:', err);
    } finally {
      setLoading(false);
    }
  }, []);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: {
      'application/pdf': ['.pdf'],
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document': ['.docx'],
      'application/msword': ['.doc']
    },
    multiple: false,
    maxSize: 10 * 1024 * 1024 // 10MB
  });

  const handleInputChange = (field, value) => {
    setDraftData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleSubmit = () => {
    if (!uploadedFile || !draftData?.name) {
      setError('Por favor complete la información requerida');
      return;
    }

    const newDraft = {
      id: Date.now(),
      name: draftData?.name,
      template: draftData?.template,
      tenant: draftData?.tenant || 'Por definir',
      owner: draftData?.owner || 'Por definir',
      property: draftData?.property || 'Por definir',
      createdDate: new Date()?.toISOString()?.split('T')?.[0],
      lastModified: new Date()?.toISOString()?.split('T')?.[0],
      status: 'in-progress',
      completion: 25,
      collaborators: ['Usuario Actual'],
      type: draftData?.template?.includes('Comercial') ? 'commercial' : 'residential',
      originalFile: uploadedFile,
      extractedContent: draftData?.extractedContent
    };

    onUpload?.(newDraft);
  };

  const templateOptions = [
    { value: 'Arriendo Residencial', label: 'Arriendo Residencial' },
    { value: 'Arriendo Comercial', label: 'Arriendo Comercial' },
    { value: 'Arriendo Temporal', label: 'Arriendo Temporal' },
    { value: 'Subarriendo', label: 'Subarriendo' }
  ];

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-[1000] flex items-center justify-center p-4">
      <div className="bg-white rounded-lg max-w-2xl w-full max-h-[90vh] overflow-y-auto">
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <h2 className="text-xl font-bold text-gray-900">Subir Borrador de Contrato</h2>
            <Button variant="ghost" size="sm" onClick={onClose}>
              <Icon name="X" size={20} />
            </Button>
          </div>
        </div>

        <div className="p-6 space-y-6">
          {/* File Upload Area */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Archivo del Contrato
            </label>
            <div
              {...getRootProps()}
              className={`
                border-2 border-dashed rounded-lg p-8 text-center cursor-pointer transition-colors
                ${isDragActive 
                  ? 'border-primary bg-primary/5' 
                  : uploadedFile 
                    ? 'border-success bg-success/5' :'border-gray-300 hover:border-gray-400'
                }
              `}
            >
              <input {...getInputProps()} />
              
              {loading ? (
                <div className="flex items-center justify-center space-x-2">
                  <Icon name="Loader2" size={20} className="animate-spin text-primary" />
                  <span className="text-gray-600">Procesando archivo...</span>
                </div>
              ) : uploadedFile ? (
                <div className="flex items-center justify-center space-x-2">
                  <Icon name="FileText" size={20} className="text-success" />
                  <span className="text-success font-medium">{uploadedFile?.name}</span>
                </div>
              ) : (
                <div>
                  <Icon name="Upload" size={24} className="mx-auto text-gray-400 mb-2" />
                  <p className="text-gray-600">
                    {isDragActive 
                      ? 'Suelta el archivo aquí...' :'Arrastra un archivo o haz clic para seleccionar'
                    }
                  </p>
                  <p className="text-xs text-gray-500 mt-2">
                    Formatos soportados: PDF, Word (.docx, .doc) - Máximo 10MB
                  </p>
                </div>
              )}
            </div>
          </div>

          {/* Form Fields */}
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Nombre del Borrador *
              </label>
              <Input
                value={draftData?.name}
                onChange={(e) => handleInputChange('name', e?.target?.value)}
                placeholder="Ej: Contrato Depto Las Condes"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Tipo de Plantilla
              </label>
              <Select
                value={draftData?.template}
                onValueChange={(value) => handleInputChange('template', value)}
              >
                {templateOptions?.map(option => (
                  <option key={option?.value} value={option?.value}>
                    {option?.label}
                  </option>
                ))}
              </Select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Inquilino
              </label>
              <Input
                value={draftData?.tenant}
                onChange={(e) => handleInputChange('tenant', e?.target?.value)}
                placeholder="Nombre del inquilino"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Propietario
              </label>
              <Input
                value={draftData?.owner}
                onChange={(e) => handleInputChange('owner', e?.target?.value)}
                placeholder="Nombre del propietario"
              />
            </div>

            <div className="md:col-span-2">
              <label className="block text-sm font-medium text-gray-700 mb-1">
                Propiedad
              </label>
              <Input
                value={draftData?.property}
                onChange={(e) => handleInputChange('property', e?.target?.value)}
                placeholder="Descripción de la propiedad"
              />
            </div>
          </div>

          {/* Extracted Content Preview */}
          {draftData?.extractedContent && (
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Vista Previa del Contenido Extraído
              </label>
              <div className="bg-gray-50 border rounded-lg p-4 max-h-32 overflow-y-auto">
                <p className="text-sm text-gray-600 whitespace-pre-wrap">
                  {draftData?.extractedContent?.substring(0, 500)}
                  {(draftData?.extractedContent?.length || 0) > 500 && '...'}
                </p>
              </div>
            </div>
          )}

          {error && (
            <div className="bg-error/10 border border-error/20 rounded-lg p-4">
              <div className="flex items-center space-x-2">
                <Icon name="AlertCircle" size={16} className="text-error" />
                <span className="text-sm text-error">{error}</span>
              </div>
            </div>
          )}
        </div>

        <div className="p-6 border-t border-gray-200 flex items-center justify-end space-x-3">
          <Button variant="outline" onClick={onClose}>
            Cancelar
          </Button>
          <Button 
            onClick={handleSubmit}
            disabled={!uploadedFile || !draftData?.name || loading}
          >
            <Icon name="Upload" size={16} className="mr-2" />
            Crear Borrador
          </Button>
        </div>
      </div>
    </div>
  );
};

export default UploadDraftModal;