import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import Button from '../../components/ui/Button';
import Input from '../../components/ui/Input';

import Icon from '../../components/AppIcon';
import DraftTable from './components/DraftTable';
import DraftFilters from './components/DraftFilters';
import UploadDraftModal from './components/UploadDraftModal';
import DraftPreviewModal from './components/DraftPreviewModal';
import ContractTemplateSelector from './components/ContractTemplateSelector';
import BulkActionsBar from './components/BulkActionsBar';

const DraftContractManager = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [loading, setLoading] = useState(true);
  const [drafts, setDrafts] = useState([]);
  const [filteredDrafts, setFilteredDrafts] = useState([]);
  const [selectedDrafts, setSelectedDrafts] = useState([]);
  const [searchTerm, setSearchTerm] = useState('');
  const [currentFilters, setCurrentFilters] = useState({
    status: 'all',
    template: 'all',
    dateRange: 'all'
  });
  
  // Modal states
  const [showUploadModal, setShowUploadModal] = useState(false);
  const [showPreviewModal, setShowPreviewModal] = useState(false);
  const [showTemplateSelector, setShowTemplateSelector] = useState(false);
  const [selectedDraft, setSelectedDraft] = useState(null);

  const navigate = useNavigate();

  // Mock draft data
  const mockDrafts = [
    {
      id: 1,
      name: 'Contrato Depto Las Condes 301',
      template: 'Arriendo Residencial',
      createdDate: '2024-08-15',
      lastModified: '2024-08-20',
      status: 'in-progress',
      completion: 75,
      collaborators: ['Juan Pérez', 'María Silva'],
      type: 'residential',
      tenant: 'Carlos González',
      owner: 'Ana Martínez',
      property: 'Depto 301, Las Condes'
    },
    {
      id: 2,
      name: 'Contrato Oficina Providencia',
      template: 'Arriendo Comercial',
      createdDate: '2024-08-10',
      lastModified: '2024-08-25',
      status: 'review',
      completion: 90,
      collaborators: ['Pedro López'],
      type: 'commercial',
      tenant: 'Empresa ABC Ltda.',
      owner: 'Inversiones XYZ',
      property: 'Oficina 15, Providencia'
    },
    {
      id: 3,
      name: 'Contrato Casa Ñuñoa',
      template: 'Arriendo Residencial',
      createdDate: '2024-08-05',
      lastModified: '2024-08-26',
      status: 'completed',
      completion: 100,
      collaborators: ['Luis Rivera', 'Carmen Torres'],
      type: 'residential',
      tenant: 'Familia Rodriguez',
      owner: 'Jorge Mendoza',
      property: 'Casa Ñuñoa Norte'
    }
  ];

  useEffect(() => {
    // Simulate data loading
    const timer = setTimeout(() => {
      setDrafts(mockDrafts);
      setFilteredDrafts(mockDrafts);
      setLoading(false);
    }, 1500);

    return () => clearTimeout(timer);
  }, []);

  // Filter and search logic
  useEffect(() => {
    let filtered = drafts;

    // Apply search filter
    if (searchTerm) {
      filtered = filtered?.filter(draft =>
        draft?.name?.toLowerCase()?.includes(searchTerm?.toLowerCase()) ||
        draft?.tenant?.toLowerCase()?.includes(searchTerm?.toLowerCase()) ||
        draft?.owner?.toLowerCase()?.includes(searchTerm?.toLowerCase())
      );
    }

    // Apply status filter
    if (currentFilters?.status !== 'all') {
      filtered = filtered?.filter(draft => draft?.status === currentFilters?.status);
    }

    // Apply template filter
    if (currentFilters?.template !== 'all') {
      filtered = filtered?.filter(draft => draft?.template === currentFilters?.template);
    }

    // Apply date range filter
    if (currentFilters?.dateRange !== 'all') {
      const today = new Date();
      const filterDate = new Date();
      
      switch (currentFilters?.dateRange) {
        case 'week':
          filterDate?.setDate(today?.getDate() - 7);
          break;
        case 'month':
          filterDate?.setMonth(today?.getMonth() - 1);
          break;
        case 'quarter':
          filterDate?.setMonth(today?.getMonth() - 3);
          break;
        default:
          break;
      }
      
      if (currentFilters?.dateRange !== 'all') {
        filtered = filtered?.filter(draft => new Date(draft?.lastModified) >= filterDate);
      }
    }

    setFilteredDrafts(filtered);
  }, [drafts, searchTerm, currentFilters]);

  const handleSidebarToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleSidebarClose = () => {
    setIsSidebarOpen(false);
  };

  const handleCreateNew = () => {
    setShowTemplateSelector(true);
  };

  const handleUploadDraft = () => {
    setShowUploadModal(true);
  };

  const handleDraftSelect = (draftIds) => {
    setSelectedDrafts(draftIds);
  };

  const handleDraftEdit = (draft) => {
    // Navigate to contract editing workspace
    navigate('/contract-drafting-workspace', { state: { draftId: draft?.id, draft } });
  };

  const handleDraftPreview = (draft) => {
    setSelectedDraft(draft);
    setShowPreviewModal(true);
  };

  const handleDraftDuplicate = (draft) => {
    const newDraft = {
      ...draft,
      id: Date.now(),
      name: `${draft?.name} (Copia)`,
      createdDate: new Date()?.toISOString()?.split('T')?.[0],
      lastModified: new Date()?.toISOString()?.split('T')?.[0],
      status: 'in-progress',
      completion: 0
    };
    setDrafts(prev => [newDraft, ...prev]);
  };

  const handleDraftDelete = (draftId) => {
    setDrafts(prev => prev?.filter(d => d?.id !== draftId));
    setSelectedDrafts(prev => prev?.filter(id => id !== draftId));
  };

  const handleBulkAction = (action) => {
    switch (action) {
      case 'delete':
        setDrafts(prev => prev?.filter(d => !selectedDrafts?.includes(d?.id)));
        setSelectedDrafts([]);
        break;
      case 'export':
        // Handle bulk export
        console.log('Exporting drafts:', selectedDrafts);
        break;
      case 'share':
        // Handle bulk sharing
        console.log('Sharing drafts:', selectedDrafts);
        break;
      default:
        break;
    }
  };

  const handleFilterChange = (filters) => {
    setCurrentFilters(filters);
  };

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleSidebarToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={handleSidebarClose} />
      <main className="lg:ml-60 pt-16">
        <div className="p-6 max-w-7xl mx-auto">
          {/* Breadcrumbs */}
          <NavigationBreadcrumbs />
          
          {/* Page Header */}
          <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-8">
            <div>
              <h1 className="text-3xl font-bold text-gray-900 mb-2">
                Gestor de Borradores de Contratos
              </h1>
              <p className="text-gray-600">
                Administra y organiza todos tus borradores de contratos de arriendo
              </p>
            </div>
            
            <div className="mt-4 lg:mt-0 flex flex-col sm:flex-row gap-3">
              <Button
                variant="outline"
                onClick={handleUploadDraft}
                className="flex items-center space-x-2"
              >
                <Icon name="Upload" size={16} />
                <span>Subir Borrador</span>
              </Button>
              
              <Button
                onClick={handleCreateNew}
                className="flex items-center space-x-2"
              >
                <Icon name="Plus" size={16} />
                <span>Nuevo Contrato</span>
              </Button>
            </div>
          </div>

          {/* Stats Cards */}
          <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
            <div className="glass-card p-6 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Total Borradores</p>
                  <p className="text-2xl font-bold text-gray-900">{drafts?.length}</p>
                </div>
                <div className="w-12 h-12 bg-primary/10 rounded-lg flex items-center justify-center">
                  <Icon name="FileText" size={24} className="text-primary" />
                </div>
              </div>
            </div>
            
            <div className="glass-card p-6 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">En Progreso</p>
                  <p className="text-2xl font-bold text-warning">
                    {drafts?.filter(d => d?.status === 'in-progress')?.length}
                  </p>
                </div>
                <div className="w-12 h-12 bg-warning/10 rounded-lg flex items-center justify-center">
                  <Icon name="Clock" size={24} className="text-warning" />
                </div>
              </div>
            </div>
            
            <div className="glass-card p-6 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">En Revisión</p>
                  <p className="text-2xl font-bold text-accent">
                    {drafts?.filter(d => d?.status === 'review')?.length}
                  </p>
                </div>
                <div className="w-12 h-12 bg-accent/10 rounded-lg flex items-center justify-center">
                  <Icon name="Eye" size={24} className="text-accent" />
                </div>
              </div>
            </div>
            
            <div className="glass-card p-6 rounded-lg">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm text-gray-600 mb-1">Completados</p>
                  <p className="text-2xl font-bold text-success">
                    {drafts?.filter(d => d?.status === 'completed')?.length}
                  </p>
                </div>
                <div className="w-12 h-12 bg-success/10 rounded-lg flex items-center justify-center">
                  <Icon name="CheckCircle" size={24} className="text-success" />
                </div>
              </div>
            </div>
          </div>

          {/* Search and Filters */}
          <div className="glass-card p-6 rounded-lg mb-6">
            <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
              <div className="flex-1 max-w-md">
                <div className="relative">
                  <Icon name="Search" size={16} className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
                  <Input
                    placeholder="Buscar por nombre, inquilino, propietario..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e?.target?.value)}
                    className="pl-10"
                  />
                </div>
              </div>
              
              <DraftFilters
                filters={currentFilters}
                onFiltersChange={handleFilterChange}
                loading={loading}
              />
            </div>
          </div>

          {/* Bulk Actions */}
          {selectedDrafts?.length > 0 && (
            <BulkActionsBar
              selectedCount={selectedDrafts?.length}
              onBulkAction={handleBulkAction}
            />
          )}

          {/* Drafts Table */}
          <div className="glass-card rounded-lg overflow-hidden">
            <DraftTable
              drafts={filteredDrafts}
              selectedDrafts={selectedDrafts}
              onDraftSelect={handleDraftSelect}
              onEdit={handleDraftEdit}
              onPreview={handleDraftPreview}
              onDuplicate={handleDraftDuplicate}
              onDelete={handleDraftDelete}
              loading={loading}
            />
          </div>
        </div>
      </main>
      {/* Modals */}
      {showUploadModal && (
        <UploadDraftModal
          isOpen={showUploadModal}
          onClose={() => setShowUploadModal(false)}
          onUpload={(draftData) => {
            setDrafts(prev => [draftData, ...prev]);
            setShowUploadModal(false);
          }}
        />
      )}
      {showPreviewModal && selectedDraft && (
        <DraftPreviewModal
          isOpen={showPreviewModal}
          onClose={() => {
            setShowPreviewModal(false);
            setSelectedDraft(null);
          }}
          draft={selectedDraft}
          onEdit={() => {
            setShowPreviewModal(false);
            handleDraftEdit(selectedDraft);
          }}
        />
      )}
      {showTemplateSelector && (
        <ContractTemplateSelector
          isOpen={showTemplateSelector}
          onClose={() => setShowTemplateSelector(false)}
          onSelect={(template) => {
            // Navigate to contract creation with selected template
            navigate('/contract-drafting-workspace', { 
              state: { templateId: template?.id, template, isNew: true } 
            });
            setShowTemplateSelector(false);
          }}
        />
      )}
    </div>
  );
};

export default DraftContractManager;