import React from 'react';
import Icon from '../../../components/AppIcon';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const BillingPeriodStep = ({ billingPeriod, onPeriodChange }) => {
  const currentDate = new Date();
  const currentYear = currentDate?.getFullYear();
  
  const months = [
    { value: 1, label: 'Enero' },
    { value: 2, label: 'Febrero' },
    { value: 3, label: 'Marzo' },
    { value: 4, label: 'Abril' },
    { value: 5, label: 'Mayo' },
    { value: 6, label: 'Junio' },
    { value: 7, label: 'Julio' },
    { value: 8, label: 'Agosto' },
    { value: 9, label: 'Septiembre' },
    { value: 10, label: 'Octubre' },
    { value: 11, label: 'Noviembre' },
    { value: 12, label: 'Diciembre' }
  ];

  const years = Array.from({ length: 5 }, (_, i) => ({
    value: currentYear - 2 + i,
    label: (currentYear - 2 + i)?.toString()
  }));

  const handleMonthChange = (month) => {
    const monthNumber = parseInt(month);
    const year = billingPeriod?.year || currentYear;
    
    // Calculate start and end dates based on month/year
    const startDate = new Date(year, monthNumber - 1, 1)?.toISOString()?.split('T')?.[0];
    const endDate = new Date(year, monthNumber, 0)?.toISOString()?.split('T')?.[0];
    
    onPeriodChange({
      ...billingPeriod,
      month: monthNumber,
      startDate,
      endDate
    });
  };

  const handleYearChange = (year) => {
    const yearNumber = parseInt(year);
    const month = billingPeriod?.month || currentDate?.getMonth() + 1;
    
    const startDate = new Date(yearNumber, month - 1, 1)?.toISOString()?.split('T')?.[0];
    const endDate = new Date(yearNumber, month, 0)?.toISOString()?.split('T')?.[0];
    
    onPeriodChange({
      ...billingPeriod,
      year: yearNumber,
      startDate,
      endDate
    });
  };

  const handleCustomDateChange = (field, value) => {
    onPeriodChange({
      ...billingPeriod,
      [field]: value
    });
  };

  const isPastPeriod = () => {
    const selectedDate = new Date(billingPeriod?.year || currentYear, (billingPeriod?.month || 1) - 1);
    return selectedDate < new Date(currentYear, currentDate.getMonth());
  };

  const isFuturePeriod = () => {
    const selectedDate = new Date(billingPeriod?.year || currentYear, (billingPeriod?.month || 1) - 1);
    return selectedDate > new Date(currentYear, currentDate.getMonth());
  };

  return (
    <div className="space-y-6">
      {/* Quick Period Selection */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div className="space-y-4">
          <h3 className="text-lg font-medium text-gray-900">Selección Rápida de Período</h3>
          
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Mes de Facturación
              </label>
              <Select
                options={months}
                value={billingPeriod?.month || ''}
                onChange={handleMonthChange}
                placeholder="Selecciona un mes..."
                className="w-full"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Año
              </label>
              <Select
                options={years}
                value={billingPeriod?.year || ''}
                onChange={handleYearChange}
                placeholder="Selecciona un año..."
                className="w-full"
              />
            </div>
          </div>

          {/* Period Info */}
          {billingPeriod?.month && billingPeriod?.year && (
            <div className={`rounded-lg p-4 ${
              isPastPeriod() ? 'bg-orange-50' : isFuturePeriod() ? 'bg-blue-50' : 'bg-green-50'
            }`}>
              <div className="flex items-start space-x-3">
                <Icon 
                  name={isPastPeriod() ? 'Clock' : isFuturePeriod() ? 'Calendar' : 'CheckCircle'} 
                  size={20} 
                  className={`${
                    isPastPeriod() ? 'text-orange-600' : isFuturePeriod() ? 'text-blue-600' : 'text-green-600'
                  } mt-0.5`} 
                />
                <div>
                  <h4 className="font-medium text-gray-900">
                    Período: {months?.find(m => m?.value === billingPeriod?.month)?.label} {billingPeriod?.year}
                  </h4>
                  <p className={`text-sm mt-1 ${
                    isPastPeriod() ? 'text-orange-700' : isFuturePeriod() ? 'text-blue-700' : 'text-green-700'
                  }`}>
                    {isPastPeriod() && 'Período pasado - Se generará comprobante retroactivo'}
                    {isFuturePeriod() && 'Período futuro - Se generará comprobante anticipado'}
                    {!isPastPeriod() && !isFuturePeriod() && 'Período actual - Generación normal'}
                  </p>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Custom Date Range */}
        <div className="space-y-4">
          <h3 className="text-lg font-medium text-gray-900">Rango de Fechas Personalizado</h3>
          
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Fecha de Inicio
              </label>
              <Input
                type="date"
                value={billingPeriod?.startDate || ''}
                onChange={(e) => handleCustomDateChange('startDate', e?.target?.value)}
                className="w-full"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Fecha de Término
              </label>
              <Input
                type="date"
                value={billingPeriod?.endDate || ''}
                onChange={(e) => handleCustomDateChange('endDate', e?.target?.value)}
                className="w-full"
                min={billingPeriod?.startDate}
              />
            </div>
          </div>

          {/* Custom Range Info */}
          {billingPeriod?.startDate && billingPeriod?.endDate && (
            <div className="bg-purple-50 rounded-lg p-4">
              <div className="flex items-start space-x-3">
                <Icon name="Calendar" size={20} className="text-purple-600 mt-0.5" />
                <div>
                  <h4 className="font-medium text-gray-900">Rango Personalizado</h4>
                  <p className="text-sm text-purple-700 mt-1">
                    Del {new Date(billingPeriod?.startDate)?.toLocaleDateString('es-CL')} al {' '}
                    {new Date(billingPeriod?.endDate)?.toLocaleDateString('es-CL')}
                  </p>
                  <p className="text-xs text-purple-600 mt-1">
                    {Math.ceil((new Date(billingPeriod?.endDate) - new Date(billingPeriod?.startDate)) / (1000 * 60 * 60 * 24))} días
                  </p>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>
      {/* Calendar Interface */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h3 className="text-lg font-medium text-gray-900 mb-4">Vista de Calendario</h3>
        <div className="grid grid-cols-7 gap-1 text-center text-sm">
          {['Dom', 'Lun', 'Mar', 'Mié', 'Jue', 'Vie', 'Sáb']?.map((day) => (
            <div key={day} className="p-2 font-medium text-gray-700 bg-gray-50 rounded">
              {day}
            </div>
          ))}
        </div>
        
        <div className="mt-4 text-center py-8 text-gray-500">
          <Icon name="Calendar" size={48} className="mx-auto mb-2 text-gray-300" />
          <p>Interfaz de calendario en desarrollo</p>
          <p className="text-xs mt-1">Usa los selectores de fecha mientras tanto</p>
        </div>
      </div>
      {/* Period Validation */}
      {(!billingPeriod?.month || !billingPeriod?.year) && (!billingPeriod?.startDate || !billingPeriod?.endDate) && (
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
          <div className="flex items-start space-x-3">
            <Icon name="AlertTriangle" size={20} className="text-yellow-600 flex-shrink-0 mt-0.5" />
            <div>
              <p className="text-sm text-yellow-800">
                <strong>Período Requerido:</strong> Selecciona un período de facturación usando la selección rápida o el rango personalizado.
              </p>
            </div>
          </div>
        </div>
      )}
      {/* Period Summary */}
      <div className="bg-gray-50 rounded-lg p-4">
        <h4 className="font-medium text-gray-900 mb-2">Resumen del Período</h4>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 text-sm">
          <div className="flex items-center space-x-2">
            <Icon name="Calendar" size={16} className="text-gray-500" />
            <span className="text-gray-600">Período:</span>
            <span className="font-medium">
              {billingPeriod?.month && billingPeriod?.year
                ? `${months?.find(m => m?.value === billingPeriod?.month)?.label} ${billingPeriod?.year}`
                : billingPeriod?.startDate && billingPeriod?.endDate
                  ? 'Personalizado'
                  : 'No seleccionado'
              }
            </span>
          </div>
          <div className="flex items-center space-x-2">
            <Icon name="Clock" size={16} className="text-gray-500" />
            <span className="text-gray-600">Tipo:</span>
            <span className="font-medium">
              {isPastPeriod() ? 'Retroactivo' : isFuturePeriod() ? 'Anticipado' : 'Actual'}
            </span>
          </div>
          <div className="flex items-center space-x-2">
            <Icon name="FileText" size={16} className="text-gray-500" />
            <span className="text-gray-600">Estado:</span>
            <span className="font-medium">Configurable</span>
          </div>
        </div>
      </div>
    </div>
  );
};

export default BillingPeriodStep;