import React from 'react';
import Icon from '../../../components/AppIcon';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const DiscountApplicationStep = ({ discounts, totals, onDiscountsChange }) => {
  const discountTypes = [
    { value: 'percentage', label: 'Porcentaje (%)' },
    { value: 'fixed', label: 'Monto Fijo ($)' }
  ];

  const discountReasons = [
    { value: 'tenant_discount', label: 'Descuento al Arrendatario' },
    { value: 'owner_discount', label: 'Descuento al Propietario' },
    { value: 'maintenance_discount', label: 'Descuento por Mantención' },
    { value: 'early_payment', label: 'Descuento por Pago Anticipado' },
    { value: 'loyalty_discount', label: 'Descuento por Fidelidad' },
    { value: 'negotiated_discount', label: 'Descuento Negociado' },
    { value: 'other', label: 'Otro Motivo' }
  ];

  const handleDiscountChange = (field, value) => {
    onDiscountsChange({
      ...discounts,
      [field]: value
    });
  };

  const calculateDiscountAmount = () => {
    if (!discounts?.amount) return 0;
    
    if (discounts?.type === 'percentage') {
      return (totals?.subtotal || 0) * (discounts?.amount / 100);
    } else {
      return discounts?.amount;
    }
  };

  const getDiscountedTotal = () => {
    return Math.max(0, (totals?.subtotal || 0) - calculateDiscountAmount());
  };

  const isValidDiscount = () => {
    const discountAmount = calculateDiscountAmount();
    return discountAmount <= (totals?.subtotal || 0);
  };

  return (
    <div className="space-y-6">
      {/* Discount Configuration */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h3 className="text-lg font-medium text-gray-900 mb-4">Configuración de Descuentos</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Tipo de Descuento
              </label>
              <Select
                options={discountTypes}
                value={discounts?.type || 'percentage'}
                onChange={(value) => handleDiscountChange('type', value)}
                className="w-full"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                {discounts?.type === 'percentage' ? 'Porcentaje de Descuento (%)' : 'Monto del Descuento ($)'}
              </label>
              <Input
                type="number"
                value={discounts?.amount || 0}
                onChange={(e) => handleDiscountChange('amount', parseFloat(e?.target?.value) || 0)}
                placeholder={discounts?.type === 'percentage' ? 'Ej: 10' : 'Ej: 50000'}
                className="w-full"
                min="0"
                max={discounts?.type === 'percentage' ? '100' : undefined}
                step={discounts?.type === 'percentage' ? '0.1' : '1000'}
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Motivo del Descuento
              </label>
              <Select
                options={discountReasons}
                value={discounts?.reason || ''}
                onChange={(value) => handleDiscountChange('reason', value)}
                placeholder="Selecciona un motivo..."
                className="w-full"
              />
            </div>
          </div>

          <div className="space-y-4">
            {/* Discount Preview */}
            <div className="bg-blue-50 rounded-lg p-4">
              <h4 className="font-medium text-blue-900 mb-3">Vista Previa del Descuento</h4>
              <div className="space-y-2 text-sm">
                <div className="flex justify-between">
                  <span className="text-blue-700">Tipo:</span>
                  <span className="font-medium">
                    {discounts?.type === 'percentage' ? 'Porcentaje' : 'Monto Fijo'}
                  </span>
                </div>
                <div className="flex justify-between">
                  <span className="text-blue-700">Descuento:</span>
                  <span className="font-medium">
                    {discounts?.type === 'percentage' 
                      ? `${discounts?.amount || 0}%` 
                      : `$${(discounts?.amount || 0)?.toLocaleString()}`
                    }
                  </span>
                </div>
                <div className="flex justify-between">
                  <span className="text-blue-700">Monto Descontado:</span>
                  <span className="font-bold text-blue-900">
                    $${calculateDiscountAmount()?.toLocaleString()}
                  </span>
                </div>
              </div>
            </div>

            {/* Validation Warning */}
            {discounts?.amount > 0 && !isValidDiscount() && (
              <div className="bg-red-50 border border-red-200 rounded-lg p-4">
                <div className="flex items-start space-x-3">
                  <Icon name="AlertTriangle" size={20} className="text-red-600 flex-shrink-0 mt-0.5" />
                  <div>
                    <h4 className="font-medium text-red-900">Descuento Excesivo</h4>
                    <p className="text-sm text-red-700 mt-1">
                      El descuento no puede ser mayor al subtotal del comprobante.
                    </p>
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
      {/* Common Discount Presets */}
      <div className="bg-gray-50 rounded-lg p-6">
        <h4 className="font-medium text-gray-900 mb-4">Descuentos Comunes</h4>
        <div className="grid grid-cols-2 md:grid-cols-4 gap-3">
          {[
            { label: '5%', type: 'percentage', amount: 5, reason: 'early_payment' },
            { label: '10%', type: 'percentage', amount: 10, reason: 'loyalty_discount' },
            { label: '$25.000', type: 'fixed', amount: 25000, reason: 'tenant_discount' },
            { label: '$50.000', type: 'fixed', amount: 50000, reason: 'maintenance_discount' }
          ]?.map((preset, index) => (
            <button
              key={index}
              onClick={() => onDiscountsChange({
                type: preset?.type,
                amount: preset?.amount,
                reason: preset?.reason
              })}
              className="p-3 text-center border border-gray-300 rounded-lg hover:bg-white hover:border-primary hover:text-primary transition-colors"
            >
              <span className="font-medium">{preset?.label}</span>
            </button>
          ))}
        </div>
      </div>
      {/* Discount Summary */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h4 className="font-medium text-gray-900 mb-4">Resumen de Cálculos</h4>
        <div className="space-y-3">
          <div className="flex justify-between items-center">
            <span className="text-gray-600">Subtotal (antes de descuentos):</span>
            <span className="font-medium">
              ${(totals?.subtotal || 0)?.toLocaleString()}
            </span>
          </div>
          
          {discounts?.amount > 0 && (
            <>
              <div className="flex justify-between items-center">
                <span className="text-gray-600">
                  Descuento ({discounts?.type === 'percentage' ? `${discounts?.amount}%` : 'Fijo'}):
                </span>
                <span className="font-medium text-red-600">
                  -${calculateDiscountAmount()?.toLocaleString()}
                </span>
              </div>
              
              {discounts?.reason && (
                <div className="flex justify-between items-center text-sm">
                  <span className="text-gray-500">Motivo:</span>
                  <span className="text-gray-700">
                    {discountReasons?.find(r => r?.value === discounts?.reason)?.label || discounts?.reason}
                  </span>
                </div>
              )}
            </>
          )}
          
          <hr className="border-gray-300" />
          <div className="flex justify-between items-center text-lg">
            <span className="font-semibold text-gray-900">Total Final:</span>
            <span className={`font-bold ${isValidDiscount() ? 'text-green-600' : 'text-red-600'}`}>
              ${getDiscountedTotal()?.toLocaleString()}
            </span>
          </div>
        </div>
      </div>
      {/* Additional Notes */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h4 className="font-medium text-gray-900 mb-4">Notas Adicionales</h4>
        <textarea
          value={discounts?.notes || ''}
          onChange={(e) => handleDiscountChange('notes', e?.target?.value)}
          placeholder="Agregar notas adicionales sobre el descuento aplicado..."
          className="w-full h-24 p-3 border border-gray-300 rounded-md resize-none focus:ring-2 focus:ring-primary focus:border-primary"
        />
      </div>
      {/* Discount History */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <div className="flex items-start space-x-3">
          <Icon name="Info" size={20} className="text-blue-600 mt-0.5" />
          <div>
            <h4 className="font-medium text-blue-900">Información sobre Descuentos</h4>
            <div className="text-sm text-blue-700 mt-2 space-y-1">
              <p>• Los descuentos al arrendatario reducen el monto a pagar por el inquilino</p>
              <p>• Los descuentos al propietario reducen la comisión o retención del administrador</p>
              <p>• Todos los descuentos quedan registrados en el historial del comprobante</p>
              <p>• Los descuentos pueden ser editados posteriormente si es necesario</p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default DiscountApplicationStep;