import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';

const PreviewAndGenerateStep = ({ voucherData, onGenerate }) => {
  const [isGenerating, setIsGenerating] = useState(false);

  const calculateTotals = () => {
    const rentAmount = voucherData?.voucherDetails?.rentAmount || 0;
    const ipcAmount = voucherData?.voucherDetails?.ipcAdjustment 
      ? (rentAmount * (voucherData?.voucherDetails?.ipcPercentage || 0)) / 100 
      : 0;
    
    const additionalExpensesTotal = voucherData?.voucherDetails?.additionalExpenses?.reduce((sum, expense) => sum + (expense?.amount || 0), 0) || 0;
    const servicesTotal = Object.values(voucherData?.servicePayments || {})?.reduce((sum, service) => sum + (service?.amount || 0), 0) || 0;
    const subtotal = rentAmount + ipcAmount + additionalExpensesTotal + servicesTotal;
    
    let discountAmount = 0;
    if (voucherData?.discounts?.type === 'percentage') {
      discountAmount = (subtotal * (voucherData?.discounts?.amount || 0)) / 100;
    } else {
      discountAmount = voucherData?.discounts?.amount || 0;
    }
    
    const finalTotal = Math.max(0, subtotal - discountAmount);

    return {
      rentAmount,
      ipcAmount,
      additionalExpensesTotal,
      servicesTotal,
      subtotal,
      discountAmount,
      finalTotal
    };
  };

  const totals = calculateTotals();

  const handleGenerate = async () => {
    setIsGenerating(true);
    try {
      await onGenerate();
    } finally {
      setIsGenerating(false);
    }
  };

  const formatCurrency = (amount) => {
    return `$${amount?.toLocaleString()} ${voucherData?.voucherDetails?.currency || 'CLP'}`;
  };

  const formatDate = (date) => {
    if (!date) return 'No especificada';
    return new Date(date)?.toLocaleDateString('es-CL');
  };

  return (
    <div className="space-y-6">
      {/* Preview Header */}
      <div className="bg-gradient-to-r from-primary to-blue-600 text-white rounded-lg p-6">
        <div className="flex items-center space-x-4">
          <div className="w-16 h-16 bg-white bg-opacity-20 rounded-lg flex items-center justify-center">
            <Icon name="FileText" size={32} className="text-white" />
          </div>
          <div>
            <h2 className="text-2xl font-bold">Comprobante de Arriendo</h2>
            <p className="text-blue-100 mt-1">
              {voucherData?.property?.name || 'Propiedad no seleccionada'}
            </p>
            <p className="text-blue-200 text-sm">
              Número: COMP-{new Date()?.getFullYear()}-{String(Math.floor(Math.random() * 9999))?.padStart(4, '0')}
            </p>
          </div>
        </div>
      </div>
      {/* Property and Tenant Information */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <div className="bg-white border border-gray-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Icon name="Home" size={20} className="mr-2 text-blue-600" />
            Información de la Propiedad
          </h3>
          <div className="space-y-3">
            <div>
              <span className="text-sm text-gray-600">Nombre:</span>
              <p className="font-medium">{voucherData?.property?.name || 'No especificado'}</p>
            </div>
            <div>
              <span className="text-sm text-gray-600">Dirección:</span>
              <p className="font-medium">{voucherData?.property?.address || 'No especificada'}</p>
            </div>
          </div>
        </div>

        <div className="bg-white border border-gray-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
            <Icon name="User" size={20} className="mr-2 text-green-600" />
            Información del Inquilino
          </h3>
          <div className="space-y-3">
            <div>
              <span className="text-sm text-gray-600">Nombre:</span>
              <p className="font-medium">{voucherData?.tenant?.name || 'No especificado'}</p>
            </div>
            <div>
              <span className="text-sm text-gray-600">Email:</span>
              <p className="font-medium">{voucherData?.tenant?.email || 'No especificado'}</p>
            </div>
            <div>
              <span className="text-sm text-gray-600">Teléfono:</span>
              <p className="font-medium">{voucherData?.tenant?.phone || 'No especificado'}</p>
            </div>
          </div>
        </div>
      </div>
      {/* Billing Period */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
          <Icon name="Calendar" size={20} className="mr-2 text-purple-600" />
          Período de Facturación
        </h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div>
            <span className="text-sm text-gray-600">Mes/Año:</span>
            <p className="font-medium">
              {voucherData?.billingPeriod?.month && voucherData?.billingPeriod?.year
                ? `${voucherData?.billingPeriod?.month}/${voucherData?.billingPeriod?.year}`
                : 'Personalizado'
              }
            </p>
          </div>
          <div>
            <span className="text-sm text-gray-600">Fecha Inicio:</span>
            <p className="font-medium">{formatDate(voucherData?.billingPeriod?.startDate)}</p>
          </div>
          <div>
            <span className="text-sm text-gray-600">Fecha Término:</span>
            <p className="font-medium">{formatDate(voucherData?.billingPeriod?.endDate)}</p>
          </div>
        </div>
      </div>
      {/* Detailed Breakdown */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center">
          <Icon name="Calculator" size={20} className="mr-2 text-yellow-600" />
          Desglose Detallado
        </h3>

        <div className="space-y-4">
          {/* Base Rent */}
          <div className="flex justify-between items-center py-2">
            <span className="text-gray-700">Arriendo Base:</span>
            <span className="font-medium">{formatCurrency(totals?.rentAmount)}</span>
          </div>

          {/* IPC Adjustment */}
          {voucherData?.voucherDetails?.ipcAdjustment && (
            <div className="flex justify-between items-center py-2">
              <span className="text-gray-700">
                Reajuste IPC ({voucherData?.voucherDetails?.ipcPercentage}%):
              </span>
              <span className="font-medium text-green-600">
                +{formatCurrency(totals?.ipcAmount)}
              </span>
            </div>
          )}

          {/* Additional Expenses */}
          {voucherData?.voucherDetails?.additionalExpenses?.length > 0 && (
            <>
              <div className="border-t pt-2">
                <p className="text-gray-700 font-medium mb-2">Gastos Adicionales:</p>
                {voucherData?.voucherDetails?.additionalExpenses?.map((expense) => (
                  <div key={expense?.id} className="flex justify-between items-center py-1 pl-4">
                    <span className="text-gray-600">• {expense?.description}:</span>
                    <span className="font-medium text-blue-600">
                      +{formatCurrency(expense?.amount)}
                    </span>
                  </div>
                ))}
              </div>
            </>
          )}

          {/* Services */}
          <div className="border-t pt-2">
            <p className="text-gray-700 font-medium mb-2">Servicios Básicos:</p>
            {Object.entries(voucherData?.servicePayments || {})?.map(([key, service]) => {
              if (!service?.amount) return null;
              const serviceLabels = {
                electricity: 'Electricidad',
                water: 'Agua',
                gas: 'Gas',
                commonExpenses: 'Gastos Comunes'
              };
              
              return (
                <div key={key} className="flex justify-between items-center py-1 pl-4">
                  <div className="flex items-center space-x-2">
                    <span className="text-gray-600">• {serviceLabels?.[key]}:</span>
                    {service?.connected && (
                      <Icon name="Wifi" size={12} className="text-green-600" />
                    )}
                  </div>
                  <span className="font-medium text-orange-600">
                    +{formatCurrency(service?.amount)}
                  </span>
                </div>
              );
            })}
          </div>

          {/* Subtotal */}
          <div className="border-t pt-2">
            <div className="flex justify-between items-center py-2">
              <span className="text-gray-900 font-semibold">Subtotal:</span>
              <span className="font-semibold">{formatCurrency(totals?.subtotal)}</span>
            </div>
          </div>

          {/* Discounts */}
          {voucherData?.discounts?.amount > 0 && (
            <div className="border-t pt-2">
              <div className="flex justify-between items-center py-2">
                <span className="text-gray-700">
                  Descuento ({voucherData?.discounts?.type === 'percentage' 
                    ? `${voucherData?.discounts?.amount}%` 
                    : 'Fijo'
                  }):
                </span>
                <span className="font-medium text-red-600">
                  -{formatCurrency(totals?.discountAmount)}
                </span>
              </div>
              {voucherData?.discounts?.reason && (
                <div className="flex justify-between items-center py-1 text-sm">
                  <span className="text-gray-500 pl-4">• Motivo:</span>
                  <span className="text-gray-600">{voucherData?.discounts?.reason}</span>
                </div>
              )}
            </div>
          )}

          {/* Final Total */}
          <div className="border-t-2 pt-4">
            <div className="flex justify-between items-center">
              <span className="text-xl font-bold text-gray-900">Total a Pagar:</span>
              <span className="text-2xl font-bold text-primary">
                {formatCurrency(totals?.finalTotal)}
              </span>
            </div>
          </div>
        </div>
      </div>
      {/* Generation Options */}
      <div className="bg-gray-50 rounded-lg p-6">
        <h4 className="font-medium text-gray-900 mb-4">Opciones de Generación</h4>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="flex items-center space-x-2">
            <Icon name="FileText" size={16} className="text-gray-600" />
            <span className="text-sm text-gray-700">Formato PDF automático</span>
          </div>
          <div className="flex items-center space-x-2">
            <Icon name="Download" size={16} className="text-gray-600" />
            <span className="text-sm text-gray-700">Descarga inmediata</span>
          </div>
          <div className="flex items-center space-x-2">
            <Icon name="Archive" size={16} className="text-gray-600" />
            <span className="text-sm text-gray-700">Almacenamiento automático</span>
          </div>
        </div>
      </div>
      {/* Generation Actions */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between space-y-4 lg:space-y-0">
          <div>
            <h3 className="text-lg font-semibold text-gray-900">¿Todo se ve correcto?</h3>
            <p className="text-sm text-gray-600 mt-1">
              Revisa todos los detalles antes de generar el comprobante final.
            </p>
          </div>
          
          <div className="flex space-x-3">
            <Button
              variant="outline"
              iconName="Eye"
              iconPosition="left"
              onClick={() => console.log('Preview voucher')}
            >
              Vista Previa PDF
            </Button>
            
            <Button
              iconName={isGenerating ? 'Loader2' : 'FileText'}
              iconPosition="left"
              onClick={handleGenerate}
              disabled={isGenerating}
              className={`bg-green-600 hover:bg-green-700 ${isGenerating ? 'animate-spin' : ''}`}
            >
              {isGenerating ? 'Generando...' : 'Generar Comprobante'}
            </Button>
          </div>
        </div>
      </div>
      {/* Validation Warnings */}
      <div className="space-y-3">
        {!voucherData?.property && (
          <div className="bg-red-50 border border-red-200 rounded-lg p-4">
            <div className="flex items-start space-x-3">
              <Icon name="AlertTriangle" size={20} className="text-red-600 flex-shrink-0 mt-0.5" />
              <div>
                <p className="text-sm text-red-800">
                  <strong>Falta Propiedad:</strong> Debes seleccionar una propiedad para generar el comprobante.
                </p>
              </div>
            </div>
          </div>
        )}
        
        {!voucherData?.voucherDetails?.rentAmount && (
          <div className="bg-red-50 border border-red-200 rounded-lg p-4">
            <div className="flex items-start space-x-3">
              <Icon name="AlertTriangle" size={20} className="text-red-600 flex-shrink-0 mt-0.5" />
              <div>
                <p className="text-sm text-red-800">
                  <strong>Falta Monto:</strong> Debes especificar el monto del arriendo.
                </p>
              </div>
            </div>
          </div>
        )}

        {totals?.finalTotal === 0 && (
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
            <div className="flex items-start space-x-3">
              <Icon name="AlertTriangle" size={20} className="text-yellow-600 flex-shrink-0 mt-0.5" />
              <div>
                <p className="text-sm text-yellow-800">
                  <strong>Total Cero:</strong> El total final del comprobante es $0. Verifica los descuentos aplicados.
                </p>
              </div>
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

export default PreviewAndGenerateStep;