import React from 'react';
import Icon from '../../../components/AppIcon';
import Select from '../../../components/ui/Select';

const PropertySelectionStep = ({ properties, selectedProperty, onPropertySelect, onTenantSelect }) => {
  const handlePropertyChange = (propertyId) => {
    const property = properties?.find(p => p?.id === propertyId);
    if (property) {
      onPropertySelect(property);
      onTenantSelect(property?.tenant);
    }
  };

  return (
    <div className="space-y-6">
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        {/* Property Selection */}
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Seleccionar Propiedad *
            </label>
            <Select
              options={properties?.map(property => ({
                value: property?.id,
                label: property?.name
              })) || []}
              value={selectedProperty?.id || ''}
              onChange={handlePropertyChange}
              placeholder="Selecciona una propiedad..."
              className="w-full"
            />
          </div>

          {selectedProperty && (
            <div className="bg-blue-50 rounded-lg p-4">
              <div className="flex items-start space-x-3">
                <div className="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center">
                  <Icon name="Home" size={20} className="text-blue-600" />
                </div>
                <div className="flex-1">
                  <h3 className="font-medium text-gray-900">{selectedProperty?.name}</h3>
                  <p className="text-sm text-gray-600 mt-1">{selectedProperty?.address}</p>
                  <div className="flex items-center space-x-4 mt-2">
                    <span className="text-sm text-gray-500">Arriendo Actual:</span>
                    <span className="font-medium text-blue-600">
                      ${selectedProperty?.currentRent?.toLocaleString()} {selectedProperty?.currency}
                    </span>
                  </div>
                </div>
              </div>
            </div>
          )}
        </div>

        {/* Tenant Information */}
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Información del Inquilino
            </label>
            {selectedProperty?.tenant ? (
              <div className="bg-green-50 rounded-lg p-4">
                <div className="flex items-start space-x-3">
                  <div className="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center">
                    <Icon name="User" size={20} className="text-green-600" />
                  </div>
                  <div className="flex-1">
                    <h3 className="font-medium text-gray-900">{selectedProperty?.tenant?.name}</h3>
                    <div className="space-y-1 mt-2">
                      <div className="flex items-center space-x-2">
                        <Icon name="Mail" size={14} className="text-gray-400" />
                        <span className="text-sm text-gray-600">{selectedProperty?.tenant?.email}</span>
                      </div>
                      <div className="flex items-center space-x-2">
                        <Icon name="Phone" size={14} className="text-gray-400" />
                        <span className="text-sm text-gray-600">{selectedProperty?.tenant?.phone}</span>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            ) : (
              <div className="bg-gray-50 rounded-lg p-4 text-center">
                <Icon name="Users" size={32} className="text-gray-400 mx-auto mb-2" />
                <p className="text-gray-500">Selecciona una propiedad para ver el inquilino</p>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Property Grid View */}
      <div>
        <h3 className="text-lg font-medium text-gray-900 mb-4">Propiedades Disponibles</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
          {properties?.map((property) => (
            <div
              key={property?.id}
              className={`p-4 rounded-lg border-2 cursor-pointer transition-all ${
                selectedProperty?.id === property?.id
                  ? 'border-primary bg-primary/5' :'border-gray-200 hover:border-gray-300 hover:bg-gray-50'
              }`}
              onClick={() => handlePropertyChange(property?.id)}
            >
              <div className="flex items-start space-x-3">
                <div className={`w-10 h-10 rounded-lg flex items-center justify-center ${
                  selectedProperty?.id === property?.id
                    ? 'bg-primary text-white' :'bg-gray-100 text-gray-600'
                }`}>
                  <Icon name="Home" size={20} />
                </div>
                <div className="flex-1 min-w-0">
                  <h4 className="font-medium text-gray-900 truncate">{property?.name}</h4>
                  <p className="text-sm text-gray-600 mt-1 line-clamp-2">{property?.address}</p>
                  <div className="flex items-center justify-between mt-3">
                    <span className="text-sm font-medium text-green-600">
                      ${property?.currentRent?.toLocaleString()} {property?.currency}
                    </span>
                    <span className="text-xs text-gray-500">{property?.tenant?.name}</span>
                  </div>
                </div>
                {selectedProperty?.id === property?.id && (
                  <div className="flex-shrink-0">
                    <Icon name="CheckCircle" size={20} className="text-primary" />
                  </div>
                )}
              </div>
            </div>
          ))}
        </div>
      </div>

      {!selectedProperty && (
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
          <div className="flex items-start space-x-3">
            <Icon name="AlertTriangle" size={20} className="text-yellow-600 flex-shrink-0 mt-0.5" />
            <div>
              <p className="text-sm text-yellow-800">
                <strong>Selección Requerida:</strong> Debes seleccionar una propiedad para continuar con la creación del comprobante.
              </p>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default PropertySelectionStep;