import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

import Button from '../../../components/ui/Button';
import { CHILEAN_UTILITY_COMPANIES, fetchUtilityBill, generateMockUtilityData } from '../../../utils/chileanUtilities';

const ServicePaymentsStep = ({ servicePayments, selectedProperty, billingPeriod, onServicesChange }) => {
  const [loadingServices, setLoadingServices] = useState({});

  const serviceTypes = [
    { key: 'electricity', label: 'Electricidad', icon: 'Zap', color: 'yellow' },
    { key: 'water', label: 'Agua', icon: 'Droplets', color: 'blue' },
    { key: 'gas', label: 'Gas', icon: 'Flame', color: 'orange' },
    { key: 'commonExpenses', label: 'Gastos Comunes', icon: 'Building', color: 'green' }
  ];

  const handleServiceChange = (serviceKey, field, value) => {
    const updatedServices = {
      ...servicePayments,
      [serviceKey]: {
        ...servicePayments?.[serviceKey],
        [field]: value
      }
    };
    onServicesChange(updatedServices);
  };

  const handleAPIConnection = async (serviceKey) => {
    const service = servicePayments?.[serviceKey];
    if (!service?.companyId || !service?.accountNumber) {
      alert('Por favor selecciona una compañía e ingresa el número de cuenta');
      return;
    }

    setLoadingServices(prev => ({ ...prev, [serviceKey]: true }));

    try {
      // Simulate API call
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      const billData = await fetchUtilityBill(
        service?.companyId,
        service?.accountNumber,
        serviceKey,
        billingPeriod?.month || new Date()?.getMonth() + 1,
        billingPeriod?.year || new Date()?.getFullYear()
      );

      handleServiceChange(serviceKey, 'amount', billData?.amount || 0);
      handleServiceChange(serviceKey, 'connected', true);
      
      alert(`Conexión exitosa con ${billData?.company}. Monto obtenido: $${billData?.amount?.toLocaleString()}`);
      
    } catch (error) {
      console.error('Error connecting to API:', error);
      alert('Error al conectar con la API. Usando datos de prueba.');
      
      // Generate mock data on error
      const mockData = generateMockUtilityData(serviceKey, service?.companyId);
      handleServiceChange(serviceKey, 'amount', mockData?.currentBill || 0);
      handleServiceChange(serviceKey, 'connected', false);
    } finally {
      setLoadingServices(prev => ({ ...prev, [serviceKey]: false }));
    }
  };

  const handleManualOverride = (serviceKey) => {
    handleServiceChange(serviceKey, 'connected', false);
  };

  const getCompanyOptions = (serviceKey) => {
    const companies = CHILEAN_UTILITY_COMPANIES?.[serviceKey] || [];
    return companies?.map(company => ({
      value: company?.id,
      label: `${company?.name}${!company?.apiSupported ? ' (Sin API)' : ''}`,
      disabled: !company?.apiSupported
    }));
  };

  const getTotalServices = () => {
    return Object.values(servicePayments || {})?.reduce((total, service) => {
      return total + (service?.amount || 0);
    }, 0);
  };

  const getColorClasses = (color, connected) => {
    const colorMap = {
      yellow: connected ? 'bg-yellow-50 border-yellow-200' : 'bg-gray-50 border-gray-200',
      blue: connected ? 'bg-blue-50 border-blue-200' : 'bg-gray-50 border-gray-200',
      orange: connected ? 'bg-orange-50 border-orange-200' : 'bg-gray-50 border-gray-200',
      green: connected ? 'bg-green-50 border-green-200' : 'bg-gray-50 border-gray-200'
    };
    return colorMap?.[color] || 'bg-gray-50 border-gray-200';
  };

  const getIconColor = (color, connected) => {
    if (!connected) return 'text-gray-400';
    const colorMap = {
      yellow: 'text-yellow-600',
      blue: 'text-blue-600',
      orange: 'text-orange-600',
      green: 'text-green-600'
    };
    return colorMap?.[color] || 'text-gray-400';
  };

  return (
    <div className="space-y-6">
      {/* Service Integration Header */}
      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <div className="flex items-start space-x-3">
          <Icon name="Wifi" size={20} className="text-blue-600 mt-0.5" />
          <div>
            <h3 className="font-medium text-blue-900">Integración con APIs de Servicios</h3>
            <p className="text-sm text-blue-700 mt-1">
              Conecta con las compañías de servicios básicos para obtener automáticamente los montos facturados.
              También puedes ingresar los montos manualmente si prefieres.
            </p>
          </div>
        </div>
      </div>
      {/* Services Configuration */}
      <div className="space-y-6">
        {serviceTypes?.map((service) => {
          const serviceData = servicePayments?.[service?.key] || {};
          const isLoading = loadingServices?.[service?.key];
          const isConnected = serviceData?.connected;
          
          return (
            <div key={service?.key} className={`border rounded-lg p-6 ${getColorClasses(service?.color, isConnected)}`}>
              <div className="flex items-center justify-between mb-4">
                <div className="flex items-center space-x-3">
                  <div className={`w-10 h-10 rounded-lg flex items-center justify-center ${
                    isConnected 
                      ? `bg-${service?.color}-100` 
                      : 'bg-gray-100'
                  }`}>
                    <Icon name={service?.icon} size={20} className={getIconColor(service?.color, isConnected)} />
                  </div>
                  <div>
                    <h4 className="font-medium text-gray-900">{service?.label}</h4>
                    <p className="text-sm text-gray-600">
                      {isConnected ? 'Conectado automáticamente' : 'Configuración manual'}
                    </p>
                  </div>
                </div>
                
                {isConnected && (
                  <div className="flex items-center space-x-2">
                    <Icon name="CheckCircle" size={16} className="text-green-600" />
                    <span className="text-sm text-green-700 font-medium">Conectado</span>
                  </div>
                )}
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                {/* Company Selection */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Compañía
                  </label>
                  <Select
                    options={getCompanyOptions(service?.key)}
                    value={serviceData?.companyId || ''}
                    onChange={(value) => handleServiceChange(service?.key, 'companyId', value)}
                    placeholder="Seleccionar..."
                    className="w-full"
                    disabled={isConnected}
                  />
                </div>

                {/* Account Number */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Número de Cuenta
                  </label>
                  <Input
                    type="text"
                    value={serviceData?.accountNumber || ''}
                    onChange={(e) => handleServiceChange(service?.key, 'accountNumber', e?.target?.value)}
                    placeholder="Ej: 12345678"
                    className="w-full"
                    disabled={isConnected}
                  />
                </div>

                {/* Amount */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Monto ($)
                  </label>
                  <Input
                    type="number"
                    value={serviceData?.amount || 0}
                    onChange={(e) => handleServiceChange(service?.key, 'amount', parseFloat(e?.target?.value) || 0)}
                    placeholder="0"
                    className="w-full"
                    min="0"
                  />
                  {isConnected && (
                    <p className="text-xs text-gray-500 mt-1">Obtenido automáticamente</p>
                  )}
                </div>

                {/* Actions */}
                <div>
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Acciones
                  </label>
                  <div className="flex flex-col space-y-2">
                    {!isConnected ? (
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleAPIConnection(service?.key)}
                        disabled={!serviceData?.companyId || !serviceData?.accountNumber || isLoading}
                        iconName={isLoading ? 'Loader2' : 'Wifi'}
                        className={`w-full ${isLoading ? 'animate-spin' : ''}`}
                      >
                        {isLoading ? 'Conectando...' : 'Conectar API'}
                      </Button>
                    ) : (
                      <Button
                        variant="outline"
                        size="sm"
                        onClick={() => handleManualOverride(service?.key)}
                        iconName="Edit"
                        className="w-full"
                      >
                        Editar Manual
                      </Button>
                    )}
                  </div>
                </div>
              </div>
              {/* API Warning for non-supported companies */}
              {serviceData?.companyId && !isConnected && (
                <div className="mt-4">
                  {CHILEAN_UTILITY_COMPANIES?.[service?.key]?.find(c => c?.id === serviceData?.companyId)?.apiSupported ? (
                    <div className="bg-green-50 border border-green-200 rounded p-3">
                      <div className="flex items-center space-x-2">
                        <Icon name="CheckCircle" size={16} className="text-green-600" />
                        <span className="text-sm text-green-700">API disponible para conexión automática</span>
                      </div>
                    </div>
                  ) : (
                    <div className="bg-yellow-50 border border-yellow-200 rounded p-3">
                      <div className="flex items-center space-x-2">
                        <Icon name="AlertTriangle" size={16} className="text-yellow-600" />
                        <span className="text-sm text-yellow-700">Esta compañía no tiene API disponible. Ingreso manual requerido.</span>
                      </div>
                    </div>
                  )}
                </div>
              )}
            </div>
          );
        })}
      </div>
      {/* Centralized API Configuration Info */}
      <div className="bg-gray-50 rounded-lg p-6">
        <div className="flex items-start space-x-3">
          <Icon name="Settings" size={20} className="text-gray-600 mt-0.5" />
          <div className="flex-1">
            <h4 className="font-medium text-gray-900">Configuración de APIs Centralizadas</h4>
            <p className="text-sm text-gray-600 mt-1">
              Las conexiones a APIs se configuran desde el panel de administración general. 
              Desde allí puedes gestionar las credenciales y configuraciones de todas las compañías de servicios.
            </p>
            <Button
              variant="ghost"
              size="sm"
              className="mt-2"
              iconName="ExternalLink"
              onClick={() => console.log('Navigate to API settings')}
            >
              Ir a Configuración de APIs
            </Button>
          </div>
        </div>
      </div>
      {/* Services Summary */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h4 className="font-medium text-gray-900 mb-4">Resumen de Servicios</h4>
        <div className="space-y-3">
          {serviceTypes?.map((service) => {
            const serviceData = servicePayments?.[service?.key] || {};
            const amount = serviceData?.amount || 0;
            
            return (
              <div key={service?.key} className="flex justify-between items-center">
                <div className="flex items-center space-x-3">
                  <Icon name={service?.icon} size={16} className={getIconColor(service?.color, serviceData?.connected)} />
                  <span className="text-gray-600">{service?.label}:</span>
                </div>
                <div className="flex items-center space-x-2">
                  <span className="font-medium">
                    ${amount?.toLocaleString()}
                  </span>
                  {serviceData?.connected && (
                    <Icon name="Wifi" size={14} className="text-green-600" />
                  )}
                </div>
              </div>
            );
          })}
          
          <hr className="border-gray-300" />
          <div className="flex justify-between items-center text-lg">
            <span className="font-semibold text-gray-900">Total Servicios:</span>
            <span className="font-bold text-primary">
              ${getTotalServices()?.toLocaleString()}
            </span>
          </div>
        </div>
      </div>
    </div>
  );
};

export default ServicePaymentsStep;