import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import { Checkbox } from '../../../components/ui/Checkbox';
import Button from '../../../components/ui/Button';

const VoucherDetailsStep = ({ voucherDetails, selectedProperty, onDetailsChange }) => {
  const [newExpense, setNewExpense] = useState({ description: '', amount: 0 });

  const currencyOptions = [
    { value: 'CLP', label: 'Pesos Chilenos (CLP)' },
    { value: 'UF', label: 'Unidad de Fomento (UF)' }
  ];

  // Mock UF to CLP conversion rate
  const UF_TO_CLP_RATE = 37500;

  const handleInputChange = (field, value) => {
    onDetailsChange({
      ...voucherDetails,
      [field]: value
    });
  };

  const handleIPCAdjustmentToggle = (checked) => {
    onDetailsChange({
      ...voucherDetails,
      ipcAdjustment: checked,
      ipcPercentage: checked ? voucherDetails?.ipcPercentage || 3.5 : 0
    });
  };

  const addExpense = () => {
    if (newExpense?.description && newExpense?.amount > 0) {
      const updatedExpenses = [...(voucherDetails?.additionalExpenses || []), {
        id: Date.now()?.toString(),
        ...newExpense
      }];
      
      onDetailsChange({
        ...voucherDetails,
        additionalExpenses: updatedExpenses
      });
      
      setNewExpense({ description: '', amount: 0 });
    }
  };

  const removeExpense = (expenseId) => {
    const updatedExpenses = (voucherDetails?.additionalExpenses || [])?.filter(
      expense => expense?.id !== expenseId
    );
    
    onDetailsChange({
      ...voucherDetails,
      additionalExpenses: updatedExpenses
    });
  };

  const convertAmount = (amount, fromCurrency, toCurrency) => {
    if (fromCurrency === toCurrency) return amount;
    
    if (fromCurrency === 'UF' && toCurrency === 'CLP') {
      return amount * UF_TO_CLP_RATE;
    } else if (fromCurrency === 'CLP' && toCurrency === 'UF') {
      return amount / UF_TO_CLP_RATE;
    }
    
    return amount;
  };

  const formatCurrency = (amount, currency) => {
    if (currency === 'UF') {
      return `${amount?.toFixed(2)} UF`;
    }
    return `$${amount?.toLocaleString()} CLP`;
  };

  const calculateIPCAmount = () => {
    if (!voucherDetails?.ipcAdjustment) return 0;
    return (voucherDetails?.rentAmount || 0) * (voucherDetails?.ipcPercentage || 0) / 100;
  };

  return (
    <div className="space-y-6">
      {/* Base Rent Configuration */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h3 className="text-lg font-medium text-gray-900 mb-4">Configuración del Arriendo Base</h3>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Monto del Arriendo *
              </label>
              <Input
                type="number"
                value={voucherDetails?.rentAmount || selectedProperty?.currentRent || 0}
                onChange={(e) => handleInputChange('rentAmount', parseFloat(e?.target?.value) || 0)}
                placeholder="Ingresa el monto del arriendo"
                className="w-full"
                min="0"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Tipo de Moneda
              </label>
              <Select
                options={currencyOptions}
                value={voucherDetails?.currency || 'CLP'}
                onChange={(value) => handleInputChange('currency', value)}
                className="w-full"
              />
            </div>
          </div>

          <div className="space-y-4">
            {/* Currency Conversion Display */}
            {voucherDetails?.currency && voucherDetails?.rentAmount > 0 && (
              <div className="bg-blue-50 rounded-lg p-4">
                <h4 className="font-medium text-blue-900 mb-2">Conversión de Moneda</h4>
                <div className="space-y-2 text-sm">
                  <div className="flex justify-between">
                    <span className="text-blue-700">Monto en {voucherDetails?.currency}:</span>
                    <span className="font-medium">
                      {formatCurrency(voucherDetails?.rentAmount, voucherDetails?.currency)}
                    </span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-blue-700">
                      Equivalente en {voucherDetails?.currency === 'CLP' ? 'UF' : 'CLP'}:
                    </span>
                    <span className="font-medium">
                      {formatCurrency(
                        convertAmount(
                          voucherDetails?.rentAmount,
                          voucherDetails?.currency,
                          voucherDetails?.currency === 'CLP' ? 'UF' : 'CLP'
                        ),
                        voucherDetails?.currency === 'CLP' ? 'UF' : 'CLP'
                      )}
                    </span>
                  </div>
                  <div className="text-xs text-blue-600 pt-2 border-t border-blue-200">
                    Tasa de conversión: 1 UF = ${UF_TO_CLP_RATE?.toLocaleString()} CLP
                  </div>
                </div>
              </div>
            )}
          </div>
        </div>
      </div>
      {/* IPC Adjustment */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <div className="flex items-start space-x-3 mb-4">
          <Checkbox
            checked={voucherDetails?.ipcAdjustment || false}
            onChange={handleIPCAdjustmentToggle}
            className="mt-1"
          />
          <div className="flex-1">
            <h3 className="text-lg font-medium text-gray-900">Reajuste Automático por IPC</h3>
            <p className="text-sm text-gray-600 mt-1">
              Aplica un reajuste automático basado en el Índice de Precios al Consumidor
            </p>
          </div>
        </div>

        {voucherDetails?.ipcAdjustment && (
          <div className="mt-4 space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Porcentaje de IPC (%)
                </label>
                <Input
                  type="number"
                  value={voucherDetails?.ipcPercentage || 0}
                  onChange={(e) => handleInputChange('ipcPercentage', parseFloat(e?.target?.value) || 0)}
                  placeholder="Ej: 3.5"
                  className="w-full"
                  step="0.1"
                  min="0"
                  max="100"
                />
              </div>

              <div className="bg-green-50 rounded-lg p-4">
                <div className="flex items-center space-x-2 mb-2">
                  <Icon name="TrendingUp" size={16} className="text-green-600" />
                  <span className="text-sm font-medium text-green-900">Monto del Reajuste</span>
                </div>
                <p className="text-lg font-semibold text-green-700">
                  {formatCurrency(calculateIPCAmount(), voucherDetails?.currency || 'CLP')}
                </p>
                <p className="text-xs text-green-600 mt-1">
                  {voucherDetails?.ipcPercentage || 0}% de {formatCurrency(voucherDetails?.rentAmount || 0, voucherDetails?.currency || 'CLP')}
                </p>
              </div>
            </div>
          </div>
        )}
      </div>
      {/* Additional Expenses */}
      <div className="bg-white border border-gray-200 rounded-lg p-6">
        <h3 className="text-lg font-medium text-gray-900 mb-4">Gastos Adicionales</h3>
        
        {/* Add New Expense */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
          <div className="md:col-span-2">
            <Input
              type="text"
              value={newExpense?.description}
              onChange={(e) => setNewExpense({...newExpense, description: e?.target?.value})}
              placeholder="Descripción del gasto (ej: Reparación ventana)"
              className="w-full"
            />
          </div>
          <div className="flex space-x-2">
            <Input
              type="number"
              value={newExpense?.amount}
              onChange={(e) => setNewExpense({...newExpense, amount: parseFloat(e?.target?.value) || 0})}
              placeholder="Monto"
              className="flex-1"
              min="0"
            />
            <Button
              onClick={addExpense}
              disabled={!newExpense?.description || newExpense?.amount <= 0}
              iconName="Plus"
              size="sm"
            >
              Agregar
            </Button>
          </div>
        </div>

        {/* Expenses List */}
        {voucherDetails?.additionalExpenses?.length > 0 ? (
          <div className="space-y-2">
            {voucherDetails?.additionalExpenses?.map((expense) => (
              <div key={expense?.id} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                <div className="flex-1">
                  <span className="font-medium text-gray-900">{expense?.description}</span>
                </div>
                <div className="flex items-center space-x-3">
                  <span className="font-medium text-green-600">
                    {formatCurrency(expense?.amount, voucherDetails?.currency || 'CLP')}
                  </span>
                  <Button
                    variant="ghost"
                    size="sm"
                    onClick={() => removeExpense(expense?.id)}
                    iconName="Trash2"
                    className="text-red-600 hover:text-red-700 hover:bg-red-50"
                  />
                </div>
              </div>
            ))}
          </div>
        ) : (
          <div className="text-center py-6 text-gray-500">
            <Icon name="Plus" size={32} className="mx-auto mb-2 text-gray-300" />
            <p>No hay gastos adicionales agregados</p>
            <p className="text-xs mt-1">Usa el formulario de arriba para agregar gastos extras</p>
          </div>
        )}
      </div>
      {/* Summary Section */}
      <div className="bg-gray-50 rounded-lg p-6">
        <h4 className="font-medium text-gray-900 mb-4">Resumen de Detalles</h4>
        <div className="space-y-3">
          <div className="flex justify-between items-center">
            <span className="text-gray-600">Arriendo Base:</span>
            <span className="font-medium">
              {formatCurrency(voucherDetails?.rentAmount || 0, voucherDetails?.currency || 'CLP')}
            </span>
          </div>
          
          {voucherDetails?.ipcAdjustment && (
            <div className="flex justify-between items-center">
              <span className="text-gray-600">Reajuste IPC ({voucherDetails?.ipcPercentage}%):</span>
              <span className="font-medium text-green-600">
                +{formatCurrency(calculateIPCAmount(), voucherDetails?.currency || 'CLP')}
              </span>
            </div>
          )}
          
          {voucherDetails?.additionalExpenses?.length > 0 && (
            <div className="flex justify-between items-center">
              <span className="text-gray-600">Gastos Adicionales:</span>
              <span className="font-medium text-blue-600">
                +{formatCurrency(
                  voucherDetails?.additionalExpenses?.reduce((sum, expense) => sum + (expense?.amount || 0), 0),
                  voucherDetails?.currency || 'CLP'
                )}
              </span>
            </div>
          )}
          
          <hr className="border-gray-300" />
          <div className="flex justify-between items-center text-lg">
            <span className="font-semibold text-gray-900">Subtotal:</span>
            <span className="font-bold text-primary">
              {formatCurrency(
                (voucherDetails?.rentAmount || 0) + 
                calculateIPCAmount() + 
                (voucherDetails?.additionalExpenses?.reduce((sum, expense) => sum + (expense?.amount || 0), 0) || 0),
                voucherDetails?.currency || 'CLP'
              )}
            </span>
          </div>
        </div>
      </div>
      {/* Validation */}
      {!voucherDetails?.rentAmount && (
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
          <div className="flex items-start space-x-3">
            <Icon name="AlertTriangle" size={20} className="text-yellow-600 flex-shrink-0 mt-0.5" />
            <div>
              <p className="text-sm text-yellow-800">
                <strong>Monto Requerido:</strong> Ingresa el monto del arriendo para continuar.
              </p>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default VoucherDetailsStep;