import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import Icon from '../../components/AppIcon';
import Button from '../../components/ui/Button';



import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';

// Import page components
import PropertySelectionStep from './components/PropertySelectionStep';
import BillingPeriodStep from './components/BillingPeriodStep';
import VoucherDetailsStep from './components/VoucherDetailsStep';
import ServicePaymentsStep from './components/ServicePaymentsStep';
import DiscountApplicationStep from './components/DiscountApplicationStep';
import PreviewAndGenerateStep from './components/PreviewAndGenerateStep';

const IndividualVoucherCreation = () => {
  const navigate = useNavigate();
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [currentStep, setCurrentStep] = useState(1);
  const [isMobile, setIsMobile] = useState(false);

  // Form state
  const [voucherData, setVoucherData] = useState({
    property: null,
    tenant: null,
    billingPeriod: {
      startDate: '',
      endDate: '',
      month: new Date()?.getMonth() + 1,
      year: new Date()?.getFullYear()
    },
    voucherDetails: {
      rentAmount: 0,
      currency: 'CLP',
      ipcAdjustment: false,
      ipcPercentage: 0,
      additionalExpenses: []
    },
    servicePayments: {
      electricity: { amount: 0, connected: false, companyId: '', accountNumber: '' },
      water: { amount: 0, connected: false, companyId: '', accountNumber: '' },
      gas: { amount: 0, connected: false, companyId: '', accountNumber: '' },
      commonExpenses: { amount: 0, connected: false, companyId: '', accountNumber: '' }
    },
    discounts: {
      type: 'percentage',
      amount: 0,
      reason: ''
    },
    totals: {
      subtotal: 0,
      totalServices: 0,
      totalDiscounts: 0,
      finalTotal: 0
    }
  });

  // Mock properties data
  const mockProperties = [
    {
      id: 'prop-001',
      name: 'Departamento Las Condes #301',
      address: 'Av. Apoquindo 3000, Las Condes',
      tenant: {
        id: 'tenant-001',
        name: 'María González Pérez',
        email: 'maria.gonzalez@email.com',
        phone: '+56 9 1234 5678'
      },
      currentRent: 850000,
      currency: 'CLP'
    },
    {
      id: 'prop-002',
      name: 'Casa Providencia #45',
      address: 'Manuel Montt 1245, Providencia',
      tenant: {
        id: 'tenant-002',
        name: 'Carlos Rodríguez Silva',
        email: 'carlos.rodriguez@email.com',
        phone: '+56 9 8765 4321'
      },
      currentRent: 1200000,
      currency: 'CLP'
    }
  ];

  const steps = [
    { id: 1, title: 'Selección de Propiedad', component: 'property' },
    { id: 2, title: 'Período de Facturación', component: 'period' },
    { id: 3, title: 'Detalles del Comprobante', component: 'details' },
    { id: 4, title: 'Pagos de Servicios', component: 'services' },
    { id: 5, title: 'Aplicar Descuentos', component: 'discounts' },
    { id: 6, title: 'Vista Previa y Generar', component: 'preview' }
  ];

  useEffect(() => {
    const checkMobile = () => {
      const mobile = window.innerWidth < 1024;
      setIsMobile(mobile);
      if (!mobile) {
        setIsSidebarOpen(false);
      }
    };

    checkMobile();
    window.addEventListener('resize', checkMobile);
    return () => window.removeEventListener('resize', checkMobile);
  }, []);

  useEffect(() => {
    calculateTotals();
  }, [voucherData?.voucherDetails, voucherData?.servicePayments, voucherData?.discounts]);

  const calculateTotals = () => {
    const rentAmount = voucherData?.voucherDetails?.rentAmount || 0;
    const ipcAmount = voucherData?.voucherDetails?.ipcAdjustment 
      ? (rentAmount * (voucherData?.voucherDetails?.ipcPercentage || 0)) / 100 
      : 0;
    
    const additionalExpensesTotal = voucherData?.voucherDetails?.additionalExpenses?.reduce((sum, expense) => sum + (expense?.amount || 0), 0) || 0;
    
    const servicesTotal = Object.values(voucherData?.servicePayments || {})?.reduce((sum, service) => sum + (service?.amount || 0), 0) || 0;
    
    const subtotal = rentAmount + ipcAmount + additionalExpensesTotal + servicesTotal;
    
    let discountAmount = 0;
    if (voucherData?.discounts?.type === 'percentage') {
      discountAmount = (subtotal * (voucherData?.discounts?.amount || 0)) / 100;
    } else {
      discountAmount = voucherData?.discounts?.amount || 0;
    }
    
    const finalTotal = subtotal - discountAmount;

    setVoucherData(prev => ({
      ...prev,
      totals: {
        subtotal,
        totalServices: servicesTotal,
        totalDiscounts: discountAmount,
        finalTotal
      }
    }));
  };

  const handleMenuToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleNext = () => {
    if (currentStep < steps?.length) {
      setCurrentStep(currentStep + 1);
    }
  };

  const handlePrevious = () => {
    if (currentStep > 1) {
      setCurrentStep(currentStep - 1);
    }
  };

  const handleStepChange = (stepNumber) => {
    setCurrentStep(stepNumber);
  };

  const handleDataChange = (section, data) => {
    setVoucherData(prev => ({
      ...prev,
      [section]: { ...prev?.[section], ...data }
    }));
  };

  const handleGenerateVoucher = async () => {
    try {
      console.log('Generating individual voucher:', voucherData);
      
      // Simulate voucher generation
      const voucherNumber = `COMP-${new Date()?.getFullYear()}-${String(Math.floor(Math.random() * 9999))?.padStart(4, '0')}`;
      
      // Show success message
      alert(`Comprobante ${voucherNumber} generado exitosamente`);
      
      // Navigate back to vouchers management
      navigate('/vouchers-management');
      
    } catch (error) {
      console.error('Error generating voucher:', error);
      alert('Error al generar el comprobante. Inténtelo nuevamente.');
    }
  };

  const renderCurrentStep = () => {
    switch (currentStep) {
      case 1:
        return (
          <PropertySelectionStep
            properties={mockProperties}
            selectedProperty={voucherData?.property}
            onPropertySelect={(property) => handleDataChange('property', property)}
            onTenantSelect={(tenant) => handleDataChange('tenant', tenant)}
          />
        );
      case 2:
        return (
          <BillingPeriodStep
            billingPeriod={voucherData?.billingPeriod}
            onPeriodChange={(period) => handleDataChange('billingPeriod', period)}
          />
        );
      case 3:
        return (
          <VoucherDetailsStep
            voucherDetails={voucherData?.voucherDetails}
            selectedProperty={voucherData?.property}
            onDetailsChange={(details) => handleDataChange('voucherDetails', details)}
          />
        );
      case 4:
        return (
          <ServicePaymentsStep
            servicePayments={voucherData?.servicePayments}
            selectedProperty={voucherData?.property}
            billingPeriod={voucherData?.billingPeriod}
            onServicesChange={(services) => handleDataChange('servicePayments', services)}
          />
        );
      case 5:
        return (
          <DiscountApplicationStep
            discounts={voucherData?.discounts}
            totals={voucherData?.totals}
            onDiscountsChange={(discounts) => handleDataChange('discounts', discounts)}
          />
        );
      case 6:
        return (
          <PreviewAndGenerateStep
            voucherData={voucherData}
            onGenerate={handleGenerateVoucher}
          />
        );
      default:
        return null;
    }
  };

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleMenuToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={() => setIsSidebarOpen(false)} />
      <main className={`transition-all duration-300 ${isSidebarOpen && isMobile ? 'lg:ml-60' : 'lg:ml-60'}`}>
        <div className="pt-16">
          <div className="flex">
            {/* Step Wizard Sidebar */}
            <div className="w-80 bg-white border-r border-gray-200 p-6 hidden lg:block">
              <div className="mb-6">
                <h2 className="text-lg font-semibold text-gray-900 mb-2">Crear Comprobante Individual</h2>
                <p className="text-sm text-gray-600">Sigue los pasos para generar un comprobante personalizado</p>
              </div>
              
              <div className="space-y-4">
                {steps?.map((step) => (
                  <div
                    key={step?.id}
                    className={`flex items-center space-x-3 p-3 rounded-lg cursor-pointer transition-colors ${
                      currentStep === step?.id
                        ? 'bg-primary text-white'
                        : currentStep > step?.id
                          ? 'bg-green-50 text-green-700 hover:bg-green-100' :'text-gray-500 hover:bg-gray-50'
                    }`}
                    onClick={() => handleStepChange(step?.id)}
                  >
                    <div
                      className={`w-8 h-8 rounded-full flex items-center justify-center text-sm font-medium ${
                        currentStep === step?.id
                          ? 'bg-white text-primary'
                          : currentStep > step?.id
                            ? 'bg-green-500 text-white' :'bg-gray-200 text-gray-500'
                      }`}
                    >
                      {currentStep > step?.id ? (
                        <Icon name="Check" size={16} />
                      ) : (
                        step?.id
                      )}
                    </div>
                    <span className="font-medium">{step?.title}</span>
                  </div>
                ))}
              </div>

              {/* Progress Summary */}
              <div className="mt-8 p-4 bg-gray-50 rounded-lg">
                <h3 className="text-sm font-medium text-gray-900 mb-2">Progreso</h3>
                <div className="flex items-center space-x-2">
                  <div className="flex-1 bg-gray-200 rounded-full h-2">
                    <div
                      className="bg-primary h-2 rounded-full transition-all duration-300"
                      style={{ width: `${(currentStep / steps?.length) * 100}%` }}
                    />
                  </div>
                  <span className="text-sm text-gray-600">
                    {currentStep}/{steps?.length}
                  </span>
                </div>
              </div>
            </div>

            {/* Main Content */}
            <div className="flex-1 p-6">
              {/* Breadcrumbs */}
              <NavigationBreadcrumbs />

              {/* Page Header */}
              <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between mb-8">
                <div className="flex items-center space-x-4 mb-4 lg:mb-0">
                  <div className="w-12 h-12 bg-primary text-white rounded-lg flex items-center justify-center">
                    <Icon name="FileText" size={24} />
                  </div>
                  <div>
                    <h1 className="text-2xl font-bold text-gray-900">Crear Comprobante Individual</h1>
                    <p className="text-gray-600">Genera comprobantes personalizados por propiedad con total control</p>
                  </div>
                </div>

                {/* Mobile Steps Indicator */}
                <div className="lg:hidden">
                  <div className="flex items-center space-x-2 text-sm text-gray-600">
                    <span>Paso {currentStep} de {steps?.length}</span>
                    <div className="w-20 bg-gray-200 rounded-full h-2 ml-2">
                      <div
                        className="bg-primary h-2 rounded-full transition-all duration-300"
                        style={{ width: `${(currentStep / steps?.length) * 100}%` }}
                      />
                    </div>
                  </div>
                </div>
              </div>

              {/* Current Step Content */}
              <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-8">
                <div className="mb-6">
                  <h2 className="text-xl font-semibold text-gray-900">
                    {steps?.[currentStep - 1]?.title}
                  </h2>
                  <div className="mt-2 h-1 bg-gray-100 rounded">
                    <div
                      className="h-1 bg-primary rounded transition-all duration-300"
                      style={{ width: `${(currentStep / steps?.length) * 100}%` }}
                    />
                  </div>
                </div>

                {renderCurrentStep()}
              </div>

              {/* Navigation Buttons */}
              <div className="flex items-center justify-between">
                <Button
                  variant="outline"
                  onClick={handlePrevious}
                  disabled={currentStep === 1}
                  iconName="ChevronLeft"
                  iconPosition="left"
                >
                  Anterior
                </Button>

                <div className="flex items-center space-x-2">
                  <span className="text-sm text-gray-600">
                    {currentStep} de {steps?.length}
                  </span>
                </div>

                {currentStep < steps?.length ? (
                  <Button
                    onClick={handleNext}
                    iconName="ChevronRight"
                    iconPosition="right"
                  >
                    Siguiente
                  </Button>
                ) : (
                  <Button
                    onClick={handleGenerateVoucher}
                    iconName="FileText"
                    iconPosition="left"
                    className="bg-green-600 hover:bg-green-700"
                  >
                    Generar Comprobante
                  </Button>
                )}
              </div>
            </div>
          </div>
        </div>
      </main>
      <QuickActionFloatingButton />
    </div>
  );
};

export default IndividualVoucherCreation;