import React, { useState } from 'react';

import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const AddOwnerModal = ({ isOpen, onClose, onSave }) => {
  const [formData, setFormData] = useState({
    name: '',
    rut: '',
    email: '',
    phone: '',
    address: '',
    city: '',
    region: '',
    bankName: '',
    accountType: '',
    accountNumber: '',
    commissionRate: '',
    notes: ''
  });

  const [errors, setErrors] = useState({});
  const [isLoading, setIsLoading] = useState(false);

  const regionOptions = [
    { value: 'metropolitana', label: 'Región Metropolitana' },
    { value: 'valparaiso', label: 'Región de Valparaíso' },
    { value: 'biobio', label: 'Región del Biobío' },
    { value: 'araucania', label: 'Región de La Araucanía' },
    { value: 'los-lagos', label: 'Región de Los Lagos' }
  ];

  const bankOptions = [
    { value: 'banco-chile', label: 'Banco de Chile' },
    { value: 'banco-estado', label: 'BancoEstado' },
    { value: 'santander', label: 'Banco Santander' },
    { value: 'bci', label: 'Banco BCI' },
    { value: 'scotiabank', label: 'Scotiabank' }
  ];

  const accountTypeOptions = [
    { value: 'corriente', label: 'Cuenta Corriente' },
    { value: 'ahorro', label: 'Cuenta de Ahorro' },
    { value: 'vista', label: 'Cuenta Vista' }
  ];

  const handleInputChange = (field, value) => {
    setFormData(prev => ({ ...prev, [field]: value }));
    if (errors?.[field]) {
      setErrors(prev => ({ ...prev, [field]: '' }));
    }
  };

  const validateForm = () => {
    const newErrors = {};

    if (!formData?.name?.trim()) newErrors.name = 'El nombre es requerido';
    if (!formData?.rut?.trim()) newErrors.rut = 'El RUT es requerido';
    if (!formData?.email?.trim()) newErrors.email = 'El email es requerido';
    if (!formData?.phone?.trim()) newErrors.phone = 'El teléfono es requerido';
    if (!formData?.commissionRate) newErrors.commissionRate = 'La tasa de comisión es requerida';

    // Email validation
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (formData?.email && !emailRegex?.test(formData?.email)) {
      newErrors.email = 'Email inválido';
    }

    // RUT validation (basic)
    if (formData?.rut && !/^\d{7,8}-[\dkK]$/?.test(formData?.rut)) {
      newErrors.rut = 'Formato de RUT inválido (ej: 12345678-9)';
    }

    // Commission rate validation
    if (formData?.commissionRate && (isNaN(formData?.commissionRate) || formData?.commissionRate < 0 || formData?.commissionRate > 100)) {
      newErrors.commissionRate = 'La comisión debe ser un número entre 0 y 100';
    }

    setErrors(newErrors);
    return Object.keys(newErrors)?.length === 0;
  };

  const handleSubmit = async (e) => {
    e?.preventDefault();
    
    if (!validateForm()) return;

    setIsLoading(true);
    try {
      await onSave(formData);
      handleClose();
    } catch (error) {
      console.error('Error saving owner:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleClose = () => {
    setFormData({
      name: '',
      rut: '',
      email: '',
      phone: '',
      address: '',
      city: '',
      region: '',
      bankName: '',
      accountType: '',
      accountNumber: '',
      commissionRate: '',
      notes: ''
    });
    setErrors({});
    onClose();
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-1000 p-4">
      <div className="bg-white rounded-lg shadow-xl w-full max-w-2xl max-h-[90vh] overflow-hidden">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <h2 className="text-xl font-semibold text-gray-900">Agregar Nuevo Propietario</h2>
          <Button
            variant="ghost"
            size="sm"
            iconName="X"
            onClick={handleClose}
          />
        </div>

        {/* Form */}
        <form onSubmit={handleSubmit} className="overflow-y-auto max-h-[calc(90vh-140px)]">
          <div className="p-6 space-y-6">
            {/* Personal Information */}
            <div>
              <h3 className="text-lg font-medium text-gray-900 mb-4">Información Personal</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <Input
                  label="Nombre Completo"
                  type="text"
                  placeholder="Ingrese el nombre completo"
                  value={formData?.name}
                  onChange={(e) => handleInputChange('name', e?.target?.value)}
                  error={errors?.name}
                  required
                />
                <Input
                  label="RUT"
                  type="text"
                  placeholder="12345678-9"
                  value={formData?.rut}
                  onChange={(e) => handleInputChange('rut', e?.target?.value)}
                  error={errors?.rut}
                  required
                />
                <Input
                  label="Email"
                  type="email"
                  placeholder="propietario@email.com"
                  value={formData?.email}
                  onChange={(e) => handleInputChange('email', e?.target?.value)}
                  error={errors?.email}
                  required
                />
                <Input
                  label="Teléfono"
                  type="tel"
                  placeholder="+56 9 1234 5678"
                  value={formData?.phone}
                  onChange={(e) => handleInputChange('phone', e?.target?.value)}
                  error={errors?.phone}
                  required
                />
              </div>
            </div>

            {/* Address Information */}
            <div>
              <h3 className="text-lg font-medium text-gray-900 mb-4">Dirección</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="md:col-span-2">
                  <Input
                    label="Dirección"
                    type="text"
                    placeholder="Calle, número, departamento"
                    value={formData?.address}
                    onChange={(e) => handleInputChange('address', e?.target?.value)}
                  />
                </div>
                <Input
                  label="Ciudad"
                  type="text"
                  placeholder="Santiago"
                  value={formData?.city}
                  onChange={(e) => handleInputChange('city', e?.target?.value)}
                />
                <Select
                  label="Región"
                  placeholder="Seleccione una región"
                  options={regionOptions}
                  value={formData?.region}
                  onChange={(value) => handleInputChange('region', value)}
                />
              </div>
            </div>

            {/* Banking Information */}
            <div>
              <h3 className="text-lg font-medium text-gray-900 mb-4">Información Bancaria</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <Select
                  label="Banco"
                  placeholder="Seleccione un banco"
                  options={bankOptions}
                  value={formData?.bankName}
                  onChange={(value) => handleInputChange('bankName', value)}
                />
                <Select
                  label="Tipo de Cuenta"
                  placeholder="Seleccione tipo de cuenta"
                  options={accountTypeOptions}
                  value={formData?.accountType}
                  onChange={(value) => handleInputChange('accountType', value)}
                />
                <Input
                  label="Número de Cuenta"
                  type="text"
                  placeholder="123456789"
                  value={formData?.accountNumber}
                  onChange={(e) => handleInputChange('accountNumber', e?.target?.value)}
                />
                <Input
                  label="Tasa de Comisión (%)"
                  type="number"
                  placeholder="8.5"
                  min="0"
                  max="100"
                  step="0.1"
                  value={formData?.commissionRate}
                  onChange={(e) => handleInputChange('commissionRate', e?.target?.value)}
                  error={errors?.commissionRate}
                  required
                />
              </div>
            </div>

            {/* Notes */}
            <div>
              <Input
                label="Notas Adicionales"
                type="text"
                placeholder="Información adicional sobre el propietario"
                value={formData?.notes}
                onChange={(e) => handleInputChange('notes', e?.target?.value)}
                description="Opcional: Cualquier información relevante"
              />
            </div>
          </div>

          {/* Footer */}
          <div className="flex items-center justify-end space-x-3 p-6 border-t border-gray-200 bg-gray-50">
            <Button
              type="button"
              variant="outline"
              onClick={handleClose}
              disabled={isLoading}
            >
              Cancelar
            </Button>
            <Button
              type="submit"
              variant="default"
              loading={isLoading}
              iconName="Save"
            >
              Guardar Propietario
            </Button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default AddOwnerModal;