import React from 'react';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart, Pie, Cell } from 'recharts';
import Icon from '../../../components/AppIcon';

const CommissionDashboard = ({ commissionData }) => {
  const monthlyData = [
    { month: 'Ene', amount: 2450000 },
    { month: 'Feb', amount: 2680000 },
    { month: 'Mar', amount: 2890000 },
    { month: 'Abr', amount: 3120000 },
    { month: 'May', amount: 2950000 },
    { month: 'Jun', amount: 3340000 }
  ];

  const statusData = [
    { name: 'Pagadas', value: 65, color: '#38b000' },
    { name: 'Pendientes', value: 25, color: '#ff9e00' },
    { name: 'Vencidas', value: 10, color: '#ff006e' }
  ];

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  const stats = [
    {
      title: 'Comisiones del Mes',
      value: formatCurrency(3340000),
      change: '+12.5%',
      changeType: 'positive',
      icon: 'DollarSign',
      color: 'bg-success'
    },
    {
      title: 'Comisiones Pendientes',
      value: formatCurrency(890000),
      change: '-5.2%',
      changeType: 'negative',
      icon: 'Clock',
      color: 'bg-warning'
    },
    {
      title: 'Propietarios Activos',
      value: '24',
      change: '+2',
      changeType: 'positive',
      icon: 'Users',
      color: 'bg-primary'
    },
    {
      title: 'Tasa Promedio',
      value: '8.5%',
      change: '+0.3%',
      changeType: 'positive',
      icon: 'Percent',
      color: 'bg-secondary'
    }
  ];

  return (
    <div className="space-y-6 mb-8">
      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {stats?.map((stat, index) => (
          <div key={index} className="bg-white rounded-lg border border-gray-200 p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm font-medium text-gray-600">{stat?.title}</p>
                <p className="text-2xl font-bold text-gray-900 mt-2">{stat?.value}</p>
                <div className="flex items-center mt-2">
                  <span className={`text-sm font-medium ${
                    stat?.changeType === 'positive' ? 'text-success' : 'text-error'
                  }`}>
                    {stat?.change}
                  </span>
                  <span className="text-sm text-gray-500 ml-1">vs mes anterior</span>
                </div>
              </div>
              <div className={`w-12 h-12 ${stat?.color} rounded-lg flex items-center justify-center`}>
                <Icon name={stat?.icon} size={24} className="text-white" />
              </div>
            </div>
          </div>
        ))}
      </div>
      {/* Charts */}
      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Monthly Commission Chart */}
        <div className="lg:col-span-2 bg-white rounded-lg border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-medium text-gray-900">Comisiones Mensuales</h3>
            <div className="flex items-center space-x-2 text-sm text-gray-500">
              <Icon name="TrendingUp" size={16} />
              <span>Últimos 6 meses</span>
            </div>
          </div>
          <div className="h-64">
            <ResponsiveContainer width="100%" height="100%">
              <BarChart data={monthlyData}>
                <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
                <XAxis 
                  dataKey="month" 
                  axisLine={false}
                  tickLine={false}
                  tick={{ fontSize: 12, fill: '#6b7280' }}
                />
                <YAxis 
                  axisLine={false}
                  tickLine={false}
                  tick={{ fontSize: 12, fill: '#6b7280' }}
                  tickFormatter={(value) => `$${(value / 1000000)?.toFixed(1)}M`}
                />
                <Tooltip 
                  formatter={(value) => [formatCurrency(value), 'Comisión']}
                  labelStyle={{ color: '#374151' }}
                  contentStyle={{ 
                    backgroundColor: 'white', 
                    border: '1px solid #e5e7eb',
                    borderRadius: '8px',
                    boxShadow: '0 4px 6px -1px rgba(0, 0, 0, 0.1)'
                  }}
                />
                <Bar 
                  dataKey="amount" 
                  fill="#3a86ff" 
                  radius={[4, 4, 0, 0]}
                />
              </BarChart>
            </ResponsiveContainer>
          </div>
        </div>

        {/* Commission Status Pie Chart */}
        <div className="bg-white rounded-lg border border-gray-200 p-6">
          <div className="flex items-center justify-between mb-6">
            <h3 className="text-lg font-medium text-gray-900">Estado Comisiones</h3>
            <Icon name="PieChart" size={16} className="text-gray-500" />
          </div>
          <div className="h-48">
            <ResponsiveContainer width="100%" height="100%">
              <PieChart>
                <Pie
                  data={statusData}
                  cx="50%"
                  cy="50%"
                  innerRadius={40}
                  outerRadius={80}
                  paddingAngle={5}
                  dataKey="value"
                >
                  {statusData?.map((entry, index) => (
                    <Cell key={`cell-${index}`} fill={entry?.color} />
                  ))}
                </Pie>
                <Tooltip 
                  formatter={(value) => [`${value}%`, 'Porcentaje']}
                  contentStyle={{ 
                    backgroundColor: 'white', 
                    border: '1px solid #e5e7eb',
                    borderRadius: '8px',
                    boxShadow: '0 4px 6px -1px rgba(0, 0, 0, 0.1)'
                  }}
                />
              </PieChart>
            </ResponsiveContainer>
          </div>
          <div className="space-y-2 mt-4">
            {statusData?.map((item, index) => (
              <div key={index} className="flex items-center justify-between">
                <div className="flex items-center space-x-2">
                  <div 
                    className="w-3 h-3 rounded-full"
                    style={{ backgroundColor: item?.color }}
                  />
                  <span className="text-sm text-gray-600">{item?.name}</span>
                </div>
                <span className="text-sm font-medium text-gray-900">{item?.value}%</span>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
};

export default CommissionDashboard;