import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import Button from '../../../components/ui/Button';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const OwnerProfileModal = ({ isOpen, onClose, owner }) => {
  const [activeTab, setActiveTab] = useState('overview');

  if (!isOpen || !owner) return null;

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  const formatDate = (date) => {
    return new Date(date)?.toLocaleDateString('es-CL', {
      day: '2-digit',
      month: '2-digit',
      year: 'numeric'
    });
  };

  const mockProperties = [
    {
      id: 1,
      address: 'Av. Providencia 1234, Providencia',
      type: 'Departamento',
      bedrooms: 2,
      bathrooms: 2,
      rentAmount: 850000,
      status: 'occupied',
      tenant: 'María González'
    },
    {
      id: 2,
      address: 'Los Leones 567, Las Condes',
      type: 'Casa',
      bedrooms: 3,
      bathrooms: 3,
      rentAmount: 1200000,
      status: 'vacant',
      tenant: null
    }
  ];

  const mockCommissionHistory = [
    {
      id: 1,
      month: 'Junio 2024',
      amount: 145000,
      status: 'paid',
      paymentDate: '2024-07-05',
      properties: 2
    },
    {
      id: 2,
      month: 'Mayo 2024',
      amount: 138000,
      status: 'paid',
      paymentDate: '2024-06-05',
      properties: 2
    },
    {
      id: 3,
      month: 'Abril 2024',
      amount: 142000,
      status: 'paid',
      paymentDate: '2024-05-05',
      properties: 2
    }
  ];

  const tabs = [
    { key: 'overview', label: 'Resumen', icon: 'User' },
    { key: 'properties', label: 'Propiedades', icon: 'Building2' },
    { key: 'commissions', label: 'Comisiones', icon: 'DollarSign' },
    { key: 'documents', label: 'Documentos', icon: 'FileText' }
  ];

  const renderOverview = () => (
    <div className="space-y-6">
      {/* Personal Information */}
      <div className="bg-gray-50 rounded-lg p-6">
        <h4 className="text-lg font-medium text-gray-900 mb-4">Información Personal</h4>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <p className="text-sm text-gray-500">Nombre Completo</p>
            <p className="text-sm font-medium text-gray-900">{owner?.name}</p>
          </div>
          <div>
            <p className="text-sm text-gray-500">RUT</p>
            <p className="text-sm font-medium text-gray-900">{owner?.rut}</p>
          </div>
          <div>
            <p className="text-sm text-gray-500">Email</p>
            <p className="text-sm font-medium text-gray-900">{owner?.email}</p>
          </div>
          <div>
            <p className="text-sm text-gray-500">Teléfono</p>
            <p className="text-sm font-medium text-gray-900">{owner?.phone}</p>
          </div>
        </div>
      </div>

      {/* Portfolio Summary */}
      <div className="bg-gray-50 rounded-lg p-6">
        <h4 className="text-lg font-medium text-gray-900 mb-4">Resumen de Cartera</h4>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="text-center">
            <p className="text-2xl font-bold text-primary">{owner?.propertiesCount}</p>
            <p className="text-sm text-gray-500">Propiedades</p>
          </div>
          <div className="text-center">
            <p className="text-2xl font-bold text-success">{formatCurrency(owner?.portfolioValue)}</p>
            <p className="text-sm text-gray-500">Valor Total</p>
          </div>
          <div className="text-center">
            <p className="text-2xl font-bold text-secondary">{owner?.commissionRate}%</p>
            <p className="text-sm text-gray-500">Comisión</p>
          </div>
        </div>
      </div>

      {/* Banking Information */}
      <div className="bg-gray-50 rounded-lg p-6">
        <h4 className="text-lg font-medium text-gray-900 mb-4">Información Bancaria</h4>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <p className="text-sm text-gray-500">Banco</p>
            <p className="text-sm font-medium text-gray-900">Banco de Chile</p>
          </div>
          <div>
            <p className="text-sm text-gray-500">Tipo de Cuenta</p>
            <p className="text-sm font-medium text-gray-900">Cuenta Corriente</p>
          </div>
          <div>
            <p className="text-sm text-gray-500">Número de Cuenta</p>
            <p className="text-sm font-medium text-gray-900">****-****-1234</p>
          </div>
          <div>
            <p className="text-sm text-gray-500">Estado de Pago</p>
            <StatusIndicatorBadges status={owner?.paymentStatus} size="sm" />
          </div>
        </div>
      </div>
    </div>
  );

  const renderProperties = () => (
    <div className="space-y-4">
      {mockProperties?.map((property) => (
        <div key={property?.id} className="bg-gray-50 rounded-lg p-6">
          <div className="flex items-start justify-between mb-4">
            <div>
              <h4 className="text-lg font-medium text-gray-900">{property?.address}</h4>
              <p className="text-sm text-gray-500">{property?.type}</p>
            </div>
            <StatusIndicatorBadges status={property?.status} size="sm" />
          </div>
          
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mb-4">
            <div>
              <p className="text-sm text-gray-500">Dormitorios</p>
              <p className="text-sm font-medium text-gray-900">{property?.bedrooms}</p>
            </div>
            <div>
              <p className="text-sm text-gray-500">Baños</p>
              <p className="text-sm font-medium text-gray-900">{property?.bathrooms}</p>
            </div>
            <div>
              <p className="text-sm text-gray-500">Renta Mensual</p>
              <p className="text-sm font-medium text-gray-900">{formatCurrency(property?.rentAmount)}</p>
            </div>
            <div>
              <p className="text-sm text-gray-500">Inquilino</p>
              <p className="text-sm font-medium text-gray-900">{property?.tenant || 'Sin inquilino'}</p>
            </div>
          </div>

          <div className="flex items-center space-x-2">
            <Button variant="outline" size="sm" iconName="Eye">
              Ver Detalles
            </Button>
            <Button variant="outline" size="sm" iconName="Edit">
              Editar
            </Button>
          </div>
        </div>
      ))}
    </div>
  );

  const renderCommissions = () => (
    <div className="space-y-4">
      {/* Commission Summary */}
      <div className="bg-gray-50 rounded-lg p-6 mb-6">
        <h4 className="text-lg font-medium text-gray-900 mb-4">Resumen de Comisiones</h4>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="text-center">
            <p className="text-2xl font-bold text-success">{formatCurrency(owner?.monthlyCommission)}</p>
            <p className="text-sm text-gray-500">Comisión Mensual</p>
          </div>
          <div className="text-center">
            <p className="text-2xl font-bold text-primary">{formatCurrency(425000)}</p>
            <p className="text-sm text-gray-500">Total Último Trimestre</p>
          </div>
          <div className="text-center">
            <p className="text-2xl font-bold text-secondary">{formatCurrency(1680000)}</p>
            <p className="text-sm text-gray-500">Total Año 2024</p>
          </div>
        </div>
      </div>

      {/* Commission History */}
      <div className="space-y-3">
        {mockCommissionHistory?.map((commission) => (
          <div key={commission?.id} className="bg-gray-50 rounded-lg p-4">
            <div className="flex items-center justify-between">
              <div>
                <h5 className="font-medium text-gray-900">{commission?.month}</h5>
                <p className="text-sm text-gray-500">{commission?.properties} propiedades</p>
              </div>
              <div className="text-right">
                <p className="font-medium text-gray-900">{formatCurrency(commission?.amount)}</p>
                <div className="flex items-center space-x-2">
                  <StatusIndicatorBadges status={commission?.status} size="sm" />
                  <span className="text-xs text-gray-500">
                    {commission?.paymentDate && `Pagado: ${formatDate(commission?.paymentDate)}`}
                  </span>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );

  const renderDocuments = () => (
    <div className="space-y-4">
      <div className="text-center py-12">
        <Icon name="FileText" size={48} className="text-gray-400 mx-auto mb-4" />
        <h3 className="text-lg font-medium text-gray-900 mb-2">Documentos del Propietario</h3>
        <p className="text-gray-500 mb-6">Aquí se mostrarán los documentos relacionados con el propietario</p>
        <Button variant="default" iconName="Upload">
          Subir Documento
        </Button>
      </div>
    </div>
  );

  const renderTabContent = () => {
    switch (activeTab) {
      case 'overview':
        return renderOverview();
      case 'properties':
        return renderProperties();
      case 'commissions':
        return renderCommissions();
      case 'documents':
        return renderDocuments();
      default:
        return renderOverview();
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-1000 p-4">
      <div className="bg-white rounded-lg shadow-xl w-full max-w-4xl max-h-[90vh] overflow-hidden">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <div className="flex items-center space-x-4">
            <div className="w-12 h-12 bg-gray-200 rounded-full overflow-hidden">
              {owner?.avatar ? (
                <Image src={owner?.avatar} alt={owner?.name} className="w-full h-full object-cover" />
              ) : (
                <div className="w-full h-full bg-primary text-white flex items-center justify-center text-lg font-medium">
                  {owner?.name?.split(' ')?.map(n => n?.[0])?.join('')?.toUpperCase()}
                </div>
              )}
            </div>
            <div>
              <h2 className="text-xl font-semibold text-gray-900">{owner?.name}</h2>
              <p className="text-sm text-gray-500">{owner?.email}</p>
            </div>
          </div>
          <div className="flex items-center space-x-2">
            <Button variant="outline" size="sm" iconName="Mail">
              Enviar Mensaje
            </Button>
            <Button variant="outline" size="sm" iconName="Edit">
              Editar
            </Button>
            <Button
              variant="ghost"
              size="sm"
              iconName="X"
              onClick={onClose}
            />
          </div>
        </div>

        {/* Tabs */}
        <div className="border-b border-gray-200">
          <nav className="flex space-x-8 px-6">
            {tabs?.map((tab) => (
              <button
                key={tab?.key}
                onClick={() => setActiveTab(tab?.key)}
                className={`
                  flex items-center space-x-2 py-4 px-1 border-b-2 font-medium text-sm transition-colors duration-150
                  ${activeTab === tab?.key
                    ? 'border-primary text-primary' :'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                  }
                `}
              >
                <Icon name={tab?.icon} size={16} />
                <span>{tab?.label}</span>
              </button>
            ))}
          </nav>
        </div>

        {/* Content */}
        <div className="p-6 overflow-y-auto max-h-[calc(90vh-200px)]">
          {renderTabContent()}
        </div>
      </div>
    </div>
  );
};

export default OwnerProfileModal;