import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const OwnersFilters = ({ onFiltersChange, onClearFilters }) => {
  const [filters, setFilters] = useState({
    search: '',
    paymentStatus: '',
    propertiesRange: '',
    commissionRange: '',
    sortBy: 'name'
  });

  const paymentStatusOptions = [
    { value: '', label: 'Todos los estados' },
    { value: 'paid', label: 'Pagado' },
    { value: 'pending', label: 'Pendiente' },
    { value: 'overdue', label: 'Vencido' }
  ];

  const propertiesRangeOptions = [
    { value: '', label: 'Cualquier cantidad' },
    { value: '1-5', label: '1-5 propiedades' },
    { value: '6-10', label: '6-10 propiedades' },
    { value: '11-20', label: '11-20 propiedades' },
    { value: '20+', label: 'Más de 20 propiedades' }
  ];

  const commissionRangeOptions = [
    { value: '', label: 'Cualquier comisión' },
    { value: '0-5', label: '0% - 5%' },
    { value: '6-10', label: '6% - 10%' },
    { value: '11-15', label: '11% - 15%' },
    { value: '15+', label: 'Más de 15%' }
  ];

  const sortByOptions = [
    { value: 'name', label: 'Nombre' },
    { value: 'propertiesCount', label: 'Cantidad de propiedades' },
    { value: 'portfolioValue', label: 'Valor de cartera' },
    { value: 'commissionRate', label: 'Tasa de comisión' },
    { value: 'monthlyCommission', label: 'Comisión mensual' }
  ];

  const handleFilterChange = (key, value) => {
    const newFilters = { ...filters, [key]: value };
    setFilters(newFilters);
    onFiltersChange(newFilters);
  };

  const handleClearFilters = () => {
    const clearedFilters = {
      search: '',
      paymentStatus: '',
      propertiesRange: '',
      commissionRange: '',
      sortBy: 'name'
    };
    setFilters(clearedFilters);
    onClearFilters();
  };

  const hasActiveFilters = filters?.search || filters?.paymentStatus || filters?.propertiesRange || filters?.commissionRange;

  return (
    <div className="bg-white rounded-lg border border-gray-200 p-6 mb-6">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-lg font-medium text-gray-900">Filtros de Búsqueda</h3>
        {hasActiveFilters && (
          <Button
            variant="ghost"
            size="sm"
            iconName="X"
            onClick={handleClearFilters}
          >
            Limpiar Filtros
          </Button>
        )}
      </div>
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-5 gap-4">
        {/* Search Input */}
        <div className="lg:col-span-2">
          <Input
            type="search"
            placeholder="Buscar por nombre, RUT o email..."
            value={filters?.search}
            onChange={(e) => handleFilterChange('search', e?.target?.value)}
            className="w-full"
          />
        </div>

        {/* Payment Status Filter */}
        <Select
          placeholder="Estado de pago"
          options={paymentStatusOptions}
          value={filters?.paymentStatus}
          onChange={(value) => handleFilterChange('paymentStatus', value)}
        />

        {/* Properties Range Filter */}
        <Select
          placeholder="Cantidad propiedades"
          options={propertiesRangeOptions}
          value={filters?.propertiesRange}
          onChange={(value) => handleFilterChange('propertiesRange', value)}
        />

        {/* Commission Range Filter */}
        <Select
          placeholder="Rango comisión"
          options={commissionRangeOptions}
          value={filters?.commissionRange}
          onChange={(value) => handleFilterChange('commissionRange', value)}
        />
      </div>
      {/* Advanced Filters Toggle */}
      <div className="mt-4 pt-4 border-t border-gray-200">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-4">
            <Select
              label="Ordenar por"
              options={sortByOptions}
              value={filters?.sortBy}
              onChange={(value) => handleFilterChange('sortBy', value)}
              className="w-48"
            />
          </div>

          <div className="flex items-center space-x-2 text-sm text-gray-500">
            <Icon name="Filter" size={16} />
            <span>
              {hasActiveFilters ? 'Filtros activos' : 'Sin filtros aplicados'}
            </span>
          </div>
        </div>
      </div>
    </div>
  );
};

export default OwnersFilters;