import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';
import Button from '../../../components/ui/Button';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const OwnersTable = ({ owners, onEdit, onDelete, onViewProfile, onSendMessage }) => {
  const [selectedOwners, setSelectedOwners] = useState([]);
  const navigate = useNavigate();

  const handleSelectAll = (checked) => {
    if (checked) {
      setSelectedOwners(owners?.map(owner => owner?.id));
    } else {
      setSelectedOwners([]);
    }
  };

  const handleSelectOwner = (ownerId, checked) => {
    if (checked) {
      setSelectedOwners([...selectedOwners, ownerId]);
    } else {
      setSelectedOwners(selectedOwners?.filter(id => id !== ownerId));
    }
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  const formatPercentage = (rate) => {
    return `${rate}%`;
  };

  return (
    <div className="bg-white rounded-lg border border-gray-200 overflow-hidden">
      {/* Table Header */}
      <div className="px-6 py-4 border-b border-gray-200 bg-gray-50">
        <div className="flex items-center justify-between">
          <div className="flex items-center space-x-3">
            <input
              type="checkbox"
              checked={selectedOwners?.length === owners?.length && owners?.length > 0}
              onChange={(e) => handleSelectAll(e?.target?.checked)}
              className="w-4 h-4 text-primary border-gray-300 rounded focus:ring-primary"
            />
            <span className="text-sm font-medium text-gray-700">
              {selectedOwners?.length > 0 ? `${selectedOwners?.length} seleccionados` : 'Seleccionar todo'}
            </span>
          </div>
          
          {selectedOwners?.length > 0 && (
            <div className="flex items-center space-x-2">
              <Button variant="outline" size="sm" iconName="Mail">
                Enviar Mensaje
              </Button>
              <Button variant="outline" size="sm" iconName="FileText">
                Generar Reporte
              </Button>
              <Button variant="outline" size="sm" iconName="CreditCard">
                Procesar Comisiones
              </Button>
            </div>
          )}
        </div>
      </div>
      {/* Desktop Table */}
      <div className="hidden lg:block overflow-x-auto">
        <table className="w-full">
          <thead className="bg-gray-50 border-b border-gray-200">
            <tr>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Propietario
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Contacto
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Propiedades
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Valor Cartera
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Comisión
              </th>
              <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                Estado Pago
              </th>
              <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                Acciones
              </th>
            </tr>
          </thead>
          <tbody className="bg-white divide-y divide-gray-200">
            {owners?.map((owner) => (
              <tr key={owner?.id} className="hover:bg-gray-50 transition-colors duration-150">
                <td className="px-6 py-4 whitespace-nowrap">
                  <div className="flex items-center space-x-3">
                    <input
                      type="checkbox"
                      checked={selectedOwners?.includes(owner?.id)}
                      onChange={(e) => handleSelectOwner(owner?.id, e?.target?.checked)}
                      className="w-4 h-4 text-primary border-gray-300 rounded focus:ring-primary"
                    />
                    <div className="flex items-center space-x-3">
                      <div className="w-10 h-10 bg-gray-200 rounded-full overflow-hidden">
                        {owner?.avatar ? (
                          <Image src={owner?.avatar} alt={owner?.name} className="w-full h-full object-cover" />
                        ) : (
                          <div className="w-full h-full bg-primary text-white flex items-center justify-center text-sm font-medium">
                            {owner?.name?.split(' ')?.map(n => n?.[0])?.join('')?.toUpperCase()}
                          </div>
                        )}
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-900">{owner?.name}</p>
                        <p className="text-xs text-gray-500">{owner?.rut}</p>
                      </div>
                    </div>
                  </div>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <div>
                    <p className="text-sm text-gray-900">{owner?.email}</p>
                    <p className="text-xs text-gray-500">{owner?.phone}</p>
                  </div>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <div className="flex items-center space-x-2">
                    <span className="text-sm font-medium text-gray-900">{owner?.propertiesCount}</span>
                    <span className="text-xs text-gray-500">propiedades</span>
                  </div>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <p className="text-sm font-medium text-gray-900">{formatCurrency(owner?.portfolioValue)}</p>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <div>
                    <p className="text-sm font-medium text-gray-900">{formatPercentage(owner?.commissionRate)}</p>
                    <p className="text-xs text-gray-500">{formatCurrency(owner?.monthlyCommission)}/mes</p>
                  </div>
                </td>
                <td className="px-6 py-4 whitespace-nowrap">
                  <StatusIndicatorBadges status={owner?.paymentStatus} size="sm" />
                </td>
                <td className="px-6 py-4 whitespace-nowrap text-right">
                  <div className="flex items-center justify-end space-x-2">
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Eye"
                      onClick={() => onViewProfile(owner)}
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Mail"
                      onClick={() => onSendMessage(owner)}
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="Edit"
                      onClick={() => onEdit(owner)}
                    />
                    <Button
                      variant="ghost"
                      size="sm"
                      iconName="MoreVertical"
                    />
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
      {/* Mobile Cards */}
      <div className="lg:hidden divide-y divide-gray-200">
        {owners?.map((owner) => (
          <div key={owner?.id} className="p-4 space-y-3">
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-3">
                <input
                  type="checkbox"
                  checked={selectedOwners?.includes(owner?.id)}
                  onChange={(e) => handleSelectOwner(owner?.id, e?.target?.checked)}
                  className="w-4 h-4 text-primary border-gray-300 rounded focus:ring-primary"
                />
                <div className="w-10 h-10 bg-gray-200 rounded-full overflow-hidden">
                  {owner?.avatar ? (
                    <Image src={owner?.avatar} alt={owner?.name} className="w-full h-full object-cover" />
                  ) : (
                    <div className="w-full h-full bg-primary text-white flex items-center justify-center text-sm font-medium">
                      {owner?.name?.split(' ')?.map(n => n?.[0])?.join('')?.toUpperCase()}
                    </div>
                  )}
                </div>
                <div>
                  <p className="text-sm font-medium text-gray-900">{owner?.name}</p>
                  <p className="text-xs text-gray-500">{owner?.rut}</p>
                </div>
              </div>
              <StatusIndicatorBadges status={owner?.paymentStatus} size="sm" />
            </div>

            <div className="grid grid-cols-2 gap-3 text-sm">
              <div>
                <p className="text-gray-500">Email</p>
                <p className="text-gray-900">{owner?.email}</p>
              </div>
              <div>
                <p className="text-gray-500">Teléfono</p>
                <p className="text-gray-900">{owner?.phone}</p>
              </div>
              <div>
                <p className="text-gray-500">Propiedades</p>
                <p className="text-gray-900">{owner?.propertiesCount}</p>
              </div>
              <div>
                <p className="text-gray-500">Comisión</p>
                <p className="text-gray-900">{formatPercentage(owner?.commissionRate)}</p>
              </div>
            </div>

            <div className="pt-2 border-t border-gray-100">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-xs text-gray-500">Valor Cartera</p>
                  <p className="text-sm font-medium text-gray-900">{formatCurrency(owner?.portfolioValue)}</p>
                </div>
                <div className="flex items-center space-x-2">
                  <Button variant="ghost" size="sm" iconName="Eye" onClick={() => onViewProfile(owner)} />
                  <Button variant="ghost" size="sm" iconName="Mail" onClick={() => onSendMessage(owner)} />
                  <Button variant="ghost" size="sm" iconName="Edit" onClick={() => onEdit(owner)} />
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>
      {/* Empty State */}
      {owners?.length === 0 && (
        <div className="text-center py-12">
          <Icon name="Users" size={48} className="text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No hay propietarios registrados</h3>
          <p className="text-gray-500 mb-6">Comienza agregando tu primer propietario al sistema</p>
          <Button variant="default" iconName="Plus">
            Agregar Propietario
          </Button>
        </div>
      )}
    </div>
  );
};

export default OwnersTable;