import React, { useState, useEffect } from 'react';
import { useNavigate } from 'react-router-dom';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';
import Icon from '../../components/AppIcon';
import Button from '../../components/ui/Button';

// Import components
import OwnersTable from './components/OwnersTable';
import OwnersFilters from './components/OwnersFilters';
import CommissionDashboard from './components/CommissionDashboard';
import AddOwnerModal from './components/AddOwnerModal';
import OwnerProfileModal from './components/OwnerProfileModal';

const OwnersManagement = () => {
  const navigate = useNavigate();
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [isAddModalOpen, setIsAddModalOpen] = useState(false);
  const [isProfileModalOpen, setIsProfileModalOpen] = useState(false);
  const [selectedOwner, setSelectedOwner] = useState(null);
  const [filteredOwners, setFilteredOwners] = useState([]);
  const [currentFilters, setCurrentFilters] = useState({});

  // Mock data for owners
  const mockOwners = [
    {
      id: 1,
      name: 'Carlos Mendoza Silva',
      rut: '12345678-9',
      email: 'carlos.mendoza@email.com',
      phone: '+56 9 8765 4321',
      avatar: null,
      propertiesCount: 3,
      portfolioValue: 2850000,
      commissionRate: 8.5,
      monthlyCommission: 242250,
      paymentStatus: 'paid',
      address: 'Las Condes 1234, Santiago',
      bankName: 'Banco de Chile',
      accountType: 'Cuenta Corriente',
      accountNumber: '****-****-1234',
      joinDate: '2023-01-15',
      totalEarnings: 2906000
    },
    {
      id: 2,
      name: 'María Elena Rodríguez',
      rut: '98765432-1',
      email: 'maria.rodriguez@email.com',
      phone: '+56 9 1234 5678',
      avatar: 'https://images.unsplash.com/photo-1494790108755-2616b612b786?w=150',
      propertiesCount: 2,
      portfolioValue: 1950000,
      commissionRate: 10.0,
      monthlyCommission: 195000,
      paymentStatus: 'pending',
      address: 'Providencia 567, Santiago',
      bankName: 'BancoEstado',
      accountType: 'Cuenta de Ahorro',
      accountNumber: '****-****-5678',
      joinDate: '2023-03-20',
      totalEarnings: 2340000
    },
    {
      id: 3,
      name: 'Roberto Fernández López',
      rut: '11223344-5',
      email: 'roberto.fernandez@email.com',
      phone: '+56 9 5555 6666',
      avatar: null,
      propertiesCount: 5,
      portfolioValue: 4200000,
      commissionRate: 7.5,
      monthlyCommission: 315000,
      paymentStatus: 'paid',
      address: 'Ñuñoa 890, Santiago',
      bankName: 'Banco Santander',
      accountType: 'Cuenta Corriente',
      accountNumber: '****-****-9012',
      joinDate: '2022-11-10',
      totalEarnings: 3780000
    },
    {
      id: 4,
      name: 'Ana Patricia Morales',
      rut: '55667788-9',
      email: 'ana.morales@email.com',
      phone: '+56 9 7777 8888',
      avatar: 'https://images.unsplash.com/photo-1438761681033-6461ffad8d80?w=150',
      propertiesCount: 1,
      portfolioValue: 850000,
      commissionRate: 9.0,
      monthlyCommission: 76500,
      paymentStatus: 'overdue',
      address: 'Maipú 345, Santiago',
      bankName: 'Banco BCI',
      accountType: 'Cuenta Vista',
      accountNumber: '****-****-3456',
      joinDate: '2023-06-05',
      totalEarnings: 459000
    },
    {
      id: 5,
      name: 'Luis Alberto Castillo',
      rut: '99887766-5',
      email: 'luis.castillo@email.com',
      phone: '+56 9 9999 0000',
      avatar: null,
      propertiesCount: 4,
      portfolioValue: 3400000,
      commissionRate: 8.0,
      monthlyCommission: 272000,
      paymentStatus: 'paid',
      address: 'La Reina 678, Santiago',
      bankName: 'Scotiabank',
      accountType: 'Cuenta Corriente',
      accountNumber: '****-****-7890',
      joinDate: '2022-08-18',
      totalEarnings: 3264000
    }
  ];

  // Calculate commission data from mock owners
  const commissionData = {
    totalCommissions: mockOwners.reduce((sum, owner) => sum + owner.monthlyCommission, 0),
    paidCommissions: mockOwners.filter(owner => owner.paymentStatus === 'paid').reduce((sum, owner) => sum + owner.monthlyCommission, 0),
    pendingCommissions: mockOwners.filter(owner => owner.paymentStatus === 'pending').reduce((sum, owner) => sum + owner.monthlyCommission, 0),
    overdueCommissions: mockOwners.filter(owner => owner.paymentStatus === 'overdue').reduce((sum, owner) => sum + owner.monthlyCommission, 0),
    totalOwners: mockOwners.length,
    activeOwners: mockOwners.length,
    averageCommissionRate: mockOwners.reduce((sum, owner) => sum + owner.commissionRate, 0) / mockOwners.length,
    totalPortfolioValue: mockOwners.reduce((sum, owner) => sum + owner.portfolioValue, 0)
  };

  useEffect(() => {
    setFilteredOwners(mockOwners);
  }, []);

  const handleSidebarToggle = () => {
    setIsSidebarOpen(!isSidebarOpen);
  };

  const handleCloseSidebar = () => {
    setIsSidebarOpen(false);
  };

  const handleFiltersChange = (filters) => {
    setCurrentFilters(filters);
    let filtered = [...mockOwners];

    // Search filter
    if (filters?.search) {
      const searchTerm = filters?.search?.toLowerCase();
      filtered = filtered?.filter(owner =>
        owner?.name?.toLowerCase()?.includes(searchTerm) ||
        owner?.rut?.toLowerCase()?.includes(searchTerm) ||
        owner?.email?.toLowerCase()?.includes(searchTerm)
      );
    }

    // Payment status filter
    if (filters?.paymentStatus) {
      filtered = filtered?.filter(owner => owner?.paymentStatus === filters?.paymentStatus);
    }

    // Properties range filter
    if (filters?.propertiesRange) {
      const [min, max] = filters?.propertiesRange?.split('-')?.map(n => parseInt(n));
      if (filters?.propertiesRange?.includes('+')) {
        filtered = filtered?.filter(owner => owner?.propertiesCount >= min);
      } else {
        filtered = filtered?.filter(owner => owner?.propertiesCount >= min && owner?.propertiesCount <= max);
      }
    }

    // Commission range filter
    if (filters?.commissionRange) {
      const [min, max] = filters?.commissionRange?.split('-')?.map(n => parseFloat(n));
      if (filters?.commissionRange?.includes('+')) {
        filtered = filtered?.filter(owner => owner?.commissionRate >= min);
      } else {
        filtered = filtered?.filter(owner => owner?.commissionRate >= min && owner?.commissionRate <= max);
      }
    }

    // Sort
    if (filters?.sortBy) {
      filtered?.sort((a, b) => {
        switch (filters?.sortBy) {
          case 'name':
            return a?.name?.localeCompare(b?.name);
          case 'propertiesCount':
            return b?.propertiesCount - a?.propertiesCount;
          case 'portfolioValue':
            return b?.portfolioValue - a?.portfolioValue;
          case 'commissionRate':
            return b?.commissionRate - a?.commissionRate;
          case 'monthlyCommission':
            return b?.monthlyCommission - a?.monthlyCommission;
          default:
            return 0;
        }
      });
    }

    setFilteredOwners(filtered);
  };

  const handleClearFilters = () => {
    setCurrentFilters({});
    setFilteredOwners(mockOwners);
  };

  const handleAddOwner = () => {
    setIsAddModalOpen(true);
  };

  const handleSaveOwner = async (ownerData) => {
    console.log('Saving owner:', ownerData);
    // Here you would typically save to your backend
    // For now, we'll just close the modal
    setIsAddModalOpen(false);
  };

  const handleEditOwner = (owner) => {
    console.log('Editing owner:', owner);
    // Here you would open an edit modal or navigate to edit page
  };

  const handleDeleteOwner = (owner) => {
    console.log('Deleting owner:', owner);
    // Here you would show a confirmation dialog and delete
  };

  const handleViewProfile = (owner) => {
    setSelectedOwner(owner);
    setIsProfileModalOpen(true);
  };

  const handleSendMessage = (owner) => {
    console.log('Sending message to owner:', owner);
    // Here you would open a message modal or navigate to messaging
  };

  const handleExportData = () => {
    console.log('Exporting owners data');
    // Here you would implement data export functionality
  };

  const handleBulkActions = () => {
    console.log('Performing bulk actions');
    // Here you would implement bulk actions
  };

  return (
    <div className="min-h-screen bg-background">
      <Header onMenuToggle={handleSidebarToggle} isMenuOpen={isSidebarOpen} />
      <Sidebar isOpen={isSidebarOpen} onClose={handleCloseSidebar} />
      <main className="lg:ml-60 pt-16">
        <div className="p-6">
          <NavigationBreadcrumbs />
          
          {/* Page Header */}
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-8">
            <div>
              <h1 className="text-2xl font-bold text-gray-900">Gestión de Propietarios</h1>
              <p className="text-gray-600 mt-1">
                Administra la información de propietarios y seguimiento de comisiones
              </p>
            </div>
            <div className="flex items-center space-x-3 mt-4 sm:mt-0">
              <Button
                variant="outline"
                iconName="Download"
                onClick={handleExportData}
              >
                Exportar
              </Button>
              <Button
                variant="default"
                iconName="Plus"
                onClick={handleAddOwner}
              >
                Agregar Propietario
              </Button>
            </div>
          </div>

          {/* Commission Dashboard */}
          <CommissionDashboard commissionData={commissionData} />

          {/* Filters */}
          <OwnersFilters
            onFiltersChange={handleFiltersChange}
            onClearFilters={handleClearFilters}
          />

          {/* Results Summary */}
          <div className="flex items-center justify-between mb-6">
            <div className="flex items-center space-x-4">
              <p className="text-sm text-gray-600">
                Mostrando <span className="font-medium">{filteredOwners?.length}</span> de{' '}
                <span className="font-medium">{mockOwners?.length}</span> propietarios
              </p>
              {Object.keys(currentFilters)?.some(key => currentFilters?.[key]) && (
                <div className="flex items-center space-x-2">
                  <Icon name="Filter" size={16} className="text-primary" />
                  <span className="text-sm text-primary font-medium">Filtros aplicados</span>
                </div>
              )}
            </div>
            
            <div className="hidden sm:flex items-center space-x-2">
              <Button
                variant="outline"
                size="sm"
                iconName="RefreshCw"
                onClick={() => handleFiltersChange(currentFilters)}
              >
                Actualizar
              </Button>
            </div>
          </div>

          {/* Owners Table */}
          <OwnersTable
            owners={filteredOwners}
            onEdit={handleEditOwner}
            onDelete={handleDeleteOwner}
            onViewProfile={handleViewProfile}
            onSendMessage={handleSendMessage}
          />

          {/* Modals */}
          <AddOwnerModal
            isOpen={isAddModalOpen}
            onClose={() => setIsAddModalOpen(false)}
            onSave={handleSaveOwner}
          />

          <OwnerProfileModal
            isOpen={isProfileModalOpen}
            onClose={() => setIsProfileModalOpen(false)}
            owner={selectedOwner}
          />
        </div>
      </main>
      <QuickActionFloatingButton />
    </div>
  );
};

export default OwnersManagement;