import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';

const RecoveryForm = () => {
  const [email, setEmail] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [isSubmitted, setIsSubmitted] = useState(false);
  const [error, setError] = useState('');
  const [rateLimitError, setRateLimitError] = useState(false);
  const navigate = useNavigate();

  const validateEmail = (email) => {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex?.test(email);
  };

  const handleSubmit = async (e) => {
    e?.preventDefault();
    setError('');
    setRateLimitError(false);

    if (!email?.trim()) {
      setError('Por favor ingresa tu dirección de correo electrónico');
      return;
    }

    if (!validateEmail(email)) {
      setError('Por favor ingresa una dirección de correo electrónico válida');
      return;
    }

    setIsLoading(true);

    try {
      // Simulate API call
      await new Promise(resolve => setTimeout(resolve, 2000));
      
      // Simulate different scenarios
      const scenarios = ['success', 'not_found', 'rate_limit'];
      const scenario = scenarios?.[Math.floor(Math.random() * scenarios?.length)];
      
      if (scenario === 'not_found') {
        setError('No encontramos una cuenta asociada a este correo electrónico');
      } else if (scenario === 'rate_limit') {
        setRateLimitError(true);
        setError('Has excedido el límite de solicitudes. Por favor espera 15 minutos antes de intentar nuevamente');
      } else {
        setIsSubmitted(true);
      }
    } catch (err) {
      setError('Ha ocurrido un error. Por favor intenta nuevamente');
    } finally {
      setIsLoading(false);
    }
  };

  const handleBackToLogin = () => {
    navigate('/login');
  };

  const handleResendEmail = () => {
    setIsSubmitted(false);
    setEmail('');
    setError('');
  };

  if (isSubmitted) {
    return (
      <div className="w-full max-w-md mx-auto">
        <div className="glass-card p-8 rounded-2xl elevation-2">
          {/* Success Icon */}
          <div className="flex justify-center mb-6">
            <div className="w-16 h-16 bg-success/10 rounded-full flex items-center justify-center">
              <Icon name="CheckCircle" size={32} className="text-success" />
            </div>
          </div>

          {/* Success Message */}
          <div className="text-center mb-8">
            <h2 className="text-2xl font-semibold text-gray-900 mb-3">
              ¡Correo Enviado!
            </h2>
            <p className="text-gray-600 mb-4">
              Hemos enviado un enlace de recuperación a:
            </p>
            <p className="text-primary font-medium mb-4">
              {email}
            </p>
            <p className="text-sm text-gray-500">
              El enlace expirará en 24 horas. Si no recibes el correo en los próximos 5 minutos, 
              revisa tu carpeta de spam o solicita un nuevo enlace.
            </p>
          </div>

          {/* Action Buttons */}
          <div className="space-y-3">
            <Button
              variant="outline"
              fullWidth
              onClick={handleResendEmail}
              iconName="RefreshCw"
              iconPosition="left"
            >
              Enviar Nuevo Enlace
            </Button>
            
            <Button
              variant="ghost"
              fullWidth
              onClick={handleBackToLogin}
              iconName="ArrowLeft"
              iconPosition="left"
            >
              Volver al Inicio de Sesión
            </Button>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="w-full max-w-md mx-auto">
      <div className="glass-card p-8 rounded-2xl elevation-2">
        {/* Header */}
        <div className="text-center mb-8">
          <div className="w-12 h-12 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4">
            <Icon name="KeyRound" size={24} className="text-primary" />
          </div>
          <h1 className="text-2xl font-semibold text-gray-900 mb-2">
            Recuperar Contraseña
          </h1>
          <p className="text-gray-600">
            Ingresa tu correo electrónico y te enviaremos un enlace para restablecer tu contraseña
          </p>
        </div>

        {/* Recovery Form */}
        <form onSubmit={handleSubmit} className="space-y-6">
          <Input
            type="email"
            label="Correo Electrónico"
            placeholder="tu@ejemplo.com"
            value={email}
            onChange={(e) => setEmail(e?.target?.value)}
            error={error}
            required
            disabled={isLoading}
          />

          {rateLimitError && (
            <div className="bg-warning/10 border border-warning/20 rounded-lg p-4">
              <div className="flex items-start space-x-3">
                <Icon name="AlertTriangle" size={20} className="text-warning flex-shrink-0 mt-0.5" />
                <div>
                  <p className="text-sm font-medium text-warning mb-1">
                    Límite de Solicitudes Alcanzado
                  </p>
                  <p className="text-sm text-gray-600">
                    Por seguridad, limitamos las solicitudes de recuperación. 
                    Puedes intentar nuevamente en 15 minutos.
                  </p>
                </div>
              </div>
            </div>
          )}

          <Button
            type="submit"
            variant="default"
            fullWidth
            loading={isLoading}
            disabled={isLoading}
            iconName="Send"
            iconPosition="left"
          >
            {isLoading ? 'Enviando...' : 'Enviar Enlace de Recuperación'}
          </Button>
        </form>

        {/* Back to Login */}
        <div className="mt-6 text-center">
          <button
            onClick={handleBackToLogin}
            className="inline-flex items-center space-x-2 text-sm text-gray-600 hover:text-primary transition-colors duration-150"
          >
            <Icon name="ArrowLeft" size={16} />
            <span>Volver al Inicio de Sesión</span>
          </button>
        </div>
      </div>
    </div>
  );
};

export default RecoveryForm;