import React, { useState } from 'react';

import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const AddPropertyModal = ({ isOpen, onClose, onAdd }) => {
  const [formData, setFormData] = useState({
    address: '',
    neighborhood: '',
    city: '',
    type: '',
    bedrooms: '',
    bathrooms: '',
    area: '',
    rentAmount: '',
    description: '',
    ownerId: ''
  });
  const [isSubmitting, setIsSubmitting] = useState(false);

  const propertyTypes = [
    { value: 'apartment', label: 'Departamento' },
    { value: 'house', label: 'Casa' },
    { value: 'office', label: 'Oficina' },
    { value: 'commercial', label: 'Local Comercial' },
    { value: 'warehouse', label: 'Bodega' }
  ];

  const cityOptions = [
    { value: 'santiago', label: 'Santiago' },
    { value: 'valparaiso', label: 'Valparaíso' },
    { value: 'concepcion', label: 'Concepción' },
    { value: 'la-serena', label: 'La Serena' },
    { value: 'temuco', label: 'Temuco' }
  ];

  const ownerOptions = [
    { value: '1', label: 'Carlos Mendoza' },
    { value: '2', label: 'Ana Rodríguez' },
    { value: '3', label: 'Luis Fernández' },
    { value: '4', label: 'María González' }
  ];

  const handleInputChange = (field, value) => {
    setFormData(prev => ({ ...prev, [field]: value }));
  };

  const handleSubmit = async (e) => {
    e?.preventDefault();
    setIsSubmitting(true);

    // Simulate API call
    setTimeout(() => {
      const newProperty = {
        id: Date.now(),
        ...formData,
        status: 'vacant',
        paymentStatus: 'pending',
        image: 'https://images.unsplash.com/photo-1560518883-ce09059eeffa?w=400&h=300&fit=crop',
        createdAt: new Date(),
        tenant: null
      };

      onAdd(newProperty);
      setIsSubmitting(false);
      onClose();
      
      // Reset form
      setFormData({
        address: '',
        neighborhood: '',
        city: '',
        type: '',
        bedrooms: '',
        bathrooms: '',
        area: '',
        rentAmount: '',
        description: '',
        ownerId: ''
      });
    }, 1000);
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 z-1100 flex items-center justify-center p-4">
      <div className="bg-white rounded-xl max-w-2xl w-full max-h-[90vh] overflow-y-auto">
        {/* Header */}
        <div className="flex items-center justify-between p-6 border-b border-gray-200">
          <div>
            <h2 className="text-xl font-semibold text-gray-900">Nueva Propiedad</h2>
            <p className="text-sm text-gray-500 mt-1">Agregar una nueva propiedad al portafolio</p>
          </div>
          <Button
            variant="ghost"
            size="sm"
            iconName="X"
            onClick={onClose}
          />
        </div>

        {/* Form */}
        <form onSubmit={handleSubmit} className="p-6 space-y-6">
          {/* Basic Information */}
          <div className="space-y-4">
            <h3 className="text-lg font-medium text-gray-900">Información Básica</h3>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="md:col-span-2">
                <Input
                  label="Dirección"
                  type="text"
                  placeholder="Ej: Av. Providencia 1234"
                  value={formData?.address}
                  onChange={(e) => handleInputChange('address', e?.target?.value)}
                  required
                />
              </div>
              
              <Input
                label="Barrio/Sector"
                type="text"
                placeholder="Ej: Providencia"
                value={formData?.neighborhood}
                onChange={(e) => handleInputChange('neighborhood', e?.target?.value)}
                required
              />
              
              <Select
                label="Ciudad"
                options={cityOptions}
                value={formData?.city}
                onChange={(value) => handleInputChange('city', value)}
                required
              />
            </div>
          </div>

          {/* Property Details */}
          <div className="space-y-4">
            <h3 className="text-lg font-medium text-gray-900">Detalles de la Propiedad</h3>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
              <Select
                label="Tipo"
                options={propertyTypes}
                value={formData?.type}
                onChange={(value) => handleInputChange('type', value)}
                required
              />
              
              <Input
                label="Dormitorios"
                type="number"
                min="0"
                placeholder="0"
                value={formData?.bedrooms}
                onChange={(e) => handleInputChange('bedrooms', e?.target?.value)}
                required
              />
              
              <Input
                label="Baños"
                type="number"
                min="0"
                placeholder="0"
                value={formData?.bathrooms}
                onChange={(e) => handleInputChange('bathrooms', e?.target?.value)}
                required
              />
              
              <Input
                label="Área (m²)"
                type="number"
                min="0"
                placeholder="0"
                value={formData?.area}
                onChange={(e) => handleInputChange('area', e?.target?.value)}
                required
              />
            </div>
          </div>

          {/* Financial Information */}
          <div className="space-y-4">
            <h3 className="text-lg font-medium text-gray-900">Información Financiera</h3>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <Input
                label="Monto de Arriendo (CLP)"
                type="number"
                min="0"
                placeholder="500000"
                value={formData?.rentAmount}
                onChange={(e) => handleInputChange('rentAmount', e?.target?.value)}
                required
              />
              
              <Select
                label="Propietario"
                options={ownerOptions}
                value={formData?.ownerId}
                onChange={(value) => handleInputChange('ownerId', value)}
                required
              />
            </div>
          </div>

          {/* Description */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Descripción (Opcional)
            </label>
            <textarea
              rows={4}
              className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary focus:border-transparent resize-none"
              placeholder="Descripción adicional de la propiedad..."
              value={formData?.description}
              onChange={(e) => handleInputChange('description', e?.target?.value)}
            />
          </div>

          {/* Actions */}
          <div className="flex items-center justify-end space-x-3 pt-6 border-t border-gray-200">
            <Button
              type="button"
              variant="outline"
              onClick={onClose}
              disabled={isSubmitting}
            >
              Cancelar
            </Button>
            <Button
              type="submit"
              loading={isSubmitting}
              iconName="Plus"
              iconPosition="left"
            >
              Agregar Propiedad
            </Button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default AddPropertyModal;