import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const PropertyCard = ({ property, onEdit, onSelect, isSelected, viewMode = "list" }) => {
  const [isEditing, setIsEditing] = useState(false);
  const [editedProperty, setEditedProperty] = useState(property);
  const navigate = useNavigate();

  const handleEdit = (field, value) => {
    const updated = { ...editedProperty, [field]: value };
    setEditedProperty(updated);
    // Auto-save simulation
    setTimeout(() => {
      onEdit(property?.id, updated);
    }, 500);
  };

  const handleViewDetails = () => {
    navigate('/property-details', { state: { propertyId: property?.id } });
  };

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  const formatUF = (amount) => {
    const ufValue = amount / 37000; // Mock UF conversion rate
    return `${ufValue?.toFixed(2)} UF`;
  };

  // Enhanced payment marking functionality (required by user)
  const handlePaymentMarkingClick = (paymentMethod) => {
    const updatedProperty = {
      ...property,
      paymentMethod,
      paymentStatus: 'paid',
      lastPaymentDate: new Date()?.toISOString()?.split('T')?.[0],
      paymentMarkedManually: paymentMethod === 'manual'
    };
    
    onEdit?.(property?.id, updatedProperty);
    
    // Show confirmation message
    console.log(`Pago marcado como: ${paymentMethod === 'manual' ? 'Efectivo/Manual' : 'TransBank'}`);
  };

  // Vertical list view (without images) - Modified as per user requirement
  return (
    <div className={`
      glass-card rounded-xl p-4 hover-lift transition-all duration-200 border-l-4
      ${property?.status === 'occupied' ? 'border-l-green-500' : ''}
      ${property?.status === 'vacant' ? 'border-l-blue-500' : ''}
      ${property?.status === 'maintenance' ? 'border-l-orange-500' : ''}
      ${isSelected ? 'ring-2 ring-primary bg-primary/5' : ''}
    `}>
      <div className="flex items-center justify-between">
        {/* Left Section - Basic Info */}
        <div className="flex items-center space-x-4 flex-1">
          {/* Selection Checkbox */}
          <button
            onClick={() => onSelect(property?.id)}
            className={`
              w-5 h-5 rounded border-2 flex items-center justify-center flex-shrink-0
              ${isSelected 
                ? 'bg-primary border-primary' :'border-gray-300 hover:border-primary'
              }
            `}
          >
            {isSelected && <Icon name="Check" size={12} className="text-white" />}
          </button>

          {/* Property Info - Enhanced to show only relevant information */}
          <div className="flex-1 min-w-0">
            <div className="flex items-center space-x-3 mb-1">
              <h3 className="font-semibold text-gray-900 text-lg truncate">
                {property?.address}
              </h3>
              <div className="flex items-center space-x-2">
                <StatusIndicatorBadges status={property?.status} size="sm" />
                <StatusIndicatorBadges status={property?.paymentStatus} size="sm" />
              </div>
            </div>
            
            <div className="flex items-center space-x-4 text-sm text-gray-600 mb-2">
              <span>{property?.neighborhood}, {property?.city}</span>
              <span className="bg-gray-100 px-2 py-1 rounded-full text-xs">
                {property?.type === 'apartment' ? 'Departamento' :
                 property?.type === 'house' ? 'Casa' :
                 property?.type === 'office' ? 'Oficina' :
                 property?.type === 'commercial' ? 'Local Comercial' :
                 property?.type}
              </span>
              <div className="flex items-center space-x-3">
                <div className="flex items-center space-x-1">
                  <Icon name="Bed" size={14} className="text-gray-400" />
                  <span>{property?.bedrooms}</span>
                </div>
                <div className="flex items-center space-x-1">
                  <Icon name="Bath" size={14} className="text-gray-400" />
                  <span>{property?.bathrooms}</span>
                </div>
                <div className="flex items-center space-x-1">
                  <Icon name="Square" size={14} className="text-gray-400" />
                  <span>{property?.area}m²</span>
                </div>
              </div>
            </div>

            {/* Enhanced info with utility amounts and payment status - REQUIRED BY USER */}
            <div className="flex items-center justify-between">
              <div className="flex items-center space-x-6">
                <div>
                  <span className="text-sm text-gray-600">Arriendo:</span>
                  <div className="font-semibold text-gray-900">{formatCurrency(property?.rentAmount)}</div>
                  <div className="text-xs text-gray-500">{formatUF(property?.rentAmount)}</div>
                </div>
                
                {/* Basic utility amounts display - REQUIRED BY USER */}
                <div>
                  <span className="text-sm text-gray-600">Cuentas Básicas:</span>
                  <div className="text-sm space-x-2">
                    <span className="text-blue-600">Agua: {formatCurrency(property?.waterAmount || 25000)}</span>
                    <span className="text-yellow-600">Luz: {formatCurrency(property?.electricityAmount || 45000)}</span>
                    <span className="text-orange-600">Gas: {formatCurrency(property?.gasAmount || 18000)}</span>
                  </div>
                  <div className="text-xs text-gray-500">
                    Gastos Comunes: {formatCurrency(property?.commonExpensesAmount || 45000)}
                  </div>
                </div>

                {/* Enhanced Manual payment status toggle - REQUIRED BY USER */}
                <div>
                  <span className="text-sm text-gray-600">Estado Pago:</span>
                  <div className="flex items-center space-x-2 mt-1">
                    <button
                      onClick={() => handlePaymentMarkingClick('manual')}
                      className={`
                        px-3 py-1 text-xs rounded-lg transition-colors flex items-center space-x-1
                        ${property?.paymentMethod === 'manual' && property?.paymentStatus === 'paid' ?'bg-green-100 text-green-800 border border-green-300' :'bg-gray-100 text-gray-600 hover:bg-green-50 hover:text-green-700'
                        }
                      `}
                    >
                      <Icon name="Banknote" size={12} />
                      <span>Efectivo/Manual</span>
                    </button>
                    <button
                      onClick={() => handlePaymentMarkingClick('transbank')}
                      className={`
                        px-3 py-1 text-xs rounded-lg transition-colors flex items-center space-x-1
                        ${property?.paymentMethod === 'transbank' && property?.paymentStatus === 'paid' ?'bg-blue-100 text-blue-800 border border-blue-300' :'bg-gray-100 text-gray-600 hover:bg-blue-50 hover:text-blue-700'
                        }
                      `}
                    >
                      <Icon name="CreditCard" size={12} />
                      <span>TransBank</span>
                    </button>
                  </div>
                  
                  {/* Payment confirmation display */}
                  {property?.paymentStatus === 'paid' && property?.lastPaymentDate && (
                    <div className="text-xs text-green-600 mt-1">
                      ✓ Pagado el {new Date(property?.lastPaymentDate)?.toLocaleDateString('es-CL')}
                      {property?.paymentMarkedManually && ' (Manual)'}
                    </div>
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* Right Section - Tenant and Actions */}
        <div className="flex items-center space-x-4">
          {/* Tenant Information */}
          <div className="text-right">
            {property?.tenant ? (
              <div>
                <div className="flex items-center space-x-2">
                  <Icon name="User" size={14} className="text-gray-400" />
                  <span className="text-sm font-medium text-gray-900">{property?.tenant?.name}</span>
                </div>
                <div className="text-xs text-gray-500">{property?.tenant?.phone}</div>
                {property?.contractId && (
                  <div className="text-xs text-gray-500">
                    Contrato: {property?.contractId}
                  </div>
                )}
              </div>
            ) : (
              <div className="flex items-center space-x-2">
                <Icon name="Home" size={14} className="text-gray-400" />
                <span className="text-sm text-gray-500">Disponible</span>
              </div>
            )}
            
            {/* Owner info */}
            <div className="text-xs text-gray-500 mt-1">
              Propietario: {property?.owner?.name}
            </div>
          </div>

          {/* Actions */}
          <div className="flex items-center space-x-2">
            <Button
              variant="outline"
              size="sm"
              iconName="Eye"
              iconPosition="left"
              onClick={handleViewDetails}
            >
              Ver Detalles
            </Button>
            
            <Button
              variant="ghost"
              size="sm"
              iconName="Edit"
              onClick={() => setIsEditing(!isEditing)}
            />
            
            <Button
              variant="ghost"
              size="sm"
              iconName="MoreVertical"
            />
          </div>
        </div>
      </div>
    </div>
  );
};

export default PropertyCard;