import React, { useState } from 'react';
import Icon from '../../../components/AppIcon';
import Button from '../../../components/ui/Button';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';

const PropertyFilters = ({ onFiltersChange, onSearch, totalProperties, filteredCount }) => {
  const [searchTerm, setSearchTerm] = useState('');
  const [filters, setFilters] = useState({
    type: '',
    status: '',
    paymentStatus: '',
    city: '',
    priceRange: ''
  });
  const [showAdvanced, setShowAdvanced] = useState(false);

  const propertyTypes = [
    { value: '', label: 'Todos los tipos' },
    { value: 'apartment', label: 'Departamento' },
    { value: 'house', label: 'Casa' },
    { value: 'office', label: 'Oficina' },
    { value: 'commercial', label: 'Local Comercial' },
    { value: 'warehouse', label: 'Bodega' }
  ];

  const statusOptions = [
    { value: '', label: 'Todos los estados' },
    { value: 'occupied', label: 'Ocupada' },
    { value: 'vacant', label: 'Disponible' },
    { value: 'maintenance', label: 'Mantenimiento' }
  ];

  const paymentStatusOptions = [
    { value: '', label: 'Estado de pago' },
    { value: 'paid', label: 'Pagado' },
    { value: 'pending', label: 'Pendiente' },
    { value: 'overdue', label: 'Vencido' }
  ];

  const cityOptions = [
    { value: '', label: 'Todas las ciudades' },
    { value: 'santiago', label: 'Santiago' },
    { value: 'valparaiso', label: 'Valparaíso' },
    { value: 'concepcion', label: 'Concepción' },
    { value: 'la-serena', label: 'La Serena' },
    { value: 'temuco', label: 'Temuco' }
  ];

  const priceRanges = [
    { value: '', label: 'Rango de precio' },
    { value: '0-500000', label: 'Hasta $500.000' },
    { value: '500000-1000000', label: '$500.000 - $1.000.000' },
    { value: '1000000-2000000', label: '$1.000.000 - $2.000.000' },
    { value: '2000000+', label: 'Más de $2.000.000' }
  ];

  const handleSearchChange = (e) => {
    const value = e?.target?.value;
    setSearchTerm(value);
    onSearch(value);
  };

  const handleFilterChange = (key, value) => {
    const newFilters = { ...filters, [key]: value };
    setFilters(newFilters);
    onFiltersChange(newFilters);
  };

  const clearFilters = () => {
    const clearedFilters = {
      type: '',
      status: '',
      paymentStatus: '',
      city: '',
      priceRange: ''
    };
    setFilters(clearedFilters);
    setSearchTerm('');
    onFiltersChange(clearedFilters);
    onSearch('');
  };

  const hasActiveFilters = Object.values(filters)?.some(value => value !== '') || searchTerm !== '';

  return (
    <div className="bg-white rounded-xl border border-gray-200 p-6 mb-6">
      {/* Search Bar */}
      <div className="flex items-center space-x-4 mb-4">
        <div className="flex-1 relative">
          <Icon 
            name="Search" 
            size={20} 
            className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" 
          />
          <Input
            type="search"
            placeholder="Buscar por dirección, inquilino o propietario..."
            value={searchTerm}
            onChange={handleSearchChange}
            className="pl-10"
          />
        </div>
        
        <Button
          variant="outline"
          iconName="Filter"
          iconPosition="left"
          onClick={() => setShowAdvanced(!showAdvanced)}
        >
          Filtros Avanzados
        </Button>
      </div>
      {/* Advanced Filters */}
      {showAdvanced && (
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4 pt-4 border-t border-gray-100">
          <Select
            placeholder="Tipo de propiedad"
            options={propertyTypes}
            value={filters?.type}
            onChange={(value) => handleFilterChange('type', value)}
          />
          
          <Select
            placeholder="Estado"
            options={statusOptions}
            value={filters?.status}
            onChange={(value) => handleFilterChange('status', value)}
          />
          
          <Select
            placeholder="Estado de pago"
            options={paymentStatusOptions}
            value={filters?.paymentStatus}
            onChange={(value) => handleFilterChange('paymentStatus', value)}
          />
          
          <Select
            placeholder="Ciudad"
            options={cityOptions}
            value={filters?.city}
            onChange={(value) => handleFilterChange('city', value)}
          />
          
          <Select
            placeholder="Rango de precio"
            options={priceRanges}
            value={filters?.priceRange}
            onChange={(value) => handleFilterChange('priceRange', value)}
          />
        </div>
      )}
      {/* Filter Summary */}
      <div className="flex items-center justify-between mt-4 pt-4 border-t border-gray-100">
        <div className="flex items-center space-x-4">
          <p className="text-sm text-gray-600">
            Mostrando <span className="font-medium">{filteredCount}</span> de{' '}
            <span className="font-medium">{totalProperties}</span> propiedades
          </p>
          
          {hasActiveFilters && (
            <Button
              variant="ghost"
              size="sm"
              iconName="X"
              iconPosition="left"
              onClick={clearFilters}
            >
              Limpiar filtros
            </Button>
          )}
        </div>

        <div className="flex items-center space-x-2">
          <span className="text-sm text-gray-500">Ordenar por:</span>
          <Select
            options={[
              { value: 'address', label: 'Dirección' },
              { value: 'rent-high', label: 'Precio: Mayor a menor' },
              { value: 'rent-low', label: 'Precio: Menor a mayor' },
              { value: 'status', label: 'Estado' },
              { value: 'updated', label: 'Actualización reciente' }
            ]}
            value="address"
            onChange={() => {}}
            className="w-48"
          />
        </div>
      </div>
    </div>
  );
};

export default PropertyFilters;