import React, { useState, useEffect } from 'react';
import { Helmet } from 'react-helmet';
import Header from '../../components/ui/Header';
import Sidebar from '../../components/ui/Sidebar';
import NavigationBreadcrumbs from '../../components/ui/NavigationBreadcrumbs';
import QuickActionFloatingButton from '../../components/ui/QuickActionFloatingButton';
import Button from '../../components/ui/Button';
import PropertyFilters from './components/PropertyFilters';
import PropertyGrid from './components/PropertyGrid';
import BulkActionsBar from './components/BulkActionsBar';
import AddPropertyModal from './components/AddPropertyModal';

const PropertiesManagement = () => {
  const [isSidebarOpen, setIsSidebarOpen] = useState(false);
  const [properties, setProperties] = useState([]);
  const [filteredProperties, setFilteredProperties] = useState([]);
  const [selectedProperties, setSelectedProperties] = useState([]);
  const [loading, setLoading] = useState(true);
  const [showAddModal, setShowAddModal] = useState(false);
  const [searchTerm, setSearchTerm] = useState('');
  const [filters, setFilters] = useState({});

  // Mock properties data with enhanced utility information
  const mockProperties = [
    {
      id: 1,
      address: 'Av. Providencia 1234, Depto 501',
      neighborhood: 'Providencia',
      city: 'Santiago',
      type: 'apartment',
      bedrooms: 2,
      bathrooms: 2,
      area: 85,
      rentAmount: 850000,
      status: 'occupied',
      paymentStatus: 'paid',
      paymentMethod: 'transbank',
      waterAmount: 25000,
      electricityAmount: 45000,
      gasAmount: 18000,
      commonExpensesAmount: 45000,
      utilityTotalAmount: 133000,
      tenant: { 
        name: 'María González', 
        phone: '+56 9 8765 4321',
        contractId: 'contract-001'
      },
      owner: { 
        name: 'Carlos Mendoza',
        ownerId: 'owner-001'
      },
      contractId: 'contract-001',
      createdAt: new Date('2024-01-15')
    },
    {
      id: 2,
      address: 'Calle Los Leones 456',
      neighborhood: 'Las Condes',
      city: 'Santiago',
      type: 'house',
      bedrooms: 4,
      bathrooms: 3,
      area: 180,
      rentAmount: 1500000,
      status: 'vacant',
      paymentStatus: 'pending',
      paymentMethod: 'manual',
      waterAmount: 35000,
      electricityAmount: 75000,
      gasAmount: 28000,
      commonExpensesAmount: 60000,
      utilityTotalAmount: 198000,
      tenant: null,
      owner: { 
        name: 'Ana Rodríguez',
        ownerId: 'owner-002'
      },
      contractId: null,
      createdAt: new Date('2024-02-20')
    },
    {
      id: 3,
      address: 'Av. Vitacura 2890, Oficina 12',
      neighborhood: 'Vitacura',
      city: 'Santiago',
      type: 'office',
      bedrooms: 0,
      bathrooms: 1,
      area: 45,
      rentAmount: 650000,
      status: 'occupied',
      paymentStatus: 'overdue',
      image: 'https://images.unsplash.com/photo-1497366216548-37526070297c?w=400&h=300&fit=crop',
      tenant: { name: 'Empresa Tech Solutions', phone: '+56 2 2345 6789' },
      owner: { name: 'Luis Fernández' },
      createdAt: new Date('2024-01-10')
    },
    {
      id: 4,
      address: 'Pasaje San Martín 789',
      neighborhood: 'Ñuñoa',
      city: 'Santiago',
      type: 'apartment',
      bedrooms: 1,
      bathrooms: 1,
      area: 42,
      rentAmount: 450000,
      status: 'maintenance',
      paymentStatus: 'pending',
      image: 'https://images.unsplash.com/photo-1522708323590-d24dbb6b0267?w=400&h=300&fit=crop',
      tenant: null,
      owner: { name: 'María González' },
      createdAt: new Date('2024-03-05')
    },
    {
      id: 5,
      address: 'Av. Libertador Bernardo O\'Higgins 3456',
      neighborhood: 'Santiago Centro',
      city: 'Santiago',
      type: 'commercial',
      bedrooms: 0,
      bathrooms: 2,
      area: 120,
      rentAmount: 980000,
      status: 'occupied',
      paymentStatus: 'paid',
      image: 'https://images.unsplash.com/photo-1441986300917-64674bd600d8?w=400&h=300&fit=crop',
      tenant: { name: 'Restaurante El Buen Sabor', phone: '+56 2 2987 6543' },
      owner: { name: 'Carlos Mendoza' },
      createdAt: new Date('2024-02-01')
    },
    {
      id: 6,
      address: 'Calle Merced 567, Depto 302',
      neighborhood: 'Santiago Centro',
      city: 'Santiago',
      type: 'apartment',
      bedrooms: 3,
      bathrooms: 2,
      area: 95,
      rentAmount: 720000,
      status: 'vacant',
      paymentStatus: 'pending',
      image: 'https://images.unsplash.com/photo-1484154218962-a197022b5858?w=400&h=300&fit=crop',
      tenant: null,
      owner: { name: 'Ana Rodríguez' },
      createdAt: new Date('2024-03-12')
    },
    {
      id: 7,
      address: 'Av. Apoquindo 4567, Torre B, Piso 15',
      neighborhood: 'Las Condes',
      city: 'Santiago',
      type: 'office',
      bedrooms: 0,
      bathrooms: 2,
      area: 200,
      rentAmount: 2200000,
      status: 'occupied',
      paymentStatus: 'paid',
      image: 'https://images.unsplash.com/photo-1497366811353-6870744d04b2?w=400&h=300&fit=crop',
      tenant: { name: 'Consultora Estratégica Ltda.', phone: '+56 2 2456 7890' },
      owner: { name: 'Luis Fernández' },
      createdAt: new Date('2024-01-25')
    },
    {
      id: 8,
      address: 'Calle Huérfanos 1234, Local 5',
      neighborhood: 'Santiago Centro',
      city: 'Santiago',
      type: 'commercial',
      bedrooms: 0,
      bathrooms: 1,
      area: 35,
      rentAmount: 380000,
      status: 'occupied',
      paymentStatus: 'overdue',
      image: 'https://images.unsplash.com/photo-1441986300917-64674bd600d8?w=400&h=300&fit=crop',
      tenant: { name: 'Tienda de Ropa Moderna', phone: '+56 9 7654 3210' },
      owner: { name: 'María González' },
      createdAt: new Date('2024-02-14')
    }
  ];

  useEffect(() => {
    // Simulate API call
    setTimeout(() => {
      setProperties(mockProperties);
      setFilteredProperties(mockProperties);
      setLoading(false);
    }, 1000);
  }, []);

  const handleSearch = (term) => {
    setSearchTerm(term);
    applyFilters(term, filters);
  };

  const handleFiltersChange = (newFilters) => {
    setFilters(newFilters);
    applyFilters(searchTerm, newFilters);
  };

  const applyFilters = (search, filterOptions) => {
    let filtered = [...properties];

    // Apply search
    if (search) {
      filtered = filtered?.filter(property =>
        property?.address?.toLowerCase()?.includes(search?.toLowerCase()) ||
        property?.neighborhood?.toLowerCase()?.includes(search?.toLowerCase()) ||
        property?.city?.toLowerCase()?.includes(search?.toLowerCase()) ||
        (property?.tenant && property?.tenant?.name?.toLowerCase()?.includes(search?.toLowerCase())) ||
        property?.owner?.name?.toLowerCase()?.includes(search?.toLowerCase())
      );
    }

    // Apply filters
    Object.entries(filterOptions)?.forEach(([key, value]) => {
      if (value) {
        switch (key) {
          case 'type':
            filtered = filtered?.filter(p => p?.type === value);
            break;
          case 'status':
            filtered = filtered?.filter(p => p?.status === value);
            break;
          case 'paymentStatus':
            filtered = filtered?.filter(p => p?.paymentStatus === value);
            break;
          case 'city':
            filtered = filtered?.filter(p => p?.city === value);
            break;
          case 'priceRange':
            const [min, max] = value?.split('-')?.map(v => v?.replace('+', ''));
            if (max) {
              filtered = filtered?.filter(p => p?.rentAmount >= parseInt(min) && p?.rentAmount <= parseInt(max));
            } else {
              filtered = filtered?.filter(p => p?.rentAmount >= parseInt(min));
            }
            break;
        }
      }
    });

    setFilteredProperties(filtered);
  };

  const handlePropertySelect = (propertyId) => {
    setSelectedProperties(prev => 
      prev?.includes(propertyId)
        ? prev?.filter(id => id !== propertyId)
        : [...prev, propertyId]
    );
  };

  const handleSelectAll = () => {
    if (selectedProperties?.length === filteredProperties?.length) {
      setSelectedProperties([]);
    } else {
      setSelectedProperties(filteredProperties?.map(p => p?.id));
    }
  };

  const handlePropertyEdit = (propertyId, updatedData) => {
    setProperties(prev => 
      prev?.map(property => 
        property?.id === propertyId 
          ? { ...property, ...updatedData }
          : property
      )
    );
    
    setFilteredProperties(prev => 
      prev?.map(property => 
        property?.id === propertyId 
          ? { ...property, ...updatedData }
          : property
      )
    );
  };

  const handleAddProperty = (newProperty) => {
    setProperties(prev => [newProperty, ...prev]);
    setFilteredProperties(prev => [newProperty, ...prev]);
  };

  const handleBulkAction = (action) => {
    console.log(`Executing bulk action: ${action} on properties:`, selectedProperties);
    // Implement bulk actions here
    setSelectedProperties([]);
  };

  return (
    <>
      <Helmet>
        <title>Gestión de Propiedades - PropManager</title>
        <meta name="description" content="Administra tu portafolio inmobiliario con herramientas avanzadas de gestión de propiedades" />
      </Helmet>
      <div className="min-h-screen bg-background">
        <Header 
          onMenuToggle={() => setIsSidebarOpen(!isSidebarOpen)}
          isMenuOpen={isSidebarOpen}
        />
        
        <Sidebar 
          isOpen={isSidebarOpen}
          onClose={() => setIsSidebarOpen(false)}
        />

        <main className="lg:ml-60 pt-16">
          <div className="p-6">
            {/* Page Header */}
            <div className="mb-6">
              <NavigationBreadcrumbs />
              
              <div className="flex items-center justify-between">
                <div>
                  <h1 className="text-2xl font-bold text-gray-900">Gestión de Propiedades</h1>
                  <p className="text-gray-600 mt-1">
                    Administra tu portafolio inmobiliario completo
                  </p>
                </div>
                
                <div className="flex items-center space-x-3">
                  {filteredProperties?.length > 0 && (
                    <Button
                      variant="outline"
                      iconName={selectedProperties?.length === filteredProperties?.length ? "Square" : "CheckSquare"}
                      iconPosition="left"
                      onClick={handleSelectAll}
                    >
                      {selectedProperties?.length === filteredProperties?.length ? 'Deseleccionar' : 'Seleccionar'} Todo
                    </Button>
                  )}
                  
                  <Button
                    iconName="Plus"
                    iconPosition="left"
                    onClick={() => setShowAddModal(true)}
                  >
                    Nueva Propiedad
                  </Button>
                </div>
              </div>
            </div>

            {/* Filters */}
            <PropertyFilters
              onFiltersChange={handleFiltersChange}
              onSearch={handleSearch}
              totalProperties={properties?.length}
              filteredCount={filteredProperties?.length}
            />

            {/* Properties Grid */}
            <PropertyGrid
              properties={filteredProperties}
              loading={loading}
              selectedProperties={selectedProperties}
              onPropertySelect={handlePropertySelect}
              onPropertyEdit={handlePropertyEdit}
            />

            {/* Bulk Actions Bar */}
            <BulkActionsBar
              selectedCount={selectedProperties?.length}
              onClearSelection={() => setSelectedProperties([])}
              onBulkAction={handleBulkAction}
            />

            {/* Add Property Modal */}
            <AddPropertyModal
              isOpen={showAddModal}
              onClose={() => setShowAddModal(false)}
              onAdd={handleAddProperty}
            />
          </div>
        </main>

        <QuickActionFloatingButton />
      </div>
    </>
  );
};

export default PropertiesManagement;