import React, { useState, useEffect } from 'react';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';

const BasicDataTab = ({ propertyData, onUpdate }) => {
  const [formData, setFormData] = useState(propertyData);
  const [isEditing, setIsEditing] = useState(false);
  const [hasChanges, setHasChanges] = useState(false);
  const [isSaving, setIsSaving] = useState(false);

  const propertyTypes = [
    { value: 'apartment', label: 'Departamento' },
    { value: 'house', label: 'Casa' },
    { value: 'office', label: 'Oficina' },
    { value: 'commercial', label: 'Local Comercial' },
    { value: 'warehouse', label: 'Bodega' }
  ];

  const statusOptions = [
    { value: 'occupied', label: 'Ocupada' },
    { value: 'vacant', label: 'Disponible' },
    { value: 'maintenance', label: 'En Mantenimiento' }
  ];

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    }).format(amount);
  };

  useEffect(() => {
    const hasDataChanged = JSON.stringify(formData) !== JSON.stringify(propertyData);
    setHasChanges(hasDataChanged);
  }, [formData, propertyData]);

  const handleInputChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleSave = async () => {
    setIsSaving(true);
    try {
      await new Promise(resolve => setTimeout(resolve, 1000));
      onUpdate(formData);
      setIsEditing(false);
      setHasChanges(false);
    } catch (error) {
      console.error('Error saving data:', error);
    } finally {
      setIsSaving(false);
    }
  };

  const handleCancel = () => {
    setFormData(propertyData);
    setIsEditing(false);
    setHasChanges(false);
  };

  return (
    <div className="space-y-6">
      {/* Header with Edit Controls */}
      <div className="flex items-center justify-between">
        <div className="flex items-center space-x-3">
          <h3 className="text-lg font-semibold text-gray-900">Datos Básicos</h3>
          <StatusIndicatorBadges status={formData?.status} />
        </div>
        
        <div className="flex items-center space-x-2">
          {isEditing ? (
            <>
              <Button
                variant="outline"
                size="sm"
                onClick={handleCancel}
                disabled={isSaving}
              >
                Cancelar
              </Button>
              <Button
                variant="default"
                size="sm"
                onClick={handleSave}
                loading={isSaving}
                disabled={!hasChanges}
                iconName="Save"
                iconPosition="left"
              >
                Guardar
              </Button>
            </>
          ) : (
            <Button
              variant="outline"
              size="sm"
              onClick={() => setIsEditing(true)}
              iconName="Edit"
              iconPosition="left"
            >
              Editar
            </Button>
          )}
        </div>
      </div>
      {/* Form Fields */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        {/* Property Address */}
        <div className="md:col-span-2">
          <Input
            label="Dirección de la Propiedad"
            type="text"
            value={formData?.address}
            onChange={(e) => handleInputChange('address', e?.target?.value)}
            disabled={!isEditing}
            required
            className="w-full"
          />
        </div>

        {/* Property Type */}
        <Select
          label="Tipo de Propiedad"
          options={propertyTypes}
          value={formData?.type}
          onChange={(value) => handleInputChange('type', value)}
          disabled={!isEditing}
          required
        />

        {/* Property Status */}
        <Select
          label="Estado Actual"
          options={statusOptions}
          value={formData?.status}
          onChange={(value) => handleInputChange('status', value)}
          disabled={!isEditing}
          required
        />

        {/* Built Area */}
        <Input
          label="Superficie Construida (m²)"
          type="number"
          value={formData?.builtArea}
          onChange={(e) => handleInputChange('builtArea', e?.target?.value)}
          disabled={!isEditing}
          min="1"
          placeholder="0"
        />

        {/* Total Area */}
        <Input
          label="Superficie Total (m²)"
          type="number"
          value={formData?.totalArea}
          onChange={(e) => handleInputChange('totalArea', e?.target?.value)}
          disabled={!isEditing}
          min="1"
          placeholder="0"
        />

        {/* Bedrooms */}
        <Input
          label="Dormitorios"
          type="number"
          value={formData?.bedrooms}
          onChange={(e) => handleInputChange('bedrooms', e?.target?.value)}
          disabled={!isEditing}
          min="0"
          placeholder="0"
        />

        {/* Bathrooms */}
        <Input
          label="Baños"
          type="number"
          value={formData?.bathrooms}
          onChange={(e) => handleInputChange('bathrooms', e?.target?.value)}
          disabled={!isEditing}
          min="0"
          step="0.5"
          placeholder="0"
        />

        {/* Parking Spaces */}
        <Input
          label="Estacionamientos"
          type="number"
          value={formData?.parkingSpaces}
          onChange={(e) => handleInputChange('parkingSpaces', e?.target?.value)}
          disabled={!isEditing}
          min="0"
          placeholder="0"
        />

        {/* Floor */}
        <Input
          label="Piso"
          type="text"
          value={formData?.floor}
          onChange={(e) => handleInputChange('floor', e?.target?.value)}
          disabled={!isEditing}
          placeholder="Ej: 3, PB, -1"
        />

        {/* Property Code */}
        <Input
          label="Código de Propiedad"
          type="text"
          value={formData?.propertyCode}
          onChange={(e) => handleInputChange('propertyCode', e?.target?.value)}
          disabled={!isEditing}
          placeholder="Código único"
        />

        {/* Year Built */}
        <Input
          label="Año de Construcción"
          type="number"
          value={formData?.yearBuilt}
          onChange={(e) => handleInputChange('yearBuilt', e?.target?.value)}
          disabled={!isEditing}
          min="1900"
          max={new Date()?.getFullYear()}
          placeholder="YYYY"
        />
      </div>
      {/* Description */}
      <div>
        <label className="block text-sm font-medium text-gray-700 mb-2">
          Descripción
        </label>
        <textarea
          value={formData?.description}
          onChange={(e) => handleInputChange('description', e?.target?.value)}
          disabled={!isEditing}
          rows={4}
          className={`
            w-full px-3 py-2 border rounded-lg text-sm
            ${!isEditing 
              ? 'bg-gray-50 border-gray-200 text-gray-600' : 'bg-white border-gray-300 text-gray-900 focus:ring-2 focus:ring-primary focus:border-transparent'
            }
            transition-colors duration-150
          `}
          placeholder="Descripción detallada de la propiedad..."
        />
      </div>
      {/* Auto-save Indicator */}
      {hasChanges && isEditing && (
        <div className="flex items-center space-x-2 text-sm text-warning">
          <Icon name="Clock" size={16} />
          <span>Cambios sin guardar</span>
        </div>
      )}

      {/* Enhanced Utility Summary with Basic Amounts */}
      <div className="glass-card p-6 rounded-lg">
        <h4 className="text-lg font-medium text-gray-900 mb-4">Resumen de Cuentas Básicas</h4>
        
        <div className="grid grid-cols-2 md:grid-cols-4 gap-6">
          <div className="text-center p-4 bg-blue-50 rounded-lg">
            <div className="w-12 h-12 bg-blue-500 text-white rounded-full flex items-center justify-center mx-auto mb-2">
              <Icon name="Droplets" size={20} />
            </div>
            <div className="text-lg font-bold text-blue-700">{formatCurrency(25000)}</div>
            <div className="text-sm text-blue-600">Agua</div>
            <div className="text-xs text-gray-500 mt-1">Aguas Andinas</div>
          </div>
          
          <div className="text-center p-4 bg-yellow-50 rounded-lg">
            <div className="w-12 h-12 bg-yellow-500 text-white rounded-full flex items-center justify-center mx-auto mb-2">
              <Icon name="Zap" size={20} />
            </div>
            <div className="text-lg font-bold text-yellow-700">{formatCurrency(45000)}</div>
            <div className="text-sm text-yellow-600">Electricidad</div>
            <div className="text-xs text-gray-500 mt-1">Enel Distribución</div>
          </div>
          
          <div className="text-center p-4 bg-orange-50 rounded-lg">
            <div className="w-12 h-12 bg-orange-500 text-white rounded-full flex items-center justify-center mx-auto mb-2">
              <Icon name="Flame" size={20} />
            </div>
            <div className="text-lg font-bold text-orange-700">{formatCurrency(18000)}</div>
            <div className="text-sm text-orange-600">Gas</div>
            <div className="text-xs text-gray-500 mt-1">Metrogas</div>
          </div>
          
          <div className="text-center p-4 bg-gray-50 rounded-lg">
            <div className="w-12 h-12 bg-gray-500 text-white rounded-full flex items-center justify-center mx-auto mb-2">
              <Icon name="Building" size={20} />
            </div>
            <div className="text-lg font-bold text-gray-700">{formatCurrency(45000)}</div>
            <div className="text-sm text-gray-600">Gastos Comunes</div>
            <div className="text-xs text-gray-500 mt-1">Administración</div>
          </div>
        </div>
        
        {/* Total Utilities */}
        <div className="mt-6 p-4 bg-primary/10 rounded-lg border border-primary/20">
          <div className="flex items-center justify-between">
            <span className="text-lg font-medium text-primary">Total Cuentas Básicas:</span>
            <span className="text-2xl font-bold text-primary">{formatCurrency(133000)}</span>
          </div>
          <div className="text-sm text-primary/80 mt-1">
            Este monto se incluye automáticamente en los comprobantes de arriendo
          </div>
        </div>
      </div>

      {/* Payment Status and Contract Association */}
      <div className="glass-card p-6 rounded-lg">
        <h4 className="text-lg font-medium text-gray-900 mb-4">Estado de Pago y Contrato</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Método de Pago Preferido
            </label>
            <div className="flex items-center space-x-4">
              <button
                onClick={() => handleInputChange('paymentMethod', 'manual')}
                className={`px-4 py-2 rounded-lg border-2 transition-all ${
                  formData?.paymentMethod === 'manual' ? 'border-green-500 bg-green-50 text-green-700' : 'border-gray-200 text-gray-600 hover:border-green-300'
                }`}
                disabled={!isEditing}
              >
                <Icon name="Hand" size={16} className="inline mr-2" />
                Manual
              </button>
              <button
                onClick={() => handleInputChange('paymentMethod', 'transbank')}
                className={`px-4 py-2 rounded-lg border-2 transition-all ${
                  formData?.paymentMethod === 'transbank' ? 'border-blue-500 bg-blue-50 text-blue-700' : 'border-gray-200 text-gray-600 hover:border-blue-300'
                }`}
                disabled={!isEditing}
              >
                <Icon name="CreditCard" size={16} className="inline mr-2" />
                TransBank
              </button>
            </div>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Contrato Asociado
            </label>
            <div className="flex items-center space-x-3">
              <Input
                value={formData?.contractId || 'Sin contrato asociado'}
                disabled
                className="flex-1"
              />
              <Button
                variant="outline"
                size="sm"
                iconName="Link"
                onClick={() => console.log('Asociar contrato')}
                disabled={!isEditing}
              >
                Asociar
              </Button>
            </div>
            {formData?.contractId && (
              <div className="mt-2 text-sm text-green-600">
                ✓ Contrato #{formData?.contractId} vinculado correctamente
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
};

export default BasicDataTab;