import React, { useState, useEffect } from 'react';
import Input from '../../../components/ui/Input';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';
import Image from '../../../components/AppImage';

const OwnerInformationTab = ({ ownerData, onUpdate }) => {
  const [formData, setFormData] = useState(ownerData);
  const [isEditing, setIsEditing] = useState(false);
  const [hasChanges, setHasChanges] = useState(false);
  const [isSaving, setIsSaving] = useState(false);

  useEffect(() => {
    const hasDataChanged = JSON.stringify(formData) !== JSON.stringify(ownerData);
    setHasChanges(hasDataChanged);
  }, [formData, ownerData]);

  const handleInputChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleSave = async () => {
    setIsSaving(true);
    try {
      await new Promise(resolve => setTimeout(resolve, 1000));
      onUpdate(formData);
      setIsEditing(false);
      setHasChanges(false);
    } catch (error) {
      console.error('Error saving owner data:', error);
    } finally {
      setIsSaving(false);
    }
  };

  const handleCancel = () => {
    setFormData(ownerData);
    setIsEditing(false);
    setHasChanges(false);
  };

  return (
    <div className="space-y-6">
      {/* Header with Edit Controls */}
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">Información del Propietario</h3>
        
        <div className="flex items-center space-x-2">
          {isEditing ? (
            <>
              <Button
                variant="outline"
                size="sm"
                onClick={handleCancel}
                disabled={isSaving}
              >
                Cancelar
              </Button>
              <Button
                variant="default"
                size="sm"
                onClick={handleSave}
                loading={isSaving}
                disabled={!hasChanges}
                iconName="Save"
                iconPosition="left"
              >
                Guardar
              </Button>
            </>
          ) : (
            <Button
              variant="outline"
              size="sm"
              onClick={() => setIsEditing(true)}
              iconName="Edit"
              iconPosition="left"
            >
              Editar
            </Button>
          )}
        </div>
      </div>
      {/* Owner Profile Card */}
      <div className="glass-card p-6 rounded-lg">
        <div className="flex items-start space-x-4">
          <div className="flex-shrink-0">
            <div className="w-16 h-16 bg-gray-200 rounded-full overflow-hidden">
              {formData?.avatar ? (
                <Image 
                  src={formData?.avatar} 
                  alt={formData?.fullName}
                  className="w-full h-full object-cover"
                />
              ) : (
                <div className="w-full h-full bg-primary/10 flex items-center justify-center">
                  <Icon name="User" size={24} className="text-primary" />
                </div>
              )}
            </div>
          </div>
          
          <div className="flex-1 min-w-0">
            <h4 className="text-lg font-medium text-gray-900">{formData?.fullName}</h4>
            <p className="text-sm text-gray-600">{formData?.email}</p>
            <p className="text-sm text-gray-600">{formData?.phone}</p>
            <div className="mt-2 flex items-center space-x-4 text-sm">
              <span className="text-gray-500">Propiedades: <span className="font-medium text-gray-900">{formData?.totalProperties}</span></span>
              <span className="text-gray-500">Comisión: <span className="font-medium text-primary">{formData?.commissionRate}%</span></span>
            </div>
          </div>
        </div>
      </div>
      {/* Contact Information */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Input
          label="Nombre Completo"
          type="text"
          value={formData?.fullName}
          onChange={(e) => handleInputChange('fullName', e?.target?.value)}
          disabled={!isEditing}
          required
        />

        <Input
          label="RUT"
          type="text"
          value={formData?.rut}
          onChange={(e) => handleInputChange('rut', e?.target?.value)}
          disabled={!isEditing}
          placeholder="12.345.678-9"
        />

        <Input
          label="Email"
          type="email"
          value={formData?.email}
          onChange={(e) => handleInputChange('email', e?.target?.value)}
          disabled={!isEditing}
          required
        />

        <Input
          label="Teléfono"
          type="tel"
          value={formData?.phone}
          onChange={(e) => handleInputChange('phone', e?.target?.value)}
          disabled={!isEditing}
          placeholder="+56 9 1234 5678"
        />

        <Input
          label="Teléfono Alternativo"
          type="tel"
          value={formData?.alternativePhone}
          onChange={(e) => handleInputChange('alternativePhone', e?.target?.value)}
          disabled={!isEditing}
          placeholder="+56 9 1234 5678"
        />

        <Input
          label="Profesión/Ocupación"
          type="text"
          value={formData?.occupation}
          onChange={(e) => handleInputChange('occupation', e?.target?.value)}
          disabled={!isEditing}
        />
      </div>
      {/* Address Information */}
      <div className="space-y-4">
        <h4 className="text-md font-medium text-gray-900">Dirección</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div className="md:col-span-2">
            <Input
              label="Dirección"
              type="text"
              value={formData?.address}
              onChange={(e) => handleInputChange('address', e?.target?.value)}
              disabled={!isEditing}
            />
          </div>

          <Input
            label="Ciudad"
            type="text"
            value={formData?.city}
            onChange={(e) => handleInputChange('city', e?.target?.value)}
            disabled={!isEditing}
          />

          <Input
            label="Comuna"
            type="text"
            value={formData?.commune}
            onChange={(e) => handleInputChange('commune', e?.target?.value)}
            disabled={!isEditing}
          />
        </div>
      </div>
      {/* Commission Settings */}
      <div className="space-y-4">
        <h4 className="text-md font-medium text-gray-900">Configuración de Comisiones</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Input
            label="Tasa de Comisión (%)"
            type="number"
            value={formData?.commissionRate}
            onChange={(e) => handleInputChange('commissionRate', e?.target?.value)}
            disabled={!isEditing}
            min="0"
            max="100"
            step="0.1"
          />

          <Input
            label="Comisión Mínima (CLP)"
            type="number"
            value={formData?.minimumCommission}
            onChange={(e) => handleInputChange('minimumCommission', e?.target?.value)}
            disabled={!isEditing}
            min="0"
            placeholder="0"
          />

          <Input
            label="Comisión Máxima (CLP)"
            type="number"
            value={formData?.maximumCommission}
            onChange={(e) => handleInputChange('maximumCommission', e?.target?.value)}
            disabled={!isEditing}
            min="0"
            placeholder="Sin límite"
          />
        </div>
      </div>
      {/* Bank Information */}
      <div className="space-y-4">
        <h4 className="text-md font-medium text-gray-900">Información Bancaria</h4>
        
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <Input
            label="Banco"
            type="text"
            value={formData?.bankName}
            onChange={(e) => handleInputChange('bankName', e?.target?.value)}
            disabled={!isEditing}
          />

          <Input
            label="Tipo de Cuenta"
            type="text"
            value={formData?.accountType}
            onChange={(e) => handleInputChange('accountType', e?.target?.value)}
            disabled={!isEditing}
            placeholder="Cuenta Corriente / Cuenta Vista"
          />

          <Input
            label="Número de Cuenta"
            type="text"
            value={formData?.accountNumber}
            onChange={(e) => handleInputChange('accountNumber', e?.target?.value)}
            disabled={!isEditing}
          />

          <Input
            label="Email para Transferencias"
            type="email"
            value={formData?.transferEmail}
            onChange={(e) => handleInputChange('transferEmail', e?.target?.value)}
            disabled={!isEditing}
          />
        </div>
      </div>
      {/* Auto-save Indicator */}
      {hasChanges && isEditing && (
        <div className="flex items-center space-x-2 text-sm text-warning">
          <Icon name="Clock" size={16} />
          <span>Cambios sin guardar</span>
        </div>
      )}
    </div>
  );
};

export default OwnerInformationTab;