import React, { useState } from 'react';
import Input from '../../../components/ui/Input';
import Select from '../../../components/ui/Select';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';
import StatusIndicatorBadges from '../../../components/ui/StatusIndicatorBadges';
import VoucherEditModal from './VoucherEditModal';
import VoucherDownloadModal from './VoucherDownloadModal';

const PaymentHistoryTab = ({ paymentData, propertyData }) => {
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('all');
  const [dateFilter, setDateFilter] = useState('all');
  const [currentPage, setCurrentPage] = useState(1);
  const [showVoucherModal, setShowVoucherModal] = useState(false);
  const [showDownloadModal, setShowDownloadModal] = useState(false);
  const [editingVoucher, setEditingVoucher] = useState(null);
  const [selectedPayment, setSelectedPayment] = useState(null);
  const [mockPayments, setMockPayments] = useState([
    {
      id: 1,
      month: 'Diciembre 2024',
      dueDate: '2024-12-05',
      paidDate: '2024-12-03',
      amount: 850000,
      gasAmount: 35000,
      commonExpenses: 45000,
      waterAmount: 25000,
      electricityAmount: 40000,
      internetAmount: 30000,
      totalAmount: 895000,
      status: 'paid',
      paymentMethod: 'Transferencia Bancaria',
      voucherNumber: 'VCH-2024-12-001',
      notes: 'Pago realizado antes del vencimiento',
      tenantDiscounts: [{ description: '', amount: 0 }],
      ownerDiscounts: [{ description: '', amount: 0 }],
      otherDiscounts: [{ description: '', amount: 0 }]
    },
    {
      id: 2,
      month: 'Noviembre 2024',
      dueDate: '2024-11-05',
      paidDate: '2024-11-07',
      amount: 850000,
      gasAmount: 35000,
      commonExpenses: 45000,
      waterAmount: 25000,
      electricityAmount: 40000,
      internetAmount: 30000,
      totalAmount: 895000,
      status: 'paid',
      paymentMethod: 'Transferencia Bancaria',
      voucherNumber: 'VCH-2024-11-001',
      notes: 'Pago con 2 días de retraso',
      tenantDiscounts: [{ description: 'Descuento por pago tardío', amount: 20000 }],
      ownerDiscounts: [{ description: '', amount: 0 }],
      otherDiscounts: [{ description: '', amount: 0 }]
    },
    {
      id: 3,
      month: 'Octubre 2024',
      dueDate: '2024-10-05',
      paidDate: null,
      amount: 850000,
      gasAmount: 35000,
      commonExpenses: 45000,
      waterAmount: 25000,
      electricityAmount: 40000,
      internetAmount: 30000,
      totalAmount: 895000,
      status: 'overdue',
      paymentMethod: null,
      voucherNumber: 'VCH-2024-10-001',
      notes: 'Pago pendiente - enviado recordatorio',
      tenantDiscounts: [{ description: '', amount: 0 }],
      ownerDiscounts: [{ description: '', amount: 0 }],
      otherDiscounts: [{ description: '', amount: 0 }]
    },
    {
      id: 4,
      month: 'Septiembre 2024',
      dueDate: '2024-09-05',
      paidDate: '2024-09-15',
      amount: 850000,
      gasAmount: 35000,
      commonExpenses: 45000,
      waterAmount: 25000,
      electricityAmount: 40000,
      internetAmount: 30000,
      totalAmount: 500000,
      status: 'partial',
      paymentMethod: 'Transferencia Bancaria',
      voucherNumber: 'VCH-2024-09-001',
      notes: 'Pago parcial - pendiente saldo',
      tenantDiscounts: [{ description: '', amount: 0 }],
      ownerDiscounts: [{ description: 'Descuento mantenimiento', amount: 50000 }],
      otherDiscounts: [{ description: 'Descuento especial', amount: 25000 }]
    },
    {
      id: 5,
      month: 'Agosto 2024',
      dueDate: '2024-08-05',
      paidDate: '2024-08-04',
      amount: 850000,
      gasAmount: 35000,
      commonExpenses: 45000,
      waterAmount: 25000,
      electricityAmount: 40000,
      internetAmount: 30000,
      totalAmount: 895000,
      status: 'paid',
      paymentMethod: 'Transferencia Bancaria',
      voucherNumber: 'VCH-2024-08-001',
      notes: 'Pago puntual',
      tenantDiscounts: [{ description: '', amount: 0 }],
      ownerDiscounts: [{ description: '', amount: 0 }],
      otherDiscounts: [{ description: '', amount: 0 }]
    }
  ]);

  const itemsPerPage = 10;

  const statusOptions = [
    { value: 'all', label: 'Todos los Estados' },
    { value: 'paid', label: 'Pagado' },
    { value: 'pending', label: 'Pendiente' },
    { value: 'overdue', label: 'Vencido' },
    { value: 'partial', label: 'Pago Parcial' }
  ];

  const dateFilterOptions = [
    { value: 'all', label: 'Todo el Período' },
    { value: 'current-year', label: 'Año Actual' },
    { value: 'last-6-months', label: 'Últimos 6 Meses' },
    { value: 'last-3-months', label: 'Últimos 3 Meses' },
    { value: 'current-month', label: 'Mes Actual' }
  ];

  const filteredPayments = mockPayments?.filter(payment => {
    const matchesSearch = payment?.month?.toLowerCase()?.includes(searchTerm?.toLowerCase()) ||
                         payment?.voucherNumber?.toLowerCase()?.includes(searchTerm?.toLowerCase());
    const matchesStatus = statusFilter === 'all' || payment?.status === statusFilter;
    return matchesSearch && matchesStatus;
  });

  const totalPages = Math.ceil(filteredPayments?.length / itemsPerPage);
  const startIndex = (currentPage - 1) * itemsPerPage;
  const paginatedPayments = filteredPayments?.slice(startIndex, startIndex + itemsPerPage);

  const formatCurrency = (amount) => {
    return new Intl.NumberFormat('es-CL', {
      style: 'currency',
      currency: 'CLP',
      minimumFractionDigits: 0
    })?.format(amount);
  };

  const formatDate = (dateString) => {
    if (!dateString) return '-';
    return new Date(dateString)?.toLocaleDateString('es-CL');
  };

  const calculateSummary = () => {
    const totalPaid = mockPayments?.filter(p => p?.status === 'paid')?.reduce((sum, p) => sum + p?.totalAmount, 0);
    
    const totalPending = mockPayments?.filter(p => p?.status === 'pending' || p?.status === 'overdue')?.reduce((sum, p) => sum + p?.totalAmount, 0);
    
    const paidCount = mockPayments?.filter(p => p?.status === 'paid')?.length;
    const overdueCount = mockPayments?.filter(p => p?.status === 'overdue')?.length;

    return { totalPaid, totalPending, paidCount, overdueCount };
  };

  const handleCreateVoucher = () => {
    setEditingVoucher(null);
    setShowVoucherModal(true);
  };

  const handleEditVoucher = (voucher) => {
    setEditingVoucher(voucher);
    setShowVoucherModal(true);
  };

  const handleSaveVoucher = (voucherData) => {
    if (editingVoucher) {
      // Update existing voucher
      setMockPayments(prev => 
        prev?.map(payment => 
          payment?.id === editingVoucher?.id 
            ? { ...payment, ...voucherData }
            : payment
        )
      );
    } else {
      // Create new voucher
      setMockPayments(prev => [...prev, voucherData]);
    }
    
    setShowVoucherModal(false);
    setEditingVoucher(null);
  };

  const handleDownloadVoucher = (payment) => {
    setSelectedPayment(payment);
    setShowDownloadModal(true);
  };

  const handlePreviewVoucher = (payment) => {
    // Quick preview functionality - opens a preview window
    const previewWindow = window.open('', '_blank');
    if (previewWindow) {
      const html = generateSimplePreview(payment, propertyData);
      previewWindow?.document?.write(html);
      previewWindow?.document?.close();
    }
  };

  const generateSimplePreview = (payment, propertyData) => {
    return `
      <!DOCTYPE html>
      <html>
      <head>
        <title>Vista Previa - ${payment?.voucherNumber}</title>
        <style>
          body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
          .preview-container { background: white; padding: 30px; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
          .header { text-align: center; margin-bottom: 20px; }
          .info-row { display: flex; justify-content: space-between; margin: 10px 0; }
          .total { font-weight: bold; font-size: 18px; color: #2563eb; border-top: 2px solid #e5e7eb; padding-top: 10px; }
        </style>
      </head>
      <body>
        <div class="preview-container">
          <div class="header">
            <h2>Vista Previa del Comprobante</h2>
            <p>${payment?.voucherNumber} - ${payment?.month}</p>
          </div>
          <div class="info-row">
            <span>Arriendo:</span>
            <span>${formatCurrency(payment?.amount)}</span>
          </div>
          <div class="info-row">
            <span>Gastos:</span>
            <span>${formatCurrency((payment?.gasAmount || 0) + (payment?.commonExpenses || 0) + (payment?.waterAmount || 0) + (payment?.electricityAmount || 0) + (payment?.internetAmount || 0))}</span>
          </div>
          <div class="info-row total">
            <span>Total:</span>
            <span>${formatCurrency(payment?.totalAmount)}</span>
          </div>
          <div style="margin-top: 20px; padding: 15px; background: #f0f9ff; border-radius: 4px;">
            <p><strong>Nota:</strong> Esta es solo una vista previa. Use las opciones de descarga para generar comprobantes completos.</p>
          </div>
        </div>
      </body>
      </html>
    `;
  };

  const getExpensesBreakdown = (payment) => {
    const gasAmount = payment?.gasAmount || 0;
    const commonExpenses = payment?.commonExpenses || 0;
    const otherExpenses = (payment?.waterAmount || 0) + (payment?.electricityAmount || 0) + (payment?.internetAmount || 0);
    
    return { gasAmount, commonExpenses, otherExpenses };
  };

  const summary = calculateSummary();

  const mockTenantData = {
    name: 'María González Pérez',
    email: 'maria.gonzalez@email.com',
    phone: '+56 9 8765 4321',
    rut: '18.765.432-1'
  };

  const mockOwnerData = {
    name: 'Carlos Rodríguez Silva',
    email: 'carlos.rodriguez@email.com',
    phone: '+56 9 1234 5678',
    rut: '12.345.678-9'
  };

  const enhancedPropertyData = {
    address: propertyData?.address || 'Av. Providencia 1234, Providencia, Santiago',
    type: propertyData?.type || 'Departamento',
    code: propertyData?.code || 'PROP-2024-001',
    monthlyRent: propertyData?.monthlyRent || 850000,
    commonExpenses: propertyData?.commonExpenses || 45000,
    ...propertyData
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">Historial de Pagos</h3>
        
        <div className="flex items-center space-x-2">
          <Button
            variant="outline"
            size="sm"
            iconName="Download"
            iconPosition="left"
          >
            Exportar
          </Button>
          <Button
            variant="default"
            size="sm"
            iconName="Plus"
            iconPosition="left"
            onClick={handleCreateVoucher}
          >
            Crear Comprobante
          </Button>
        </div>
      </div>
      {/* Summary Cards */}
      <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
        <div className="glass-card p-4 rounded-lg text-center">
          <div className="text-2xl font-bold text-success">{formatCurrency(summary?.totalPaid)}</div>
          <div className="text-sm text-gray-600">Total Pagado</div>
        </div>
        <div className="glass-card p-4 rounded-lg text-center">
          <div className="text-2xl font-bold text-error">{formatCurrency(summary?.totalPending)}</div>
          <div className="text-sm text-gray-600">Total Pendiente</div>
        </div>
        <div className="glass-card p-4 rounded-lg text-center">
          <div className="text-2xl font-bold text-primary">{summary?.paidCount}</div>
          <div className="text-sm text-gray-600">Pagos Realizados</div>
        </div>
        <div className="glass-card p-4 rounded-lg text-center">
          <div className="text-2xl font-bold text-warning">{summary?.overdueCount}</div>
          <div className="text-sm text-gray-600">Pagos Vencidos</div>
        </div>
      </div>
      {/* Filters */}
      <div className="glass-card p-4 rounded-lg">
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Input
            label="Buscar"
            type="search"
            value={searchTerm}
            onChange={(e) => setSearchTerm(e?.target?.value)}
            placeholder="Buscar por mes o número de comprobante..."
          />
          
          <Select
            label="Estado"
            options={statusOptions}
            value={statusFilter}
            onChange={setStatusFilter}
          />
          
          <Select
            label="Período"
            options={dateFilterOptions}
            value={dateFilter}
            onChange={setDateFilter}
          />
        </div>
      </div>
      {/* Payment History Table */}
      <div className="glass-card rounded-lg overflow-hidden">
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50 border-b border-gray-200">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Período
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Vencimiento
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Monto
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Estado
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Fecha Pago
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Acciones
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {paginatedPayments?.map((payment) => {
                const expenses = getExpensesBreakdown(payment);
                return (
                  <tr key={payment?.id} className="hover:bg-gray-50 transition-colors duration-150">
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div>
                        <div className="text-sm font-medium text-gray-900">{payment?.month}</div>
                        <div className="text-xs text-gray-500">{payment?.voucherNumber}</div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {formatDate(payment?.dueDate)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <div className="text-sm font-medium text-gray-900">
                        {formatCurrency(payment?.totalAmount)}
                      </div>
                      <div className="text-xs text-gray-500 space-y-1">
                        <div>Arriendo: {formatCurrency(payment?.amount)}</div>
                        <div>Gas: {formatCurrency(expenses?.gasAmount)}</div>
                        <div>GGCC: {formatCurrency(expenses?.commonExpenses)}</div>
                        <div>Otros: {formatCurrency(expenses?.otherExpenses)}</div>
                      </div>
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap">
                      <StatusIndicatorBadges status={payment?.status} size="sm" />
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                      {formatDate(payment?.paidDate)}
                    </td>
                    <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                      <div className="flex items-center space-x-2">
                        <button 
                          className="text-primary hover:text-primary/80 transition-colors duration-150 p-1 rounded-md hover:bg-blue-50"
                          onClick={() => handlePreviewVoucher(payment)}
                          title="Vista previa rápida"
                        >
                          <Icon name="Eye" size={16} />
                        </button>
                        <button 
                          className="text-blue-600 hover:text-blue-800 transition-colors duration-150 p-1 rounded-md hover:bg-blue-50"
                          onClick={() => handleEditVoucher(payment)}
                          title="Modificar comprobante"
                        >
                          <Icon name="Edit" size={16} />
                        </button>
                        <button 
                          className="text-green-600 hover:text-green-800 transition-colors duration-150 p-1 rounded-md hover:bg-green-50"
                          onClick={() => handleDownloadVoucher(payment)}
                          title="Descargar comprobante"
                        >
                          <Icon name="Download" size={16} />
                        </button>
                        {payment?.status !== 'paid' && (
                          <button 
                            className="text-success hover:text-success/80 transition-colors duration-150 p-1 rounded-md hover:bg-green-50"
                            title="Marcar como pagado"
                          >
                            <Icon name="CheckCircle" size={16} />
                          </button>
                        )}
                      </div>
                    </td>
                  </tr>
                );
              })}
            </tbody>
          </table>
        </div>

        {/* Pagination */}
        {totalPages > 1 && (
          <div className="px-6 py-3 border-t border-gray-200 flex items-center justify-between">
            <div className="text-sm text-gray-700">
              Mostrando {startIndex + 1} a {Math.min(startIndex + itemsPerPage, filteredPayments?.length)} de {filteredPayments?.length} registros
            </div>
            <div className="flex items-center space-x-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => setCurrentPage(prev => Math.max(prev - 1, 1))}
                disabled={currentPage === 1}
                iconName="ChevronLeft"
              >
                Anterior
              </Button>
              <span className="text-sm text-gray-600">
                Página {currentPage} de {totalPages}
              </span>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setCurrentPage(prev => Math.min(prev + 1, totalPages))}
                disabled={currentPage === totalPages}
                iconName="ChevronRight"
              >
                Siguiente
              </Button>
            </div>
          </div>
        )}
      </div>
      {/* Empty State */}
      {filteredPayments?.length === 0 && (
        <div className="glass-card p-12 rounded-lg text-center">
          <Icon name="FileText" size={48} className="text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No se encontraron pagos</h3>
          <p className="text-gray-600 mb-4">
            No hay registros de pagos que coincidan con los filtros seleccionados.
          </p>
          <Button variant="outline" onClick={() => {
            setSearchTerm('');
            setStatusFilter('all');
            setDateFilter('all');
          }}>
            Limpiar Filtros
          </Button>
        </div>
      )}

      {/* Voucher Edit Modal */}
      <VoucherEditModal
        isOpen={showVoucherModal}
        onClose={() => {
          setShowVoucherModal(false);
          setEditingVoucher(null);
        }}
        voucher={editingVoucher}
        propertyData={enhancedPropertyData}
        onSave={handleSaveVoucher}
      />

      {/* Voucher Download Modal */}
      <VoucherDownloadModal
        isOpen={showDownloadModal}
        onClose={() => {
          setShowDownloadModal(false);
          setSelectedPayment(null);
        }}
        payment={selectedPayment}
        propertyData={enhancedPropertyData}
        tenantData={mockTenantData}
        ownerData={mockOwnerData}
      />
    </div>
  );
};

export default PaymentHistoryTab;