import React, { useState, useEffect } from 'react';
import { Checkbox } from '../../../components/ui/Checkbox';
import Select from '../../../components/ui/Select';
import Button from '../../../components/ui/Button';
import Icon from '../../../components/AppIcon';

const PropertyCharacteristicsTab = ({ characteristicsData, onUpdate }) => {
  const [formData, setFormData] = useState(characteristicsData);
  const [isEditing, setIsEditing] = useState(false);
  const [hasChanges, setHasChanges] = useState(false);
  const [isSaving, setIsSaving] = useState(false);

  const conditionOptions = [
    { value: 'excellent', label: 'Excelente' },
    { value: 'very-good', label: 'Muy Bueno' },
    { value: 'good', label: 'Bueno' },
    { value: 'fair', label: 'Regular' },
    { value: 'poor', label: 'Malo' }
  ];

  const orientationOptions = [
    { value: 'north', label: 'Norte' },
    { value: 'south', label: 'Sur' },
    { value: 'east', label: 'Este' },
    { value: 'west', label: 'Oeste' },
    { value: 'northeast', label: 'Noreste' },
    { value: 'northwest', label: 'Noroeste' },
    { value: 'southeast', label: 'Sureste' },
    { value: 'southwest', label: 'Suroeste' }
  ];

  const amenityCategories = {
    interior: {
      title: 'Características Interiores',
      items: [
        { key: 'furnished', label: 'Amoblado' },
        { key: 'airConditioning', label: 'Aire Acondicionado' },
        { key: 'heating', label: 'Calefacción' },
        { key: 'fireplace', label: 'Chimenea' },
        { key: 'balcony', label: 'Balcón' },
        { key: 'terrace', label: 'Terraza' },
        { key: 'walkinCloset', label: 'Walk-in Closet' },
        { key: 'laundryRoom', label: 'Lavadero' }
      ]
    },
    building: {
      title: 'Características del Edificio',
      items: [
        { key: 'elevator', label: 'Ascensor' },
        { key: 'doorman', label: 'Portero' },
        { key: 'security', label: 'Seguridad 24/7' },
        { key: 'gym', label: 'Gimnasio' },
        { key: 'pool', label: 'Piscina' },
        { key: 'playground', label: 'Área de Juegos' },
        { key: 'eventRoom', label: 'Salón de Eventos' },
        { key: 'rooftop', label: 'Terraza Común' }
      ]
    },
    location: {
      title: 'Ubicación y Accesos',
      items: [
        { key: 'nearMetro', label: 'Cerca del Metro' },
        { key: 'nearBus', label: 'Cerca de Transporte Público' },
        { key: 'nearSchools', label: 'Cerca de Colegios' },
        { key: 'nearShopping', label: 'Cerca de Centros Comerciales' },
        { key: 'nearParks', label: 'Cerca de Parques' },
        { key: 'nearHospitals', label: 'Cerca de Hospitales' },
        { key: 'quietArea', label: 'Zona Tranquila' },
        { key: 'commercialArea', label: 'Zona Comercial' }
      ]
    }
  };

  useEffect(() => {
    const hasDataChanged = JSON.stringify(formData) !== JSON.stringify(characteristicsData);
    setHasChanges(hasDataChanged);
  }, [formData, characteristicsData]);

  const handleSelectChange = (field, value) => {
    setFormData(prev => ({
      ...prev,
      [field]: value
    }));
  };

  const handleAmenityChange = (amenityKey, checked) => {
    setFormData(prev => ({
      ...prev,
      amenities: {
        ...prev?.amenities,
        [amenityKey]: checked
      }
    }));
  };

  const handleSave = async () => {
    setIsSaving(true);
    try {
      await new Promise(resolve => setTimeout(resolve, 1000));
      onUpdate(formData);
      setIsEditing(false);
      setHasChanges(false);
    } catch (error) {
      console.error('Error saving characteristics:', error);
    } finally {
      setIsSaving(false);
    }
  };

  const handleCancel = () => {
    setFormData(characteristicsData);
    setIsEditing(false);
    setHasChanges(false);
  };

  return (
    <div className="space-y-6">
      {/* Header with Edit Controls */}
      <div className="flex items-center justify-between">
        <h3 className="text-lg font-semibold text-gray-900">Características de la Propiedad</h3>
        
        <div className="flex items-center space-x-2">
          {isEditing ? (
            <>
              <Button
                variant="outline"
                size="sm"
                onClick={handleCancel}
                disabled={isSaving}
              >
                Cancelar
              </Button>
              <Button
                variant="default"
                size="sm"
                onClick={handleSave}
                loading={isSaving}
                disabled={!hasChanges}
                iconName="Save"
                iconPosition="left"
              >
                Guardar
              </Button>
            </>
          ) : (
            <Button
              variant="outline"
              size="sm"
              onClick={() => setIsEditing(true)}
              iconName="Edit"
              iconPosition="left"
            >
              Editar
            </Button>
          )}
        </div>
      </div>
      {/* General Characteristics */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <Select
          label="Estado de la Propiedad"
          options={conditionOptions}
          value={formData?.condition}
          onChange={(value) => handleSelectChange('condition', value)}
          disabled={!isEditing}
          required
        />

        <Select
          label="Orientación"
          options={orientationOptions}
          value={formData?.orientation}
          onChange={(value) => handleSelectChange('orientation', value)}
          disabled={!isEditing}
        />
      </div>
      {/* Amenities by Category */}
      {Object.entries(amenityCategories)?.map(([categoryKey, category]) => (
        <div key={categoryKey} className="space-y-4">
          <h4 className="text-md font-medium text-gray-900 flex items-center space-x-2">
            <Icon 
              name={categoryKey === 'interior' ? 'Home' : categoryKey === 'building' ? 'Building2' : 'MapPin'} 
              size={18} 
              className="text-primary" 
            />
            <span>{category?.title}</span>
          </h4>
          
          <div className="glass-card p-4 rounded-lg">
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3">
              {category?.items?.map((item) => (
                <Checkbox
                  key={item?.key}
                  label={item?.label}
                  checked={formData?.amenities?.[item?.key] || false}
                  onChange={(e) => handleAmenityChange(item?.key, e?.target?.checked)}
                  disabled={!isEditing}
                  size="sm"
                />
              ))}
            </div>
          </div>
        </div>
      ))}
      {/* Additional Features */}
      <div className="space-y-4">
        <h4 className="text-md font-medium text-gray-900">Características Adicionales</h4>
        
        <div className="glass-card p-4 rounded-lg">
          <textarea
            value={formData?.additionalFeatures || ''}
            onChange={(e) => handleSelectChange('additionalFeatures', e?.target?.value)}
            disabled={!isEditing}
            rows={4}
            className={`
              w-full px-3 py-2 border rounded-lg text-sm
              ${!isEditing 
                ? 'bg-gray-50 border-gray-200 text-gray-600' :'bg-white border-gray-300 text-gray-900 focus:ring-2 focus:ring-primary focus:border-transparent'
              }
              transition-colors duration-150
            `}
            placeholder="Describe características adicionales, mejoras recientes, o detalles especiales de la propiedad..."
          />
        </div>
      </div>
      {/* Summary Statistics */}
      <div className="glass-card p-4 rounded-lg">
        <h4 className="text-md font-medium text-gray-900 mb-3">Resumen de Características</h4>
        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 text-center">
          <div>
            <div className="text-2xl font-bold text-primary">
              {Object.values(formData?.amenities)?.filter(Boolean)?.length}
            </div>
            <div className="text-sm text-gray-600">Amenidades</div>
          </div>
          <div>
            <div className="text-2xl font-bold text-success">
              {formData?.condition ? conditionOptions?.find(opt => opt?.value === formData?.condition)?.label : '-'}
            </div>
            <div className="text-sm text-gray-600">Estado</div>
          </div>
          <div>
            <div className="text-2xl font-bold text-secondary">
              {formData?.orientation ? orientationOptions?.find(opt => opt?.value === formData?.orientation)?.label : '-'}
            </div>
            <div className="text-sm text-gray-600">Orientación</div>
          </div>
          <div>
            <div className="text-2xl font-bold text-warning">
              {formData?.amenities?.furnished ? 'Sí' : 'No'}
            </div>
            <div className="text-sm text-gray-600">Amoblado</div>
          </div>
        </div>
      </div>
      {/* Auto-save Indicator */}
      {hasChanges && isEditing && (
        <div className="flex items-center space-x-2 text-sm text-warning">
          <Icon name="Clock" size={16} />
          <span>Cambios sin guardar</span>
        </div>
      )}
    </div>
  );
};

export default PropertyCharacteristicsTab;